<?php
require_once('sonots.class.php');
require_once('metapage.class.php');

/**
 * Table of Contents Class
 *
 * This class tries to gather all information at the constructor
 * including information which may not be used in this session
 * because a cache file must contain all information for future. 
 * Therefore, there is no setter function unlike PluginSonotsPagelist(). 
 *
 * PHP5 has static member variable supports, thus PHP5 allows
 * following smart design, but PHP4 does not
 * <code>
 *  PluginSonotsToc::syntax['headline'] = '/^(\*{1,5})/';
 *  $toc1 = new PluginSonotsToc($page1, true);
 *  $toc2 = new PluginSonotsToc($page2, true);
 *  // $cachefile = PluginSonotsToc::syntax['cachefile']($page1);
 * </code>
 * To use static member variable in PHP4, I used PHP4 static 
 * variable trick and use of this class became as followings:
 * <code>
 *  $toc = new PluginSonotsToc(); // this $toc is to set static var
 *  $toc->syntax['headline'] = '/^(\*{1,5})/';
 *  $toc1 = new PluginSonotsToc($page1, true);
 *  $toc2 = new PluginSonotsToc($page2, true);
 *  // $cachefile = $toc->syntax['cachefile']($page1);
 * </code>
 * Use this form in both PHP4 and PHP5. 
 *
 * @package    PluginSonots
 * @license    http://www.gnu.org/licenses/gpl.html GPL v2
 * @author     sonots <http://lsx.sourceforge.jp>
 * @version    $Id: toc.class.php,v 1.3 2008-06-07 07:23:17Z sonots $
 * @require    sonots     v 1.9
 * @require    metapage   v 1.4
 */
class PluginSonotsToc
{
    /**
     * Pagename
     *
     * @var string
     */
    var $page;
    /**
     * Headlines of this page
     *
     * @var array array of PluginSonotsHeadline()
     */
    var $headlines = array();
    /**
     * Including pages
     *
     * array of including pages whose keys are pagenames and 
     * values are titles of pages (if available). 
     * The first element is for this page
     *
     * @var array
     */
    var $includes = array();
    /**
     * Definition of PukiWiki Syntax to be used
     *
     * @static
     * @var array
     * - headline
     * - include
     * - title
     * - contents
     * - cachefile
     */
    var $syntax; // static

    /**
     * Constructor
     *
     * If a page is given, the true constructor self::init() is called.
     * Otherwise, this only initializes static member variables. 
     *
     * @access public
     * @param string $page
     * @param boolean $usecache
     * @uses init
     */
    function PluginSonotsToc($page = '', $usecache = true)
    {
        static $syntax = array();
        if (empty($syntax)) {
            $syntax = array(
                 'headline'   => '/^(\*{1,3})/',
                 'include'    => '/^#include.*\((.+)\)/',
                 'title'      => '/^TITLE:(.+)/',
                 'contents'   => '/^#contents/',
                 'cachefile'  => create_function('$page','return CACHE_DIR . encode($page) . ".toc";'),
            );
        }
        $this->syntax = &$syntax; // php4 static trick
        if ($page === '' && $usecache === false) return; 
        $this->init($page, $usecache);
    }

    /**
     * True Constructor
     *
     * @access private
     * @param string $page
     * @param boolean $usecache
     */
    function init($page, $usecache)
    {
        $this->page = $page;
        if (! $usecache) {
            list($this->includes, $this->headlines) = PluginSonotsToc::get_headlines($page, $this->syntax);
            return;
        }
        // use cache
        $cachefile = $this->syntax['cachefile']($page);
        $renew = FALSE; // renew cache if one of including pages are newer than cache
        if (sonots::is_page_newer($page, $cachefile, TRUE)) {
            $renew = TRUE;
        } else { // check all including pages too
            // including pages are obtained from cache
            list($this->includes, $this->headlines)
                = unserialize(file_get_contents($cachefile));
            $pages = array_keys($this->includes);
            for ($i = 1; $i < count($pages); ++$i) {// first is the $page, already done
                if (sonots::is_page_newer($pages[$i], $cachefile)) {
                    $renew = TRUE; break;
                }
            }
        }
        if ($renew) { // recreate and write
            list($this->includes, $this->headlines) = PluginSonotsToc::get_headlines($page, $this->syntax);
            $contents = serialize(array($this->includes, $this->headlines));
            file_put_contents($cachefile, $contents);
        } else { // read from cache
            // cache is already read
        }
    }

    /**
     * Get TITLE of this page
     *
     * @access public
     * @param string $page
     * @return string
     */
    function get_title()
    {
        $title = $this->includes[$this->page];
        return ($title !== '') ? $title : $this->page;
    }

    /**
     * Get First Headline String of this page
     *
     * @access public
     * @param string $page
     * @return string
     */
    function get_firsthead()
    {
        $firsthead = reset($this->headlines);
        return $firsthead->string;
    }

    /**
     * Get the specified meta informations of headlines
     *
     * @access public
     * @param string $metakey specify meta information you want to get
     * @return array metas
     */
    function get_metas($metakey)
    {
        return sonots::get_members($this->headlines, $metakey);
    }

    /**
     * Grep out headlines by speific fields
     *
     * @access public
     * @param string $meta name of meta information to be greped
     * @param string $func func name
     *  - preg     : grep by preg
     *  - ereg     : grep by ereg
     *  - mb_ereg  : grep by mb_ereg
     *  - prefix   : remains if prefix matches (strpos)
     *  - mb_prefix: (mb_strpos)
     *  - eq       : remains if equality holds
     *  - ge       : remains if greater or equal to
     *  - le       : remains if less or equal to
     * @param mixed $pattern
     * @param boolean $inverse grep -v
     * @return void
     */
    function grep_by($meta, $func, $pattern, $inverse = FALSE)
    {
        $metas = $this->get_metas($meta);
        $metas = sonots::grep_array($pattern, $metas, $func);
        if (! $inverse) {
            $this->headlines = array_intersect_key($this->headlines, $metas);
        } else {
            $this->headlines = array_diff_key($this->headlines, $metas);
        }
    }


    /**
     * Slice headlines
     *
     * @param int $offset
     * @param mixed $length int or NULL (means forever)
     * @see array_slice
     */
    function slice($offset, $length)
    {
        $this->headlines = sonots::array_slice
            ($this->headlines, $offset, $length, true);
    }

    /**
     * Display Table of Contents
     *
     * @access public
     * @param string $cssclass css class
     * @param string $optlink link option
     * @return string list html
     */
    function display_toc($cssclass = '', $optlink = 'anchor')
    {
        $links = array();
        foreach ($this->headlines as $i => $headline) {
            switch ($optlink) {
            case 'page':
                $linkstr = strip_htmltag(sonots::make_inline($headline->string));
                $link = sonots::make_pagelink_nopg($headline->page, $linkstr, '#' . $headline->anchor);
                break;
            case 'off':
                $link = $linkstr;
                break;
            case 'anchor':
            default:
                $linkstr = strip_htmltag(sonots::make_inline($headline->string));
                $link = sonots::make_pagelink_nopg('', $linkstr, '#' . $headline->anchor);
                break;
            }
            $links[$i] = $link;
        }
        $levels = $this->get_metas('depth');
        return sonots::display_list($links, $levels, $cssclass);
    }

    /**
     * Make flat all depths to 1
     *
     * @access public
     */
    function flat_depth()
    {
        foreach ($this->headlines as $i => $val) {
            $this->headlines[$i]->depth = 1;
        }
    }

    /**
     * Originally, depth of included page is 0, 
     * shift up if any included pages exist. 
     *
     * @access public
     */
    function shift_depth()
    {
        if (sizeof($this->includes) >= 2) {
            foreach ($this->headlines as $i => $val) {
                $this->headlines[$i]->depth += 1;
            }
        }
    }

    /**
     * Compact depth
     *
     * @access public
     */
    function compact_depth()
    {
        $pages = $this->get_metas('page');
        // performe compact separatory for each page
        foreach ($this->includes as $page => $tmp) {
            $keys = sonots::grep_array($page, $pages, 'eq');
            $headlines = array_intersect_key($this->headlines, $keys);
            $depths = sonots::get_members($headlines, 'depth');
            $depths = sonots::compact_list($depths);
            foreach ($depths as $key => $depth) {
                $this->headlines[$key]->depth = $depth;
            }
        }
    }

    /**
     * Get headlines of a page.
     *
     * Call get_headlines($page, $syntax). 
     * Other parameters are for recursive use. 
     *
     * @access public
     * @static
     * @param string $page
     * @param array $syntax definitions of pukiwiki syntax
     * @param string $rootpage
     * @param array $includes
     * @param boolean $detected
     * @return array
     */
    function get_headlines($page, $syntax, $rootpage = '', 
                           $includes = array(), $detected = FALSE)
    {
        if ($rootpage === '') $rootpage = $page;
        if (array_key_exists($page, $includes)) return array();
        if (! is_page($page)) return array();
        $lines = get_source($page);
        sonots::remove_multilineplugin_lines($lines);
        $includes[$page] = '';
        $headlines = array();
        foreach ($lines as $linenum => $line) {
            if ($rootpage === $page && ! $detected) { // for fromhere
                if (preg_match($syntax['contents'], $line, $matches)) {
                    $detected = TRUE;
                    continue;
                }
            }
            
            if (preg_match($syntax['headline'], $line, $matches)) {
                $depth    = strlen($matches[1]);
                list($string, $anchor) = sonots::make_heading($line);
                $headlines[] = new PluginSonotsHeadline
                    ($page, $linenum, $depth, $string, $anchor, $detected);
                continue;
            }

            if (preg_match($syntax['include'], $line, $matches)) {
                $inclargs = csv_explode(',', $matches[1]);
                $inclpage = array_shift($inclargs);
                $inclpage = get_fullname($inclpage, $page);
                if (! is_page($inclpage)) continue;
                $incloptions  = sonots::parse_options($inclargs, array('titlestr'=>'title'));
                $titlestr = PluginSonotsMetapage::linkstr($inclpage, $incloptions['titlestr']);
                $anchor = sonots::make_pageanchor($inclpage);
                $headlines[] = new PluginSonotsHeadline
                    ($inclpage, $linenum, 0, $titlestr, $anchor, $detected);
                list($includes, $inclheadlines) 
                    = PluginSonotsToc::get_headlines($inclpage, $syntax, $rootpage, $includes, $detected);
                $headlines = array_merge($headlines, $inclheadlines);
                continue;
            }

            if (preg_match($syntax['title'], $line, $matches)) {
                $title = $matches[1];
                $includes[$page] = $title;
                continue;
            }
        }
        return array($includes, $headlines);
    }
}

/**
 * Headline Structure
 *
 * @package    PluginSonots
 * @license    http://www.gnu.org/licenses/gpl.html GPL v2
 * @author     sonots <http://lsx.sourceforge.jp/>
 * @version    $Id: v 1.0 2008-06-05 11:14:46 sonots $
 */
class PluginSonotsHeadline
{
    var $page;
    var $linenum;
    var $depth;
    var $string;
    var $anchor;
    var $fromhere;
    function PluginSonotsHeadline(
        $page, $linenum, $depth, $string, $anchor = '', $fromhere = TRUE)
    {
        $this->page     = $page;
        $this->linenum  = $linenum;
        $this->depth    = $depth;
        $this->string   = $string;
        $this->anchor   = $anchor;
        $this->fromhere = $fromhere;
    }
}

?>
