/*
 *  m2m_thread.c:
 *     pthread(POSIX thread) routines.
 *
 *  Copyright (C) Taichi Nakamura <pdf30044@biglobe.ne.jp> - Feb 2000
 *
 *
 *  This file is part of m2m, a free MPEG2-Program-Stream player.
 *  It's a frontend of mpeg2dec.
 *    
 *  m2m is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  m2m is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 
 *
 */
#ifndef _LARGEFILE64_SOURCE
#define _LARGEFILE64_SOURCE
#endif
#ifndef _POSIX_C_SOURCE
#define _POSIX_C_SOURCE 199506L
#endif
#ifndef _XOPEN_SOURCE
#define _XOPEN_SOURCE  500
#endif

#include <stdlib.h>
#include <stdio.h>
#include <pthread.h>
#include <signal.h>
#include <stdint.h>
#include "m2m.h"

static m2m_buf_t *wk_buf=NULL;
static m2m_buf_t *abuf=NULL;
static m2m_buf_t *vbuf=NULL;
static m2m_buf_t *abuf_last=NULL;
static m2m_buf_t *vbuf_last=NULL;
int vcnt=0;
int acnt=0;
pthread_mutex_t bmutex= PTHREAD_MUTEX_INITIALIZER;
pthread_cond_t buf_cond=PTHREAD_COND_INITIALIZER;

void m2m_flush_buffer( )
{
  pthread_mutex_lock( &bmutex );
  if ( abuf && wk_buf ){
    m2m_buf_t * last;
    for ( last= wk_buf ; last->n ; last=last->n ) ;
    abuf->p = last;
    last->n = abuf;
  }else if( abuf ){
    wk_buf = abuf;
  }
  if ( vbuf && wk_buf ){
    m2m_buf_t * last;
    for ( last= wk_buf ; last->n ; last=last->n ) ;
    vbuf->p = last;
    last->n = vbuf;
  }else if( vbuf ){
    wk_buf = vbuf;
  }
  abuf=NULL;
  vbuf=NULL;
  pthread_mutex_unlock( &bmutex );
}

void m2m_free_buffer( m2m_buf_t *in )
{
  pthread_mutex_lock( &bmutex );
  in->p=NULL;
  in->n=wk_buf;
  if ( wk_buf ) wk_buf->p=in;
  wk_buf=in;
  pthread_mutex_unlock( &bmutex );
  sched_yield();
}

static inline m2m_buf_t * m2m_new_buffer( int isLocked )
{
    m2m_buf_t * ret ;
//fprintf(stderr,"m2m_new_buffer(%d)\n",isLocked );
    if ( !isLocked ) pthread_mutex_lock( &bmutex );
    ret = wk_buf;
    if ( ret ){
        wk_buf=wk_buf->n;
        if ( wk_buf ) wk_buf->p=NULL;
        ret->p=NULL;
        ret->n=NULL;
    }else{
        ret = (m2m_buf_t *)malloc( sizeof(m2m_buf_t) );
        memset( ret , 0x0, sizeof(m2m_buf_t) );
    }
    if( !isLocked ) pthread_mutex_unlock( &bmutex );
//fprintf(stderr,"m2m_new_buffer: %x\n",ret );
    return ret;
}

void m2m_add_last_buffer( m2m_buf_t * in ,int isLocked )
{
  if ( in == NULL ) return ;
//fprintf(stderr,"m2m_add_last_buffer:lock=%d %x\n",isLocked,in->type );
  if( !isLocked ) pthread_mutex_lock( &bmutex );
  if ( in->type == 0x0e0 ) {
    if ( vbuf_last ){
      in->p = vbuf_last;
      in->n = NULL;
      vbuf_last->n = in;
      vbuf_last = in;
    }else{
      in->p = NULL;
      in->n = NULL;
      vbuf=vbuf_last=in;
    }
    vcnt++;
  }else
  if ( in->type == 0x0c0  
    || in->type == 0x080   
    || in->type == 0x0a0 ) {
    if ( abuf ){
      in->p = abuf_last;
      in->n = NULL;
      abuf_last->n = in;
      abuf_last = in;
    }else{
      in->p = NULL;
      in->n = NULL;
      abuf=abuf_last= in;
    }
    acnt++;
  }
  if( !isLocked ) pthread_mutex_unlock( &bmutex );
//fprintf(stderr,"m2m_add_last_buffer:unlock \n" );
//fprintf(stderr," %03d %03d\r",vcnt,acnt );
  return ;
}

static inline m2m_buf_t * m2m_read_buffer( int isLocked )
{
    extern void DE_CSS(uint8_t []);
    uint32_t pes_ts;
    uint8_t *pes;
    int bytes_read,hdrlen,peslen ;
    m2m_buf_t * ret = m2m_new_buffer( isLocked );
    uint8_t * _local_buf=ret->buf;

    /* read for checking */
    for( bytes_read= 0 ; bytes_read == 0 ; bytes_read = 0 )
    {
//fprintf(stderr,"m2m_read_buffer:read sector \n" );
        bytes_read= read_sector(_local_buf);
//fprintf(stderr,"m2m_read_buffer:read sector %d \n",bytes_read );
        if( bytes_read == 0 ) return NULL;
        if( bytes_read != READ_BLOCK_SIZE ) kill(getpid(),SIGINT);
        if ( (_local_buf[4]&0xf0)==0x40 ){ //mpeg2
            pes = _local_buf + 14 + (_local_buf[13] & 7) ;
            if ( pes[0] != 0x00 || pes[1] != 0x00 || pes[2] != 0x01 )
            {
//fprintf(stderr,"m2m_read_buffer:ignore PES \n" );
                PACK_DUMP(_local_buf,"not PES(2):");
                continue;
            }
            if ( pes[3] == 0x0bb )
            {
                pes += ( pes[4] << 8 ) + pes[5] + 6 ;
                if ( pes[0] != 0x00 || pes[1] != 0x00 || pes[2] != 0x01 )
                {
//fprintf(stderr,"m2m_read_buffer:ignore PES \n" );
                        PACK_DUMP(_local_buf,"not PES(3):");
                        continue;
                }
            }
            peslen = ( pes[4] << 8 ) + pes[5] +6 ;
            hdrlen = pes[8] + 6 + 3 ;
            //DE_CSS( _local_buf ) ;
            pes_ts= ( pes[7] & 0x040 )?(GET_DTS(_local_buf)):
                    ( pes[7] & 0x080 )?(GET_PTS(_local_buf)):0;
        }else
        if ( (_local_buf[4]&0xf0)==0x20 ){ //mpeg1
            pes = _local_buf + 12;
            if ( pes[0] != 0x00 || pes[1] != 0x00 || pes[2] != 0x01 )
            {
//fprintf(stderr,"m2m_read_buffer:ignore PES \n" );
                PACK_DUMP(_local_buf,"not PES(2):");
                continue;
            }
            peslen = 2048-12;
            if ( pes[3] == 0x0bb )
            {
                pes += ( pes[4] << 8 ) + pes[5] + 6 ;
                if ( pes[0] != 0x00 || pes[1] != 0x00 || pes[2] != 0x01 )
                {
//fprintf(stderr,"m2m_read_buffer:ignore PES \n" );
                        PACK_DUMP(_local_buf,"not PES(3):");
                        continue;
                }
            }

            peslen = ( pes[4] << 8 ) + pes[5] +6 ;
            hdrlen = 6;
            while( pes[hdrlen] == 0xff ) hdrlen++;
            if ( (pes[hdrlen]&0xc0) == 0x40 ) hdrlen+=2;
            switch( pes[hdrlen] >>4 ){
            case 0: hdrlen++; break;
            case 2:
               pes_ts=(pes[hdrlen+1]<<24l)
                     |(pes[hdrlen+2]<<16l)
                     |(pes[hdrlen+3]<<8l)
                     |(pes[hdrlen+4]);
               hdrlen+=5;
               break;
            case 3:
               pes_ts=(pes[hdrlen+6]<<24l)
                     |(pes[hdrlen+7]<<16l)
                     |(pes[hdrlen+8]<<8l)
                     |(pes[hdrlen+9]);
               hdrlen+=10;
               break;
            }
            if ( hdrlen > peslen || peslen-hdrlen > 2048 ){
//fprintf(stderr,"m2m_read_buffer:ignore mp1 \n" );
                fprintf( stderr,"%d %d:",hdrlen,peslen );
                PACK_DUMP((pes+hdrlen),"mp1-1:bad");
                exit(2);
            }
        }else
        if ( _local_buf[3]==0x0f2 ){ // original
            return NULL;
        }else{
            PACK_DUMP((_local_buf),"unknown mpeg type");
            exit(3);
        }


        ret->type= pes[3]&0x0f0 ;
        if ( ret->type == 0x0e0    /* video */
          || ret->type == 0x0c0 )  /* mpeg Audio */
        {
            ret->ch  = pes[3]&0x0f ;
        }
        else
        if ( pes[3] == 0xbd    /* External Audio */
          && (pes[hdrlen]&0x0f0) == 0x080 ) /* AC3-Track */
        {
            ret->type= 0x080;
            ret->ch  = pes[hdrlen] & 0x0f;
            hdrlen += 4;
        }
        else
        if ( (pes[3] == 0xbd || pes[3] == 0xbf )    /* is External Audio */
         && (pes[hdrlen]&0x0f0) == 0x0a0 )
        {
            ret->type= 0x0a0;
            ret->ch  = pes[hdrlen] & 0x0f;
            while( ++hdrlen < peslen-1 ){
                if( pes[hdrlen] == 0x01 && pes[hdrlen+1] == 0x80 ){
                    hdrlen+=2;
                    break;
                }
            }
        }
        else
        {
//fprintf(stderr,"m2m_read_buffer:continue \n" );
            continue;
        }
        ret->p=NULL;
        ret->n=NULL;
        ret->pos=pes+hdrlen;
        ret->len=peslen-hdrlen;
        ret->ts=pes_ts;
        break;
    }
//fprintf(stderr,"m2m_read_buffer:done %x \n",ret );
    return ret;
}


int m2m_fill_buffer( int isLocked )
{
    m2m_buf_t * buf;
//fprintf(stderr,"m2m_fill_buffer.lock=%d \n",isLocked );
    if ( (buf=m2m_read_buffer(isLocked)) == NULL ){
      return -1;
    }
    m2m_add_last_buffer( buf ,isLocked ) ;
    return 0;
}

m2m_buf_t * m2m_next_buffer( int is_video )
{
  extern uint32_t volatile ts_v; /* video frame   DTS/PTS */
  extern uint32_t volatile ts_a; /* audio(master) PTS  */
  m2m_buf_t * ret;
  if ( is_video ) {
//fprintf(stderr,"m2m_next_buffer:v lock %x \n",vbuf );
    pthread_mutex_lock( &bmutex );
    while ( vbuf == NULL ){
      if ( abuf == NULL ){
        pthread_mutex_unlock( &bmutex );
        pthread_cond_signal( &buf_cond );
        pthread_mutex_lock( &bmutex );
      }
      if ( m2m_fill_buffer( 1 ) == -1 ){
        ts_v=0;
        while( abuf ) pthread_cond_wait( &buf_cond,&bmutex );
      }
//fprintf(stderr,"m2m_next_buffer:v filled %x\n",vbuf );
    }
    ret=vbuf;
    vbuf=vbuf->n;
    if( vbuf ) vbuf->p=NULL;
    else vbuf_last=NULL;
    ret->n=NULL;
    vcnt--;
//fprintf(stderr,"m2m_next_buffer:v unlock %x \n",vbuf );
    pthread_mutex_unlock( &bmutex );
  }else{
//fprintf(stderr,"m2m_next_buffer:a %x \n",abuf );
    pthread_mutex_lock( &bmutex );
    while ( abuf == NULL ){
      if ( vbuf == NULL ){
        pthread_mutex_unlock( &bmutex );
        pthread_cond_signal( &buf_cond );
        pthread_mutex_lock( &bmutex );
      }
      if ( m2m_fill_buffer( 1 ) == -1 ){
        ts_a=0;
        while( vbuf ) pthread_cond_wait( &buf_cond,&bmutex );
      }
//fprintf(stderr,"m2m_next_buffer:a filled %x\n",abuf );
    }
//fprintf(stderr,"m2m_next_buffer:a lock %x \n",abuf );
    ret=abuf;
    abuf=abuf->n;
    if( abuf ) abuf->p=NULL;
    else abuf_last=NULL;
    ret->n=NULL;
    acnt--;
//fprintf(stderr,"m2m_next_buffer:a unlock %x \n",abuf );
    pthread_mutex_unlock( &bmutex );
  }
//fprintf(stderr," %03d %03d\r",vcnt,acnt );
  return ret;
}

void * m2m_reader_thread( void *arg )
{
  while(1){
    m2m_fill_buffer(0);
  }
}

