/*
 *  m2m_audio.c:
 *     audio routines.
 *
 *  Copyright (C) Taichi Nakamura <pdf30044@biglobe.ne.jp> - Feb 2000
 *
 *
 *  This file is part of m2m, a free MPEG2-Program-Stream player.
 *  It's a frontend of mpeg2dec.
 *    
 *  m2m is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  m2m is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 
 *
 */
#ifndef _LARGEFILE64_SOURCE
#define _LARGEFILE64_SOURCE
#endif
#ifndef _POSIX_C_SOURCE
#define _POSIX_C_SOURCE 199506L
#endif
#ifndef _XOPEN_SOURCE
#define _XOPEN_SOURCE  500
#endif

#include <features.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
//include <sys/time.h>
//include <sys/mman.h>
#include <netinet/in.h>
#include <unistd.h>
//include <stdlib.h>
#include <stdio.h>
//include <fcntl.h>

#include "config.h"
#include "m2m.h"
#ifdef USE_a52dec
#include <a52dec/a52.h>
#include <a52dec/audio_out.h>
#include <a52dec/mm_accel.h>
#endif
#ifdef HAVE_LIBASOUND
#include <sys/asoundlib.h>
#endif
#ifdef HAVE_ESOUND
#include <esd.h>
#endif


static double m2m_audio_downcast=0.0;
static ao_driver_t * libao_driver=NULL;
struct oss_instance_s {
   ao_instance_t ao;
   int fd;
   int sample_rate;
   int set_params;
   int flags;
} ;
static a52_state_t * m2m_a52_state=NULL;

static inline int16_t convert (int32_t i)
{
    if (i > 0x43c07fff)
        return 32767;
    else if (i < 0x43bf8000)
        return -32768;
    else
        return i - 0x43c00000;
}

static inline void float_to_int (float * _f, int16_t * s16, int flags)
{

    int i;
    int32_t * f = (int32_t *) _f;
    switch (flags) {
    case A52_MONO:
        for (i = 0; i < 256; i++) {
            s16[5*i] = s16[5*i+1] = s16[5*i+2] = s16[5*i+3] = 0;
            s16[5*i+4] = convert (f[i]);
        }
        break;
    case A52_CHANNEL:
    case A52_STEREO:
    case A52_DOLBY:
        for (i = 0; i < 256; i++) {
            s16[2*i] = convert (f[i]);
            s16[2*i+1] = convert (f[i+256]);
        }
        break;
    case A52_3F:
        for (i = 0; i < 256; i++) {
            s16[5*i] = convert (f[i]);
            s16[5*i+1] = convert (f[i+512]);
            s16[5*i+2] = s16[5*i+3] = 0;
            s16[5*i+4] = convert (f[i+256]);
        }
        break;
    case A52_2F2R:
        for (i = 0; i < 256; i++) {
            s16[4*i] = convert (f[i]);
            s16[4*i+1] = convert (f[i+256]);
            s16[4*i+2] = convert (f[i+512]);
            s16[4*i+3] = convert (f[i+768]);
        }
        break;
    case A52_3F2R:
        for (i = 0; i < 256; i++) {
            s16[5*i] = convert (f[i]);
            s16[5*i+1] = convert (f[i+512]);
            s16[5*i+2] = convert (f[i+768]);
            s16[5*i+3] = convert (f[i+1024]);
            s16[5*i+4] = convert (f[i+256]);
        }
        break;
    case A52_MONO | A52_LFE:
        for (i = 0; i < 256; i++) {
            s16[6*i] = s16[6*i+1] = s16[6*i+2] = s16[6*i+3] = 0;
            s16[6*i+4] = convert (f[i+256]);
            s16[6*i+5] = convert (f[i]);
        }
        break;
    case A52_CHANNEL | A52_LFE:
    case A52_STEREO | A52_LFE:
    case A52_DOLBY | A52_LFE:
        for (i = 0; i < 256; i++) {
            s16[6*i] = convert (f[i+256]);
            s16[6*i+1] = convert (f[i+512]);
            s16[6*i+2] = s16[6*i+3] = s16[6*i+4] = 0;
            s16[6*i+5] = convert (f[i]);
        }
        break;
    case A52_3F | A52_LFE:
        for (i = 0; i < 256; i++) {
            s16[6*i] = convert (f[i+256]);
            s16[6*i+1] = convert (f[i+768]);
            s16[6*i+2] = s16[6*i+3] = 0;
            s16[6*i+4] = convert (f[i+512]);
            s16[6*i+5] = convert (f[i]);
        }
        break;
    case A52_2F2R | A52_LFE:
        for (i = 0; i < 256; i++) {
            s16[6*i] = convert (f[i+256]);
            s16[6*i+1] = convert (f[i+512]);
            s16[6*i+2] = convert (f[i+768]);
            s16[6*i+3] = convert (f[i+1024]);
            s16[6*i+4] = 0;
            s16[6*i+5] = convert (f[i]);
        }
        break;
    case A52_3F2R | A52_LFE:
        for (i = 0; i < 256; i++) {
            s16[6*i] = convert (f[i+256]);
            s16[6*i+1] = convert (f[i+768]);
            s16[6*i+2] = convert (f[i+1024]);
            s16[6*i+3] = convert (f[i+1280]);
            s16[6*i+4] = convert (f[i+512]);
            s16[6*i+5] = convert (f[i]);
        }
        break;
    }
}

#ifdef HAVE_LIBASOUND
static snd_pcm_t *front_handle=NULL;
static uint32_t ao_open_alsa(uint32_t bits, uint32_t rate, uint32_t channels)
{
  snd_pcm_format_t pcm_format;
  snd_pcm_channel_setup_t pcm_chan_setup;
  snd_pcm_channel_params_t pcm_chan_params;
  snd_pcm_channel_info_t pcm_chan_info;
  int pcm_default_device;
  int pcm_default_card;
  int bfsize;
  int fsize;
  int err;
  if(front_handle != NULL) {
      return 1;
  }

  if((pcm_default_device = snd_defaults_pcm_device()) < 0) {
    fprintf(stderr,"There is no default pcm device.\n");
    return 0;
  }
  if((pcm_default_card = snd_defaults_pcm_card()) < 0) {
    fprintf(stderr,"There is no default pcm card.\n");
    return 0;
  }
  if((err = snd_pcm_open_subdevice(&front_handle, pcm_default_card,
                        pcm_default_device, 0,
                        SND_PCM_OPEN_PLAYBACK| SND_PCM_OPEN_NONBLOCK)) < 0){
    fprintf(stderr,"snd_pcm_open_subdevice() failed: %s\n", snd_strerror(err));
    return 0;
  }

  memset(&pcm_chan_info, 0, sizeof(snd_pcm_channel_info_t));
  if((err = snd_pcm_channel_info(front_handle, &pcm_chan_info)) < 0) {
    fprintf(stderr,"snd_pcm_channel_info() failed: %s\n", snd_strerror(err));
    return 0;
  }
  memset(&pcm_chan_params, 0, sizeof(snd_pcm_channel_params_t));
  memset(&pcm_format, 0, sizeof(snd_pcm_format_t));
  pcm_format.format = (bits==8)?SND_PCM_SFMT_S8: (bits==24)?SND_PCM_SFMT_S24:
                      (bits==32)?SND_PCM_SFMT_S32:SND_PCM_SFMT_S16;
  pcm_format.voices = channels;
  pcm_format.rate = rate;
  pcm_format.interleave = 1;

  if(rate > pcm_chan_info.max_rate) {
    fprintf(stderr,"\n!!!! WARNING: device only seems to support %d "
         "samples/sec on this channel\n\n", pcm_chan_info.max_rate);
  }

  bfsize=0 ;
  fsize = 8192;
  while (fsize>0) {
    fsize /=2;
    bfsize++;
  }
  bfsize--;
  bfsize = (8192 << 16) | bfsize ;
  memcpy(&pcm_chan_params.format, &pcm_format, sizeof(snd_pcm_format_t));
  pcm_chan_params.mode = SND_PCM_MODE_STREAM;
  pcm_chan_params.channel = SND_PCM_CHANNEL_PLAYBACK;
  pcm_chan_params.start_mode = SND_PCM_START_DATA;
  pcm_chan_params.stop_mode = SND_PCM_STOP_ROLLOVER;
  pcm_chan_params.buf.block.frag_size = bfsize;
  pcm_chan_params.buf.block.frags_max = bfsize+1;
  pcm_chan_params.buf.block.frags_min = 1;

  snd_pcm_channel_flush(front_handle, SND_PCM_CHANNEL_PLAYBACK);
  if((err = snd_pcm_channel_params(front_handle, &pcm_chan_params)) <0){
    fprintf(stderr,"snd_pcm_channel_params() failed: %s\n", snd_strerror(err));
    return 0;
  }
  if((err = snd_pcm_channel_prepare(front_handle, SND_PCM_CHANNEL_PLAYBACK)) < 0
){
    fprintf(stderr,"snd_pcm_channel_prepare() failed: %s\n", snd_strerror(err));
    return 0;
  }

  pcm_chan_setup.mode = SND_PCM_MODE_STREAM;
  pcm_chan_setup.channel = SND_PCM_CHANNEL_PLAYBACK;
  if((err = snd_pcm_channel_setup(front_handle, &pcm_chan_setup))<0){
    fprintf( stderr,"snd_pcm_channel_setup() failed: %s\n", snd_strerror(err) );
    return 0;
  }

  return 1;
}
 
static int ao_play_alsa(ao_instance_t * _instance, int flags, sample_t * _samples)
{
  int16_t int16_samples[256*6];
  snd_pcm_channel_status_t status_front;
  int err;
  int num_samples=256;
  int channels;
  int rate;
  int bits;

  ao_open_alsa(16, ((struct oss_instance_s *)_instance)->sample_rate, 2);

  status_front.channel = SND_PCM_CHANNEL_PLAYBACK;
  if((err = snd_pcm_channel_status(front_handle, &status_front)) < 0) {
    fprintf(stderr,"snd_pcm_channel_status() failed: %s\n", snd_strerror(err));
  }
  if ( status_front.count > 100000 )
    usleep( (status_front.count-100000) *1000/48/4);

  float_to_int (_samples, int16_samples, flags);
  snd_pcm_write(front_handle, int16_samples, num_samples<<1);

  memset(&status_front, 0, sizeof(snd_pcm_channel_status_t));
  if((err = snd_pcm_channel_status(front_handle, &status_front)) < 0){
    fprintf(stderr,"snd_pcm_channel_status() failed: %s\n", snd_strerror(err));
  }
  if(status_front.underrun) {
    fprintf(stderr,"underrun, resetting front channel\n");
    snd_pcm_channel_flush(front_handle, SND_PCM_CHANNEL_PLAYBACK);
    snd_pcm_playback_prepare(front_handle);

    snd_pcm_write(front_handle, int16_samples, num_samples<<1);
    if((err = snd_pcm_channel_status(front_handle, &status_front)) < 0) {
      fprintf(stderr,"snd_pcm_channel_status() failed: %s", snd_strerror(err));
    }
    if(status_front.underrun) {
      fprintf(stderr,"front write error, giving up\n");
    }
  }
  return 0;
}
#endif

#ifdef HAVE_ESOUND
static int ao_open_esound( struct oss_instance_s *ao )
{
  esd_format_t esd_format;
  long flags ;
  int tmp;

  close(ao->fd);
  
  esd_format = ESD_STREAM | ESD_PLAY ;
  esd_format |= ESD_BITS16 ;
  esd_format |=  ESD_STEREO ;

  if( audio_rate ) tmp= ao->sample_rate=audio_rate;
  else             tmp= ao->sample_rate=48000;
  m2m_audio_downcast=(ao->sample_rate-tmp)/(double)ao->sample_rate;
  if ( m2m_audio_downcast > 0.0 )
    fprintf( stderr,"downcast %d to %d =%lf\n",ao->sample_rate,audio_rate,m2m_audio_downcast );

  ao->fd = esd_play_stream(esd_format, tmp,esd_host,"m2m esound");
  if( ao->fd < 0 )
  {
    perror( "esd_play_stream");
    fprintf(stderr, "error: esound open stream failed %x %d %s\n",esd_format, tmp,esd_host);
    return 0;
  }

/*
  flags = fcntl(m2m_audio_fd, F_GETFL);
  if((flags < 0) || (fcntl(m2m_audio_fd, F_SETFL, flags|O_NONBLOCK) < 0))
  {
    perror("Esound-driver, error: fnctl");
    close(m2m_audio_fd);
    m2m_audio_fd=-1;
    return 0;
  }
*/
  return 1;
}

static int ao_play_esound (ao_instance_t * _instance, int flags, sample_t * _samples)
{
int i;
    int16_t int16_samples[256*6];
    float_to_int (_samples, int16_samples, flags);
    write (((struct oss_instance_s *) _instance)->fd, int16_samples, 256 * sizeof (int16_t) *2);
    return 0;
}
#endif

struct oss_instance_s * audio_out_a52_init(void)
{
    extern char *ao_driver;
    uint32_t accel=MM_ACCEL_X86_MMX|MM_ACCEL_X86_MMXEXT;
    struct oss_instance_s *ao;
    ao_driver_t *drv=ao_drivers();
    while( ao_driver && drv->name && strcmp(ao_driver,drv->name) ) drv++;
    if( drv->name == NULL ) drv=ao_drivers();
    printf("select audio:%s\n",drv->name );
    ao= (struct oss_instance_s *)ao_open( drv->open  );
    m2m_a52_state = a52_init( accel );
#ifdef HAVE_LIBASOUND
    if ( IS_ALSA ){
     ao->ao.play=ao_play_alsa;
    }else
#endif
#ifdef HAVE_ESOUND
    if( IS_ESD ){
     ao_open_esound(ao);
     ao->ao.play=ao_play_esound;
    }
#endif
    return ao;
}

void a52_decode_data (struct oss_instance_s * ao,uint8_t * start, uint8_t * end)
{

    static uint8_t buf[3840];
    static uint8_t * bufptr = buf;
    static uint8_t * bufpos = buf + 7;
    static int sample_rate;
    static int flags;
    int bit_rate;

    while (start < end) {
        *bufptr++ = *start++;
        if (bufptr == bufpos) {
            if (bufpos == buf + 7) {
                int length;

                length = a52_syncinfo (buf, &flags, &sample_rate, &bit_rate);
                if (!length) {
                    fprintf (stderr, "skip\n");
                    for (bufptr = buf; bufptr < buf + 6; bufptr++)
                        bufptr[0] = bufptr[1];
                    continue;
                }
                bufpos = buf + length;
            } else {
                sample_t level, bias;
                int i;

                if (ao_setup ((ao_instance_t *)ao, sample_rate, &flags, &level, &bias)){
                    fprintf (stderr, "ao_setup:%d %x %x",sample_rate,flags,bias);
                    goto error;
                }
                flags |= A52_ADJUST_LEVEL;
                if (a52_frame (m2m_a52_state, buf, &flags, &level, bias)){
                    fprintf (stderr, "ao_setup");
                    goto error;
                }
                // disable dynamic range compression
                //a52_dynrng (m2m_a52_state, NULL, NULL);
                for (i = 0; i < 6; i++) {
                    if (a52_block (m2m_a52_state)){
                        fprintf (stderr, "a52block");
                        goto error;
                    }
                    if (ao_play ((ao_instance_t *)ao, flags, a52_samples(m2m_a52_state))){
                        fprintf (stderr, "ao_play");
                        goto error;
                    }
                }
                bufptr = buf;
                bufpos = buf + 7;
                continue;
            error:
                fprintf (stderr, "error\n");
                bufptr = buf;
                bufpos = buf + 7;
            }
        }
    }
}

static int32_t f[512];
static int f_len=0;
void ao_play_pcm( ao_instance_t * _ao,void * s16, int length)
{
    int i;
    struct oss_instance_s * ao =(struct oss_instance_s *)_ao;
    uint16_t * u = s16;
    int16_t * s = s16;
    for( i=0 ; i<length ; i++ ){
      u[i*2]       = ntohs( u[i*2] );
      u[i*2+1]     = ntohs( u[i*2+1] );
      f[f_len]     =  s[i*2]  +0x43c00000;
      f[f_len+256] =  s[i*2+1]+0x43c00000;
      if ( ++f_len >= 256 ){
          ao->sample_rate=48000;
          ao_play( (ao_instance_t *)ao,A52_STEREO,(float *)f );
          f_len=0;
      }
    }
}

