/*
 *  m2m_properties.c:
 *     pthread(POSIX thread) routines.
 *
 *  Copyright (C) Taichi Nakamura <pdf30044@biglobe.ne.jp> - Feb 2000
 *
 *
 *  This file is part of m2m, a free MPEG2-Program-Stream player.
 *  It's a frontend of mpeg2dec.
 *    
 *  m2m is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  m2m is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 
 *
 */
#ifndef _LARGEFILE64_SOURCE
#define _LARGEFILE64_SOURCE
#endif
#ifndef _POSIX_C_SOURCE
#define _POSIX_C_SOURCE 199506L
#endif
#ifndef _XOPEN_SOURCE
#define _XOPEN_SOURCE  500
#endif

#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <stdint.h>
#include "m2m.h"

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <fcntl.h>

static char *path=NULL;
typedef struct m2m_prop_s {
  char * name;
  char * value;
  struct m2m_prop_s * next;
} m2m_prop_t;
m2m_prop_t * m2m_props=NULL;

char *m2m_get_property( char in_name[] ){
  struct m2m_prop_s *cur;
  if ( in_name == NULL ) return NULL;
  for( cur=m2m_props ; cur != NULL && strcmp(in_name,cur->name) != 0 ; cur=cur->next);
  return cur?cur->value:NULL;
}

int m2m_set_property( char in_name[] , char in_value[] ){
  struct m2m_prop_s *cur;
  char *wk;
  for( cur=m2m_props ; cur != NULL && strcmp(in_name,cur->name) != 0 ; cur=cur->next);
  if ( cur ){
    wk=(char *)malloc( (strlen(in_value)+1)*sizeof(char) );
    if ( wk == NULL ) return -1;
    strcpy( wk,in_value);
    if ( cur->value ) free(cur->value);
    cur->value=wk;
  }else{
    cur=malloc( sizeof(struct m2m_prop_s) );
    if ( cur == NULL ) return -1;
    cur->name =(char *)malloc( (strlen(in_name )+1)*sizeof(char) );
    cur->value=(char *)malloc( (strlen(in_value)+1)*sizeof(char) );
    if ( cur->name == NULL || cur->value == NULL ){
      if ( cur->name  ) free(cur->name);
      if ( cur->value ) free(cur->value);
      free(cur);
      return -1;
    }
    strcpy(cur->name ,in_name );
    strcpy(cur->value,in_value);
    cur->next=m2m_props;
    m2m_props=cur;
  }
}

int m2m_remove_property( char in_name[] ){
  struct m2m_prop_s *cur=m2m_props;
  struct m2m_prop_s *pre=NULL;
  for( ; cur != NULL && strcmp(in_name,cur->name) != 0 ; pre=cur,cur=cur->next);
  if( cur == NULL ) return -1;

  if ( pre ) pre->next=cur->next;
  else  m2m_props=cur->next;

  free(cur->name);
  free(cur->value);
  free(cur);

  return 0;
}

static char *m2m_read_prop_name( char **pos,char *eod ){
  char *ret=NULL;
  int len,pad;
  if ( pos == NULL || *pos == NULL ) return NULL;
  for( ; *pos != eod && **pos < ',' ; (*pos)++ );
  for( len=0 ; (*pos)+len != eod && (*pos)[len] >= ',' && (*pos)[len] <= 'z' && (*pos)[len] != '=' ; len++ );
  for( pad=0 ; (*pos)+len+pad != eod && (*pos)[len+pad] == ' ' || (*pos)[len+pad] == '\t' ; pad++ );
  if ( *pos != eod && (*pos)[len+pad] == '=' ){
    ret=(char *)malloc( (len+1) * sizeof(char) );
  }
  if ( ret ){
    memset(ret,0x0,(len+1) * sizeof(char) );
    strncpy(ret,*pos,len);
  }
  if ( *pos != eod ){
      (*pos) += len+pad+1;
  }
  return ret;
}

static char *m2m_read_prop_value( char **pos,char *eod ){
  char *ret=NULL;
  int len,pad;
  if ( pos == NULL || *pos == NULL ) return NULL;
  for( ; *pos != eod && **pos <= ' ' ; (*pos)++ );
  for( len=0 ; (*pos)+len != eod && (*pos)[len] && (*pos)[len] != '\r' && (*pos)[len] != '\n' ; len++ );
  for( pad=0 ; (*pos)+len+pad != eod && ((*pos)[len+pad] == '\r' || (*pos)[len+pad] == '\n') ; pad++ );
  for( ; len > 0 && ((*pos)[len] == ' ' || (*pos)[len] == '\t') ; len--,pad++);
  if ( len > 0  ){
    ret=(char *)malloc( (len+1) * sizeof(char) );
  }
  if ( ret ){
    memset(ret,0x0,(len+1) * sizeof(char) );
    strncpy(ret,*pos,len);
  }
  if ( *pos != eod ){
    (*pos) += len+pad;
  }
  return ret;
}

int m2m_load_properties( char in_path[] ) {
  char *default_path[]={ "~/.m2m.conf","/usr/share/m2m/m2m.conf" };
  int fp;
  int i;
  char *wk;
  char name[4096];
  void *mapaddr;
  struct stat stbuf;

  for( i=0 ; i<3 ; i++ ){
    switch(i){
    case 0:
      if( in_path == NULL ) continue;
      strncpy( name,in_path,4096);
      break;
    case 1:
      if( getenv("HOME") == NULL ) continue;
      strcpy( name,getenv("HOME") );
      strcat( name,"/.m2m.conf" );
      break;
    case 2:
      strcpy(  name,"/usr/share/m2m/m2m.conf" );
      break;
    }
    if ( (fp=open( name,O_RDONLY)) <= 0 ){
      perror("open");
      fprintf( stderr,"can not open[%s]\n",name );
      continue;
    }
fprintf( stderr,"opened[%s]\n", name );
    if ( fstat( fp,&stbuf ) != 0 ||
     (mapaddr=mmap( NULL,stbuf.st_size,(PROT_READ),MAP_SHARED,fp,0 )) == MAP_FAILED ){
      perror("fstat/mmap");
      close(fp);
      continue;
    }
    path=malloc( (strlen(name)+1)*sizeof(char) );
    if ( path == NULL ){
      perror("malloc");
      munmap(mapaddr,stbuf.st_size);
      close(fp);
    }
    strcpy(path,name);
    break;
  }
  if ( fp < 0 ){ fprintf( stderr,"no property file %s .\n",name); exit(3); }

  for( wk=mapaddr ; wk != mapaddr+stbuf.st_size ; ){
    char *n;
    char *v;
    n=m2m_read_prop_name(&wk,mapaddr+stbuf.st_size);
    if ( n == NULL ) continue;
    v=m2m_read_prop_value(&wk,mapaddr+stbuf.st_size);
    m2m_set_property(n,v);
  }
  munmap(mapaddr,stbuf.st_size);
  close(fp);
}

void m2m_write_properties( int fd )
{
  struct m2m_prop_s *cur;
  for( cur=m2m_props ; cur ; cur=cur->next ){
    write( fd,cur->name,strlen(cur->name)*sizeof(char) );
    write( fd,"=",sizeof(char) );
    write( fd,cur->value,strlen(cur->value)*sizeof(char) );
    write( fd,"\n",sizeof(char) );
  }
}

int m2m_save_properties( char in_path[] ) {
  int fd=0;
  if ( m2m_props == NULL ) return;
  if ( in_path ){
    fd=open( in_path,O_CREAT);
  }else if ( path ){
    fd=open(path,O_CREAT);
  }
  if ( fd <= 0 ) return -1;
  m2m_write_properties(fd);
  close(fd);
}

