package jp.sourceforge.ma38su.btree;

import java.util.Comparator;

/**
 * B-Tree
 * 
 * すべてオンメモリで構成しているので現在のところ実用性はあまりありません。
 * 再帰を多用しているのと、エントリーの挿入位置を前から順に確認しているところが問題です。
 * toStringの実装が十分ではありません。
 * 
 * もしエントリーEが、Comparable<E>を実装しているなければ、
 * Comparator<? super E>を与えなければなりません。
 * 
 * @author ma38su
 * @param <E> エントリーのクラス
 */
public class BTree<E> {

	/**
	 * ルートページ
	 */
	private Page<E> root;
		
	/**
	 * B-treeのコンストラクタです。
	 * ページの最小エントリー数を指定します。
	 * ただし、ルートページに限り、最小エントリー数の制約を受けません。
	 * 最大エントリー数は最小エントリー数の2倍になります。
	 * 
	 * エントリーのクラスEがComparableを実装していない場合には、
	 * Comparatorの指定が必須です。
	 * 
	 * @param n 最小エントリー数（自然数）
	 */
	public BTree(int n) {
		this(n, null);
	}

	
	/**
	 * ページを再帰的にたどりエントリーの削除を行います。
	 * @param entry 削除するエントリー
	 * @return 削除すればtrueを返します。
	 */
	public boolean remove(E entry) {
		E result = this.root.remove(entry);
		this.root = this.root.getRoot();
		if (result == null) {
			return false;
		} else {
			return true;
		}
	}
	
	/**
	 * B-treeのコンストラクタです。
	 * ページの最小エントリー数を指定します。
	 * ただし、ルートページに限り、最小エントリー数の制約を受けません。
	 * 最大エントリー数は最小エントリー数の2倍になります。
	 * 
	 * エントリーのクラスEがComparableを実装している場合にも、
	 * Comparatorによる比較が優先されます。
	 * 
	 * @param n 最小エントリー数（自然数）
	 * @param comparator Eのコンパレータ
	 */
	public BTree(int n, Comparator<? super E> comparator) {
		if (n <= 1) {
			throw new IllegalArgumentException("次数は2以上にのみ対応しています。");
		}
		this.root = new Page<E>(n, comparator);
	}

	/**
	 * エントリーを挿入します。
	 * 重複したエントリーが存在する場合には、挿入に失敗します。
	 * 
	 * @param entry エントリー
	 * @return 挿入に成功すればtrue
	 */
	public boolean add(E entry) {
		boolean result = this.root.insert(entry);
		this.root = this.root.getRoot();
		return result;
	}
	
	/**
	 * エントリーが含まれているかどうか確認します。
	 * 
	 * @param entry エントリー
	 * @return エントリーが含まれていれば、trueを返します。
	 */
	public boolean contains(E entry) {
		return this.root.contains(entry);
	}

	@Override
	public String toString() {
		return this.root.toString();
	}
}
