package jp.sourceforge.ma38su.btree;

import java.util.Arrays;
import java.util.Comparator;

/**
 * B-Treeのページクラス
 * 
 * コンストラクタを呼び出す際に、最小エントリー数を指定する必要があります。
 * ページがルートの場合を除き、最大エントリー数は最小エントリー数の2倍です。
 * なお、要素の重複は許しません。
 * 
 * エントリーは、Comparableを実装しているか、Comparatorを与えなければならない。
 * 
 * @author ma38su
 * @param <E> エントリーのクラス
 */
class Page<E> {
	
	/**
	 * 子ページの配列
	 * 子ページは、それぞれひとつ小さいインデックスのエントリよりも大きく、
	 * かつ、同じインデックスのエントリーよりも小さいエントリーの配列を持ちます。
	 * そのための子ページは、エントリーの配列よりも1長くなっています。
	 */
	private final Page<E>[] children;

	private final Comparator<? super E> comparator;
	
	/**
	 * ページの保持するエントリーの配列
	 * エントリーは常に左寄せで持ちます。
	 */
	private final E[] entries;
		
	/**
	 * 親ページ
	 */
	private Page<E> parent;
	
	/**
	 * ページの保持しているエントリー数
	 */
	private int size;
	
	/**
	 * コンストラクタ
	 * ページの保持する最小エントリー数を指定する必要があります。
	 * ページの保持できる最大エントリー数は最小エントリー数の2倍になる。
	 * 
	 * @param n 最小のエントリー数（ルートページは除く）
	 * @param comparator コンパレータ
	 */
	@SuppressWarnings("unchecked")
	Page(int n, Comparator<? super E> comparator) {
		this.comparator = null;
		this.entries = (E[]) new Object[n << 1];
		this.children = new Page[n << 1 + 1];
		this.size = 0;
	}

	/**
	 * 親子関係の更新をおこないます。
	 * ページの分割時に呼び出します。
	 * @param index 子ページの位置
	 * @param child 子ページ
	 */
	private void adoption(int index, Page<E> child) {
		this.children[index] = child;
		if (child != null) {
			child.parent = this;
		}
	}

	/**
	 * エントリーが含まれているかどうか再帰的に子ページまで確認します。
	 * 
	 * @param entry エントリー
	 * @return エントリーが含まれていればtrueを返します。
	 */
	boolean contains(E entry) {
		int index = this.getIndex(entry);
		if (index < 0) {
			return true;
		}
		if (this.children[index] != null) {
			return this.children[index].contains(entry);
		}
		return false;
	}

	/**
	 * エントリーの挿入位置をバイナリサーチによって求めます。
	 * もし、エントリーが重複する場合の返り値は、
	 * 重複エントリーのインデックスに位置加えた後、負の値にして返します。
	 * （そうしないと重複エントリーのインデックスが0の場合に問題が起こります。）
	 * 次数1の場合の動作がテストできていません。
	 * 
	 * @param entry 挿入位置を検索するエントリー
	 * @return 挿入位置、重複する場合は負の値になる。重複するインデックスは、Math.abs(result + 1)で求まる。
	 */
	@SuppressWarnings("unchecked")
	private int getIndex(E entry) {
		int low = 0;
		int upper = this.size;
		int mid = 0;
		int result;
		try {
			while (low < upper) {
				mid = (low + upper) >> 1;
				if (this.comparator == null) {
					result = ((Comparable<E>) entry).compareTo(this.entries[mid]);
				} else {
					result = this.comparator.compare(entry, this.entries[mid]);
				}
				if (result == 0) {
					/* エントリーが重複していた場合 */
					return - mid - 1;
				} else if (result < 0) {
					upper = mid;
				} else {
					low = mid + 1;
				}
			}
		} catch (Exception e) {
			System.out.println(this);
			System.out.println("entry: "+ entry);
			System.out.println("entries.length: "+ this.entries.length);
			System.out.println("mid: "+ mid);
			e.printStackTrace(System.out);
			System.exit(0);
		}
		/* ループを抜けるとhighとrowの値は一致する */
		return upper;
	}
	
	
	
	/**
	 * 子のページのうち最大のエントリーを持つページを取得します。
	 * @return 最大のエントリーを持つページ
	 */
	private Page<E> getMaxiumPage() {
		if (this.children[this.size] != null) {
			return this.children[this.size].getMaxiumPage();
		} else {
			return this;
		}
	}	
	
	/**
	 * 子のページのうち最小のエントリーを持つページを取得します。
	 * @return 最小のエントリーを持つページ
	 */
	private Page<E> getMinimumPage() {
		if (this.children[0] == null) {
			return this;
		} else {
			return this.children[0].getMinimumPage();
		}
	}
	/**
	 * ルートページを返します。
	 * @return ルートページ
	 */
	Page<E> getRoot() {
		Page<E> page = this;
		if (page.parent != null && page.entries[0] == null) {
			throw new UnknownError("致命的なエラーが起こりました。");
		}
		while (page.entries[0] == null) {
			page = page.children[0];
		}
		while (page.parent != null) {
			page = page.parent;
		}
		return page;
	}

	/**
	 * エントリーを挿入します。
	 * @param entry 挿入するエントリー
	 * @return 挿入に成功すればtrue
	 */
	boolean insert(E entry) {
		int index = this.getIndex(entry);
		if (index < 0) {
			return false;
		}
		/* ループを抜けるとhighとrowの値は一致する */
		if (this.children[index] != null) {
			/* 子ページを持つ場合 */
			return this.children[index].insert(entry);
		} else {
			/* 子ページを持たない場合 */
			this.insertDetail(index, entry, null, null);
			return true;
		}
	}

	/**
	 * 挿入位置と子ページを指定してエントリーを挿入します。
	 * 必要に応じてページは再帰的に分割されます。
	 * 子ノードをたどることはありません。
	 *
	 * @param entry 挿入するエントリー
	 * @param index エントリーの挿入位置
	 * @param leftson エントリーの左の子ページ
	 * @param rightson エントリーの右の子ページ
	 */
	private void insertDetail(int index, E entry, Page<E> leftson, Page<E> rightson) {
		if (this.size != this.entries.length) {
			/* ページの分割が必要ない時 */
			for (int j = this.size; j > index; j--) {
				this.entries[j] = this.entries[j - 1];
				this.children[j + 1] = this.children[j];
			}
			this.entries[index] = entry;
			this.children[index] = leftson;
			this.children[index + 1] = rightson;
			if (leftson != null) {
				leftson.parent = this;
			}
			if (rightson != null) {
				rightson.parent = this;
			}
			this.size++;
		} else {
			/* ページの分割が必要な時 */
			int minLength = this.entries.length >> 1;
			Page<E> page = new Page<E>(minLength, this.comparator);
			E center;
			if (minLength == index) {
				/* entryでページを分割して挿入する場合 */
				center = entry;
				page.children[0] = rightson;
				if (rightson != null) {
					rightson.parent = page;
				}
				for (int j = 0; j < minLength; j++) {
					page.entries[j] = this.entries[minLength + j];
					this.entries[minLength + j] = null;
					page.adoption(j + 1, this.children[minLength + j + 1]);
					this.children[minLength + j + 1] = null;
				}
				this.children[minLength] = leftson;
				if (leftson != null) {
					leftson.parent = this;
				}
			} else if (minLength > index) {
				/* ページの前半にentryを挿入する場合 */
				center = this.entries[minLength - 1];
				page.adoption(0, this.children[minLength]);
				for (int j = minLength - 1; j > index; j--) {
					this.entries[j] = this.entries[j - 1];
					this.children[j + 1] = this.children[j];
				}
				this.entries[index] = entry;
				for (int j = 0; j < minLength; j++) {
					page.entries[j] = this.entries[minLength + j];
					this.entries[minLength + j] = null;
					page.adoption(j + 1, this.children[minLength + j + 1]);
					this.children[minLength + j + 1] = null;
				}
				this.children[index] = leftson;
				this.children[index + 1] = rightson;
				if (leftson != null) {
					leftson.parent = this;
				}
				if (rightson != null) {
					rightson.parent = this;
				}
			} else {
				/* ページの後半にentryを挿入する場合 */
				center = this.entries[minLength];
				for (int j = minLength + 1; j < index; j++) {
					page.entries[j - minLength - 1] = this.entries[j];
					this.entries[j] = null;
					page.adoption(j - minLength - 1, this.children[j]);
					this.children[j] = null;
				}
				page.entries[index - minLength - 1] = entry;
				page.children[index - minLength - 1] = leftson;
				page.children[index - minLength] = rightson;
				for (int j = index; j < this.entries.length; j++) {
					page.entries[j - minLength] = this.entries[j];
					this.entries[j] = null;
					page.adoption(j - minLength + 1, this.children[j + 1]);
					this.children[j + 1] = null;
				}
				this.children[index] = null;
				this.entries[minLength] = null;
				if (leftson != null) {
					leftson.parent = page;
				}
				if (rightson != null) {
					rightson.parent = page;
				}
			}
			page.size = minLength;
			this.size = minLength;
			this.insertParents(center, this, page);
		}
	}
	
	/**
	 * 親へエントリーを挿入します。
	 * @param entry 挿入するエントリー
	 * @param leftson エントリーの左の子ページ
	 * @param rightson エントリーの右の子ページ
	 */
	private void insertParents(E entry, Page<E> leftson, Page<E> rightson) {
		if (this.parent != null) {
			/* 親ページがある場合 */
			int index = this.parent.getIndex(entry);
			this.parent.insertDetail(index, entry, leftson, rightson);
		} else {
			/* 
			 * 親ページがない場合
			 * この時点では、最小エントリー数のエントリーを保持している。
			 */
			this.parent = new Page<E>(this.size, this.comparator);
			this.parent.entries[this.parent.size++] = entry;
			if (leftson != null) {
				this.parent.children[0] = leftson;
				leftson.parent = this.parent;
			}
			if (rightson != null) {
				this.parent.children[1] = rightson;
				rightson.parent = this.parent;
			}
		}
	}

	/**
	 * 右のページを親のエントリーを挟んで併合します。
	 * 弟のページから呼び出し、引数に兄のページを指定します。
	 * 親ページから削除するインデックスを指定します。
	 * @param index 親ページから削除するエントリーのインデックス
	 * @param page 右のページ（兄のページ）
	 */
	private void mergeLeft(int index, Page<E> page) {
		if (this.size + page.size < this.entries.length) {
			/* 併合しても最大エントリー数を以下に収まる場合 */
			E entry = this.parent.removeEntry(index, this);
			this.entries[this.size++] = entry;
			if (page.children[0] != null) {
				this.children[this.size] = page.children[0];
				this.children[this.size].parent = this;
			}
			for (int i = 0; i < page.size; i++) {
				this.entries[this.size++] = page.entries[i];
				if (page.children[i + 1] != null) {
					this.children[this.size] = page.children[i + 1];
					this.children[this.size].parent = this;
				}
			}
			/* pageを親ページの兄弟ページとする */
			if (this.parent.parent != null
					&& this.parent.size < this.entries.length >> 1) {
				Page<E> gparent = this.parent.parent;
				int gpindex = gparent.getIndex(this.parent.entries[0]);
				if (gpindex >= gparent.size) {
					/* ひとつしたの弟のページを取得 */
					page = gparent.children[--gpindex];
					page.mergeLeft(gpindex, this.parent);
				} else {
					/* ひとつうえの兄のページを取得 */
					page = gparent.children[gpindex + 1];
					this.parent.mergeLeft(gpindex, page);
				}
			} else if (this.parent.size == 0) {
				this.parent = null;
			}
		} else {
			int pindex = this.parent.getIndex(this.entries[0]);
			/* 併合すると最大エントリー以下に収まらない場合 */
			if (this.size < page.size) {
				/* 兄ページから弟ページへエントリーと子ページを移動させる */
				E parent = this.parent.replace(pindex, page.entries[0]);
				Page<E> shift = page.children[0];
				page.removeEntry(0, page.children[1]);
				this.entries[this.size++] = parent;
				this.children[this.size] = shift;
				if (shift != null) {
					shift.parent = this;
				}
			} else {
				/* 弟ページから兄ページへエントリーと子ページを移動させる */
				E parent = this.parent.replace(pindex, this.entries[this.size - 1]);
				Page<E> shift = this.children[this.size];
				this.removeEntry(this.size - 1, this.children[this.size - 1]);
				for (int i = page.size; i > 0; i--) {
					page.entries[i] = page.entries[i - 1];
					page.children[i + 1] = page.children[i];
				}
				page.children[1] = page.children[0];
				page.entries[0] = parent;
				page.children[0] = shift;
				page.size++;
				if (shift != null) {
					shift.parent = page;
				}
			}
		}
	}
	
	/**
	 * 指定したエントリーを削除します。
	 * @param entry 削除するエントリー
	 * @return 削除したエントリー
	 */
	E remove(E entry) {
		int index = this.getIndex(entry);
		if (index < 0) {
			/* 削除したいエントリーが見つかった場合 */
			index = - index - 1;
			return this.remove(index);
		} else if (this.children[index] != null) {
			/* 削除するエントリーがこのページでは見つからなかった場合
			 * 子ノードがあれば、再帰的に子ページに対しても削除を試みる。
			 */
			return this.children[index].remove(entry);
		}
		/* 葉ページの場合 */
		return null;
	}
	
	/**
	 * 指定したインデックスのエントリーを削除します。
	 * 指定されたインデックスには必ずエントリーが存在する必要があります。
	 * 
	 * @param index 削除するエントリーのインデックス
	 * @return 削除したエントリー
	 */
	private E remove(int index) {
		if (this.children[0] != null) {
			/* 葉ページ以外の場合 */
			E replace;
			Page<E> page = this.children[index].getMaxiumPage();
			if (page.size > this.entries.length >> 1) {
				/* 次にエントリーの次に小さい値を削除して取得します */
				replace = page.removeEntry(page.size - 1, null);
			} else {
				/* 次にエントリーの次に大きい値を削除して取得します */
				page = this.children[index + 1].getMinimumPage();
				replace = page.removeEntry(0, null);
			}

			/* 削除する値を取り出す */
			E result = this.entries[index];
			/* 置き換える */
			this.entries[index] = replace;
			if (page.size < this.entries.length >> 1) {
				index = page.parent.getIndex(page.entries[0]);
				if (index > 0) {
					/* 兄ページをこのページに併合 */
					Page<E> bros = page.parent.children[index - 1];
					bros.mergeLeft(index - 1, page);
				} else {
					/* 弟ページをこのページに併合 */
					Page<E> bros = page.parent.children[index + 1];
					page.mergeLeft(index, bros);
				}
			}
			return result;
		} else if (this.parent == null || this.size > this.entries.length >> 1) {
			/* ルートページであるか、葉ページでかつエントリー数に余裕がある場合 */
			return this.removeEntry(index, null);
		} else {
			/* 葉ページでかつエントリー数に余裕がない場合
			 * 親ページでのページ位置を確認して左右の子ページから融通します。
			 * 左右の子にも余裕がなければ、どちらか隣の子ページと併合します。
			 */
			int pindex = this.parent.getIndex(this.entries[0]);
			Page<E> bros = null;
			E result = null;
			boolean upper = true;
			if (pindex < this.parent.size) {
				/* ひとつうえの兄のページを取得 */
				bros = this.parent.children[pindex + 1];
				if (bros.size > this.entries.length >> 1) {
					/* ひとつ兄のページに余裕があった */
					result = bros.removeEntry(0, null);
					E replace = this.parent.entries[pindex];
					this.parent.entries[pindex] = result;
					result = this.replaceLeafUpper(index, replace);
				}
			}
			if (result == null && pindex > 0) {
				/* ひとつしたの弟のページを取得 */
				bros = this.parent.children[--pindex];
				if (bros.size > this.entries.length >> 1) {
					/* ひとつ弟のページに余裕があった */
					result = bros.removeEntry(bros.size - 1, null);
					E replace = this.parent.entries[pindex];
					this.parent.entries[pindex] = result;
					result = this.replaceLeafLower(index, replace);
				}
				upper = false;
			}
			if (result == null) {
				/*
				 * ひとつ隣の兄弟ページにも余裕がなかった場合、
				 * どちらかのページと併合します。
				 */
				result = this.removeEntry(index, null);
				if (upper) {
					/* 兄ページをこのページに併合 */
					this.mergeLeft(pindex, bros);
				} else {
					/* 弟ページにこのページを併合 */
					bros.mergeLeft(pindex, this);
				}
			}
			return result;
		}
	}
	
	/**
	 * ページからエントリーを削除して削除したエントリーの
	 * （削除しているのでひとつの）子ページを指定したページに更新します。
	 * 削除したエントリーを返します。
	 * @param index 削除するエントリーのインデックス
	 * @param child 更新する子ページ
	 * @return 削除したエントリー
	 */
	public E removeEntry(int index, Page<E> child) {
		E result = this.entries[index];
		this.children[index] = child;
		for (int i = index + 1; i < this.size; i++) {
			this.entries[i - 1] = this.entries[i];
			this.children[i] = this.children[i + 1];
		}
		this.children[this.size] = null;
		this.entries[--this.size] = null;
		return result;
	}

	/**
	 * 指定したインデックスのエントリーを削除して、
	 * 指定したインデックスに新たなエントリーを挿入します。
	 * @param index 削除するエントリーインデックス
	 * @param entry 挿入するエントリー
	 * @return 削除したエントリー
	 */
	private E replace(int index, E entry) {
		E result = this.entries[index];
		this.entries[index] = entry;
		return result;
	}

	/**
	 * 指定したインデックスのエントリーを削除して、
	 * 葉ページに対して最小のエントリーを挿入します。
	 * @param index 削除するエントリーインデックス
	 * @param entry 挿入するエントリー
	 * @return 削除したエントリー
	 */
	private E replaceLeafLower(int index, E entry) {
		E result = this.entries[index];
		for (int i = index; i > 0; i--) {
			this.entries[i] = this.entries[i - 1];
		}
		this.entries[0] = entry;
		return result;
	}

	/**
	 * 指定したインデックスのエントリーを削除して、
	 * 葉ページに対して最大のエントリーを挿入します。
	 * @param index 削除するエントリーインデックス
	 * @param entry 挿入するエントリー
	 * @return 削除したエントリー
	 */
	private E replaceLeafUpper(int index, E entry) {
		E result = this.entries[index];
		for (int i = index; i < this.size; i++) {
			this.entries[i] = this.entries[i + 1];
		}
		this.entries[this.size - 1] = entry;
		return result;
	}
	
	@Override
	public String toString() {
		return Arrays.toString(this.entries);
	}
}