package gui;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.util.ArrayList;
import java.util.List;
import java.util.Observable;

import javax.swing.JComponent;
import model.Edge;
import model.Node;

/**
 * グラフのデモを表示するパネルです。
 * @author ma38su 
 */
public class DemoPanel extends JComponent {
	private Image offs;
	
	private Observable observable;
	private List<Node> nodes;

	/**
	 * フレームとパネルのマージン
	 */
	private final int MARGIN = 5;
	
	private Rectangle screen;

	/**
	 * アニメーションの間隔
	 */
	private int interval = 0;

	private List<Node> route;

	/**
	 * 辺の接続状況
	 */
	private boolean[][] edges;

	public DemoPanel(Observable observable) {
		this.observable = observable;
		this.nodes = new ArrayList<Node>();
		this.route = new ArrayList<Node>();
		this.edges = null;
	}

	/**
	 * 頂点を追加します。
	 * @param x
	 * @param y
	 */
	public void add(int x, int y) {
		if (this.screen.contains(x, y)) {
			this.nodes.add(new Node(x, y));
			this.observable.notifyObservers(this.nodes.size());
			this.repaint();
		}
	}

	/**
	 * パネルを初期化します。
	 */
	public void clear() {
		this.nodes.clear();
		this.route.clear();
		this.edges = null;
		this.observable.notifyObservers(this.nodes.size());
		this.observable.notifyObservers(this.route);
		this.repaint();
	}

	/**
	 * 巡回賂を消去します。
	 */
	public void clearRoute() {
		for (Node node : this.nodes) {
			node.clear();
		}
		this.repaint();
	}

	/**
	 * 頂点を返します。
	 * @return 頂点のリスト
	 */
	public List<Node> getNodes() {
		return this.nodes;
	}

	@Override
	protected void paintComponent(Graphics g) {
		if (this.offs == null || this.getWidth() != this.offs.getWidth(null) || this.getHeight() != this.offs.getHeight(null)) {
			this.offs = this.createImage(this.getWidth(), this.getHeight());
			this.screen = new Rectangle(this.MARGIN, this.MARGIN, this.getWidth() - this.MARGIN * 2, this.getHeight() - this.MARGIN * 2);
		}
		Graphics2D g2 = (Graphics2D) this.offs.getGraphics();
		g2.setColor(Color.WHITE);
		g2.fill(this.screen);
		g2.setClip(this.screen);
		
		g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);

		g2.setColor(Color.BLACK);
		synchronized (this.route) {
			if (!this.route.isEmpty()) {
				Node n0 = this.route.get(this.route.size() - 1);
				for (Node node : this.route) {
					Edge entry = new Edge(n0, node);
					entry.draw(g2);
					n0 = node;
				}
			}
		}

		g2.setColor(Color.DARK_GRAY);
		try {
			if (this.edges != null) {
				for (int i = 0; i < this.edges.length; i++) {
					// this.nodes.size()だと頂点が増えたときに例外が起こる可能性がある。
					for (int j = 0; j < this.edges.length; j++) {
						if (this.edges[i][j]) {
							Edge edge = new Edge(this.nodes.get(i), this.nodes.get(j));
							edge.draw(g2);
						}
					}
				}
			}
			for (int i = 0; i < this.nodes.size(); i++) {
				int connection = 0;
				Node node = this.nodes.get(i);
				if (this.edges != null) {
					for (int j = 0; j < this.edges.length; j++) {
						if (i < this.edges.length && (this.edges[i][j] || this.edges[j][i])) {
							connection++;
						}
					}
				}
				if (connection == 0) {
					node.draw(g2, Color.GRAY);
				} else if (connection % 2 == 0) {
					node.draw(g2, Color.YELLOW);
				} else {
					node.draw(g2, Color.RED);
				}
			}
			g2.setClip(0, 0, this.getWidth(), this.getHeight());
			g2.setColor(Color.BLACK);
			g2.draw(this.screen);
			g.drawImage(this.offs, 0, 0, null);
		} catch (Exception e) {
			System.out.println(e.getClass().getName());
			this.repaint();
		}
	}

	/**
	 * 辺の関係を設定します。
	 * @param edges 辺の接続関係を示す二重配列
	 */
	public void set(boolean[][] edges) {
		synchronized (this.route) {
			this.route.clear();
		}
		this.observable.notifyObservers(this.route);
		this.edges = edges;
		this.repaint();
		synchronized (this) {
			if (this.interval > 0) {
				try {
					this.wait(this.interval);
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
		}
	}

	/**
	 * 巡回賂を設定します。
	 * @param route 巡回賂を示す頂点のリスト
	 */
	public void set(List<Node> route) {
		this.edges = null;
		synchronized (this.route) {
			this.route.clear();
			synchronized (route) {
				this.route.addAll(route);
			}
		}
		this.repaint();
		synchronized (this) {
			if (this.interval > 0) {
				try {
					this.wait(this.interval);
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
		}
	}
	
	/**
	 * アニメーションの間隔を設定します。
	 * @param ms アニメーションの間隔（ms）
	 */
	public void setInterval(int ms) {
		synchronized (this) {
			this.interval = ms;
		}
	}
}
