
package method.tsp;

import gui.DemoPanel;
import java.util.Comparator;
import method.GraphDemonstration;
import model.Node;
import util.HeapSet;

/**
 * Held and Karpの手法による巡回セールスマン問題の厳密解法
 * ラグランジュ緩和、ラグランジュ乗数は適当
 * @author Masayasu Fujiwara
 */
public class HeldKarp implements GraphDemonstration {
	/**
	 * 一時的に辺を表現するためのクラス
	 * @author masayasu
	 */
	class Edge {
		int s;
		int t;
		public Edge(int s, int t) {
			this.s = s;
			this.t = t;
		}
		@Override
		public boolean equals(Object obj) {
			if (obj instanceof Edge) {
				Edge edge = (Edge) obj;
				return edge.s == this.s && edge.t == this.t;
			}
			return false;
		}
		@Override
		public int hashCode() {
			return this.t;
		}
	}

	/**
	 * 1つの頂点に2つの辺がつながっているかどうか確認する。
	 * 1-treeがこれを満たせば、巡回路といえる。
	 * @param edges 辺の接続
	 * @param adjustment ラグランジュ緩和による辺の補正
	 * @return 巡回路であればtrue、巡回路でなければfalseを返す。
	 */
	private boolean checkCircuit(double dcost, boolean[][] edges, double[] adjustment) {
		boolean ret = true;
		for (int i = 0; i < edges.length; i++) {
			int connection = 0;
			for (int j = 0; j < edges[i].length; j++) {
				if (edges[i][j]) {
					connection++;
				}
			}
			if (connection < 2) {
				adjustment[i] -= dcost;
//				return false;
				ret = false;
			} else if (connection > 2) {
				adjustment[i] += dcost;
//				return false;
				ret = false;
			}
		}
		return ret;
	}

	/**
	 * ネットワークのコストを調べる。
	 * 1-treeが巡回路でない場合、巡回路の下界となる。
	 * @param table 距離テーブル
	 * @param edges 辺の接続関係
	 * @return
	 */
	private double getCost(double[][] table, boolean[][] edges) {
		double cost = 0;
		for (int i = 0; i < table.length; i++) {
			for (int j = 0; j < table.length; j++) {
				if (edges[i][j]) {
					cost += table[i][j];
				}
			}
		}
		return cost;
	}

	/**
	 * 1-treeを求める。
	 * @param panel パネル
	 * @param nodes 頂点配列
	 * @param table 距離テーブル
	 * @param adjustment ラグランジュ緩和による距離の補正
	 * @return 辺の接続関係
	 */
	private boolean[][] getOneTree(DemoPanel panel, final Node[] nodes, final double[][] table, final double[] adjustment) {
		boolean[][] edges = new boolean[nodes.length][nodes.length];
		if (nodes.length > 1) {
			Edge e;
			boolean[] close = new boolean[nodes.length];
			HeapSet<Edge> open = new HeapSet<Edge>(11, new Comparator<Edge>() {
				public int compare(Edge o1, Edge o2) {
					double diff = (table[o1.s][o1.t] + adjustment[o1.s] + adjustment[o1.t]) - (table[o2.s][o2.t] + adjustment[o2.s] + adjustment[o2.t]);
					if (diff > 0) {
						return 1;
					} else if (diff < 0) {
						return -1;
					} else {
						return 0;
					}
				}
			});

//			int s = 0;
			int s = (int) (Math.random() * nodes.length);
			close[s] = true;
			int index = (s + 1) % nodes.length;
			close[index] = true;
			do {
				for (int i = 0; i < nodes.length; i++) {
					if (i != index) {
						if (!close[i]) {
							open.add(new Edge(index, i));
						}
					}
				}
				e = open.poll();
				if (e == null) {
					break;
				}
				if (!close[e.t]) {
					edges[e.s][e.t] = true;
					edges[e.t][e.s] = true;
					index = e.t;
					close[index] = true;
					panel.set(edges);
				}
			} while (open.size() > 0);
			open.clear();
			for (int i = 0; i < nodes.length; i++) {
				if (s != i) {
					open.add(new Edge(s, i));
				}
			}
			e = open.poll();
			edges[e.s][e.t] = true;
			edges[e.t][e.s] = true;
			e = open.poll();
			edges[e.s][e.t] = true;
			edges[e.t][e.s] = true;
		}
		return edges;
	}

	public void method(DemoPanel panel) {
		final Node[] nodes = panel.getNodes().toArray(new Node[]{});
		final double[][] table = new double[nodes.length][nodes.length];
		for (int i = 0; i < nodes.length; i++) {
			for (int j = 0; j < nodes.length; j++) {
				table[i][j] = nodes[i].getDistance(nodes[j]);
			}
		}
		
		boolean[][] edges;
		double[] adjustment = new double[nodes.length];
		double value = Double.POSITIVE_INFINITY;
		int count = 0;
		double adjust = 100;
		do {
			edges = this.getOneTree(panel, nodes, table, adjustment);
			panel.set(edges);
			double cost = this.getCost(table, edges);
			if (value > cost) {
				value = cost;
			}
			adjust *= 0.9;
		} while (!this.checkCircuit(adjust, edges, adjustment) && count++ < 10000);
	}

	@Override
	public String toString() {
		return "Held and Karp";
	}
}