/* -*- mode:objc; coding:utf-8; tab-width:8; c-basic-offset:2; indent-tabs-mode:nil -*- */
/*
  Copyright (c) 2004 MacUIM Project
  http://www.digital-genes.com/~yatsu/macuim/

  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
  3. Neither the name of authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
  SUCH DAMAGE.
*/

#import <sys/types.h>
#import <sys/socket.h>
#import <iconv.h>
#import "MacUIMHelperExtra.h"
#import "Debug.h"


static MacUIMHelperExtra *sharedExtra;


static void
uimDisconnect();

static NSString *
convertHelperString(char *str);


@implementation MacUIMHelperExtra

+ (MacUIMHelperExtra *)sharedExtra
{
  return sharedExtra;
}

- initWithBundle:(NSBundle *)bundle
{
  self = [super initWithBundle:bundle];
  if (!self)
    return nil;

  sharedExtra = self;

  menu = [[NSMenu alloc] initWithTitle:@""];
  //[menu setAutoenablesItems:NO];

  view = [[MacUIMHelperView alloc] initWithFrame:[[self view] frame]
                                   menuExtra:self];
  [self setView:view];

  uim_helper_client_get_prop_list();

#ifdef NEW_HELPER
  if (uim_helper_init_client(uimDisconnect) == 0)
#else
  uimFD = uim_helper_init_client_fd(uimDisconnect);
  if (uimFD >= 0)
#endif
  {
    NSNotificationCenter *noc;

    uimHandle = [[NSFileHandle alloc]
#ifdef NEW_HELPER
                  initWithFileDescriptor:uim_helper_get_client_fd()
#else
                  initWithFileDescriptor:uimFD
#endif
                 ];

    [uimHandle waitForDataInBackgroundAndNotify];

    noc = [NSNotificationCenter defaultCenter];
    [noc addObserver:self
         selector:@selector(helperRead:)
         name:@"NSFileHandleDataAvailableNotification"
         object:uimHandle];

#ifdef NEW_HELPER
    is_helper_connect = YES;
#endif
  }

  modes = [[NSMutableArray alloc] init];
  propNames = [[NSMutableArray alloc] init];
  menuItems = [[NSMutableArray alloc] init];

  labels = [[NSMutableArray alloc] init];

  clicked = NO;

  [self updateMenu];

  return self;
}

- (void)willUnload
{
  [super willUnload];
}

- (void)dealloc
{
  [view release];
  [menu release];

  [self helperClose];

  [uimHandle release];

  [modes release];
  [propNames release];
  [menuItems release];

  [labels release];

  [super dealloc];
}

- (NSImage *)image
{
  return [self createImage:NO];
}

- (NSImage *)alternateImage
{
  return [self createImage:YES];
}

- (NSImage *)createImage:(BOOL)alter
{
  NSImage *image = nil;

  clicked = alter;

  if ([labels count] <= 1) {
    [view setFrameSize:NSMakeSize(kMenuBarWidth, [view frame].size.height)];
    [self setLength:kMenuBarWidth];
    image = [[[NSImage alloc] initWithSize:NSMakeSize(kMenuBarWidth, kMenuBarHeight)]
              autorelease];
  }
  else {
    [view setFrameSize:NSMakeSize(kMenuBarWidth2, [view frame].size.height)];
    [self setLength:kMenuBarWidth2];
    image = [[[NSImage alloc] initWithSize:NSMakeSize(kMenuBarWidth2, kMenuBarHeight)]
              autorelease];
  }

  [self renderFrame:image];
  [self renderText:image];

  return image;
}

- (void)renderFrame:(NSImage *)image
{
  NSBezierPath *framePath;
  NSColor *color;

  if (clicked == NO)
    color = [NSColor blackColor];
  else
    color = [NSColor whiteColor];

  [image lockFocus];

  if ([labels count] <= 1) {
    framePath =
      [NSBezierPath bezierPathWithRect:NSMakeRect(0.5, 3.5,
                                                  kMenuBarWidth - 1.0, kMenuBarHeight - 7.0)];
    if (clicked == NO) {
      [[NSColor whiteColor] set];
      [framePath fill];
    }
    [color set];
    [framePath stroke];
  }
  else {
    framePath =
      [NSBezierPath bezierPathWithRect:NSMakeRect(0.5, 3.5,
                                                  kMenuBarWidth2 - 1.0, kMenuBarHeight - 7.0)];
    if (clicked == NO) {
      [[NSColor whiteColor] set];
      [framePath fill];
    }
    [color set];
    [framePath stroke];

    framePath =
      [NSBezierPath bezierPathWithRect:NSMakeRect(kMenuBarWidth2 / 2.0, 3.5,
                                                  0.1, kMenuBarHeight - 7.0)];
    if (clicked == NO) {
      [[NSColor whiteColor] set];
      [framePath fill];
    }
    [color set];
    [framePath stroke];
  }

  [image unlockFocus];
}

- (void)renderText:(NSImage *)image
{
  NSMutableAttributedString *text;
  NSMutableString *label = nil;
  NSColor *color;
  int i;

  if (clicked == NO)
    color = [NSColor blackColor];
  else
    color = [NSColor whiteColor];

  [image lockFocus];

  [[NSGraphicsContext currentContext] setShouldAntialias:YES];

  if ([labels count] < 1)
    label = [[NSMutableString alloc] initWithString:@"?"];

  i = 0;
  do {
    if (!label)
      label = [[NSMutableString alloc] initWithString:[labels objectAtIndex:i]];

    //[label appendString:@" "];

    if ([labels count] <= 1) {
      // aString = [[[NSAttributedString alloc] initWithString:@"string" attributes:[self boldFontAttributes]] autorelease];
      // NSDictionary dictionaryWithObject: [NSFont boldSystemFontOfSize:[NSFont systemFontSize]]


      text = [[NSAttributedString alloc] initWithString:label
                                         attributes:[NSDictionary dictionaryWithObjectsAndKeys:
                                                                    [NSFont boldSystemFontOfSize:12],
                                                                  NSFontAttributeName,
                                                                  color,
                                                                  NSForegroundColorAttributeName,
                                                                  nil]];
      [text drawAtPoint:NSMakePoint(0.5 + 0.5 +
                                    (kMenuBarWidth - 1.0 -
                                     ceil([text size].width + 0.5)) / 2.0,
                                    (kMenuBarHeight - ceil([text size].height)) / 2.0)];
    }
    else {
      text = [[NSAttributedString alloc] initWithString:label
                                         attributes:[NSDictionary dictionaryWithObjectsAndKeys:
                                                                    [NSFont boldSystemFontOfSize:11],
                                                                  NSFontAttributeName,
                                                                  color,
                                                                  NSForegroundColorAttributeName,
                                                                  nil]];
      [text drawAtPoint:NSMakePoint(0.5 + 0.5 +
                                    (i > 0 ? (kMenuBarWidth2 - 1.0) / 2.0 : 0)
                                    + (kMenuBarWidth2 / 2.0 - 0.5 -
                                     ceil([text size].width + 0.5)) / 2.0,
                                    (kMenuBarHeight - ceil([text size].height)) / 2.0)];
    }

    [text release];
    [label release];
    label = nil;

  } while (++i < [labels count]);

  [image unlockFocus];
}

- (NSMenu *)menu
{
  //[self updateMenu];

  return menu;
}

- (void)updateMenu
{
  int i;

  if ([menu numberOfItems] > 0) {
    while ([menu numberOfItems])
      [menu removeItemAtIndex:0];
  }

  [menuItems removeAllObjects];

  for (i = 0; i < [modes count]; i++) {
    NSString *mode;
    NSMenuItem *menuItem;
    mode = [modes objectAtIndex:i];
    menuItem = (NSMenuItem *) [[menu addItemWithTitle:mode
                                     action:@selector(modeSelect:) keyEquivalent:@""]
                                setTarget:self];
    [menuItems addObject:menuItem];
  }

  [menu addItem:[NSMenuItem separatorItem]];

  [[menu addItemWithTitle:@"Preferences..."
         action:@selector(openSystemPrefs:) keyEquivalent:@""]
    setTarget:self];
}

- (void)modeSelect:(id)sender
{
  int i;
  for (i = 0; i < [menuItems count]; i++) {
    if (sender == [menuItems objectAtIndex:i]) {
      NSMutableString *msg = [[NSMutableString alloc] initWithString:@"prop_activate\n"];
      [msg appendString:[NSString stringWithString:[propNames objectAtIndex:i]]];
      [msg appendString:@"\n"];
      /*
      printf("MacUIMHelperExtra::menuSelect: msg='%s'\n",
             [msg UTF8String]);
      */
#ifdef NEW_HELPER
      uim_helper_send_message([msg UTF8String]);
#else
      uim_helper_send_message(uimFD, [msg UTF8String]);
#endif
      [msg release];
      break;
    }
  }
}

- (void)openSystemPrefs:(id)sender
{
  [NSTask launchedTaskWithLaunchPath:@"/usr/bin/open"
          arguments:[NSArray arrayWithObject:@"/Library/PreferencePanes/MacUIM.prefPane"]];
}

- (void)helperRead:(NSNotification *)notification
{
  char *tmp;

#ifdef NEW_HELPER
  uim_helper_read_proc(uim_helper_get_client_fd());
#else
  uim_helper_read_proc(uimFD);
#endif
  while ((tmp = uim_helper_get_message())) {
#if DEBUG_HELPER_EXTRA
    fprintf(stderr, "MacUIMHelperExtra::helperRead() tmp='%s'\n", tmp);
#endif
    [self helperParse:tmp];
    free(tmp);
  }

#ifdef NEW_HELPER
  if (is_helper_connect)
#else
  if (uimFD >= 0)
#endif
    [uimHandle waitForDataInBackgroundAndNotify];
}

- (void)helperParse:(char *)str
{
  NSString *nsstr;
  NSArray *array;

  if (!str || strlen(str) == 0)
    return;

  //printf("MacUIMHelperExtra::helperParse: str='%s'\n", str);

  nsstr = convertHelperString(str);

  if (nsstr) {
    array = [nsstr componentsSeparatedByString:@"\n"];
    if (array && [array count] > 0) {
      NSString *first = [array objectAtIndex:0];
      if (first) {
        if ([first compare:@"prop_list_update"
                   options:NSCaseInsensitiveSearch
                   range:NSMakeRange(0, strlen("prop_list_update"))]
            == NSOrderedSame) {
          [self propListUpdate:array];
        }
        else if ([first compare:@"prop_label_update"
                        options:NSCaseInsensitiveSearch
                        range:NSMakeRange(0, strlen("prop_label_update"))]
                 == NSOrderedSame) {
          [self propLabelUpdate:array];
        }
        /*
        else if ([first compare:@"focus_in"
                        options:NSCaseInsensitiveSearch
                        range:NSMakeRange(0, strlen("focus_in"))]
                 == NSOrderedSame) {
          [view setNeedsDisplay:YES];
        }
        else if ([first compare:@"focus_out"
                        options:NSCaseInsensitiveSearch
                        range:NSMakeRange(0, strlen("focus_out"))]
                 == NSOrderedSame) {
          [labels removeAllObjects];
        }
        */
#if DEBUG_HELPER_EXTRA
        else {
          fprintf(stderr, "MacUIMHelperExtra::helperParse() unknown string '%s'\n",
                  [first cString]);
        }
#endif
      }
    }
    [nsstr release];
  }
}

- (void)helperClose
{
#ifdef NEW_HELPER
  if (is_helper_connect)
#else
  if (uimFD >= 0)
#endif
#ifdef NEW_HELPER
    uim_helper_close_client();
#else
    uim_helper_close_client_fd(uimFD);
#endif
}

- (void)helperDisconnect
{
  [[NSNotificationCenter defaultCenter] removeObserver:self];

#ifdef NEW_HELPER
  is_helper_connect = NO;
#else
  uimFD = -1;
#endif
}

- (void)propListUpdate:(NSArray *)lines
{
  int i;
  NSString *line;
  NSArray *cols;
  NSString *col;

  if (!lines || [lines count] < 2)
    return;

  line = [lines objectAtIndex:1];

  [labels removeAllObjects];
  [modes removeAllObjects];
  [propNames removeAllObjects];

  for (i = 2; i < [lines count]; i++) {
    line = [lines objectAtIndex:i];
    if (!line || [line compare:@""] == NSOrderedSame)
      break;

    cols = [line componentsSeparatedByString:@"\t"];

    if (cols && [cols count] >= 3) {
      col = [cols objectAtIndex:0];
      if ([col compare:@"branch"] == NSOrderedSame) {
        NSMutableString *branch =
          [[NSMutableString alloc] initWithString:[cols objectAtIndex:1]];
        /*
        printf("MacUIMHelperExtra::propListUpdate: branch='%s'\n",
               [branch UTF8String]);
        */
        [labels addObject:branch];
      }
      else if ([col compare:@"leaf"] == NSOrderedSame) {
        NSMutableString *mode =
          [[NSMutableString alloc] initWithString:[cols objectAtIndex:2]];
        NSMutableString *prop =
          [[NSMutableString alloc] initWithString:[cols objectAtIndex:4]];
        /*
        printf("MacUIMHelperExtra::propListUpdate: leaf='%s'\n",
               [mode UTF8String]);
        */
        [modes addObject:mode];
        [propNames addObject:prop];
      }
    }
  }

  [self updateMenu];

  [view setNeedsDisplay:YES];

  /*
  for (i = 0; i < [lines count]; i++)
    printf("propListUpdate: %d '%s'\n",
           i, [[lines objectAtIndex:i] UTF8String]);
  */
}

- (void)propLabelUpdate:(NSArray *)lines
{
  int i;
  NSString *line;
  NSArray *cols;
  NSString *charset = nil;

  if (!lines || [lines count] < 2)
    return;

  line = [lines objectAtIndex:1];

  cols = [line componentsSeparatedByString:@"="];
  if (!cols || [cols count] < 2)
    return;

  charset = [cols objectAtIndex:1];

  //uim_helper_client_get_prop_list();

  [labels removeAllObjects];

  for (i = 2; i < [lines count]; i++) {
    line = [lines objectAtIndex:i];
    if (!line || [line compare:@""] == NSOrderedSame)
      break;

    cols = [line componentsSeparatedByString:@"\t"];
    if (cols && [cols count] >= 2) {
      NSMutableString *label = [[NSMutableString alloc]
                                 initWithString:[cols objectAtIndex:0]];
#if DEBUG_HELPER_EXTRA
      fprintf(stderr, "propLabelUpdate: label='%s'\n", [label UTF8String]);
#endif
      [labels addObject:label];
    }
  }

  [view setNeedsDisplay:YES];
}

@end

static void
uimDisconnect()
{
  NSAutoreleasePool *localPool;

  localPool = [[NSAutoreleasePool alloc] init];        
  [[MacUIMHelperExtra sharedExtra] helperDisconnect];
  [localPool release];
}


static NSString *
convertHelperString(char *str)
{
  char *line, *tmp;
  char *charset = NULL;
  NSString *convstr;

  line = strdup(str);
  if (tmp = strstr(line, "charset=")) {
    tmp += 8;
    charset = strtok(tmp, "\n");
  }
  
  if (charset && (strncmp(charset, "UTF-8", 5) != 0)) {
    CFStringRef name;
    CFStringEncoding cfencoding;
    UInt32 nsencoding;
    NSData *data;

    name = CFStringCreateWithCString(NULL, charset, kCFStringEncodingMacRoman);
    cfencoding = CFStringConvertIANACharSetNameToEncoding(name);
    CFRelease(name);
    nsencoding = CFStringConvertEncodingToNSStringEncoding(cfencoding);

    data = [NSData dataWithBytes:str length:strlen(str)];
    convstr = [[NSString alloc] initWithData:data encoding:nsencoding];
  } else {
    convstr = [[NSString alloc] initWithUTF8String:str];
  }
  free(line);

  return convstr;
}
