/* -*- mode:objc; coding:utf-8; tab-width:8; c-basic-offset:2; indent-tabs-mode:nil -*- */
/*
  Copyright (c) 2004 MacUIM Project
  http://www.digital-genes.com/~yatsu/macuim/

  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
  3. Neither the name of authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
  SUCH DAMAGE.
*/

#import "CocoaWinController.h"

static CocoaWinController *sharedController;

@implementation CocoaWinController
 
+ (CocoaWinController *)sharedController
{
  return sharedController;
}

/**
 * Initialize Cocoa
 */
- (id)init
{
  self = [super init];
  NSApplicationLoad();
  if (![NSBundle loadNibNamed:@"CocoaWindow" owner:self]) {
    NSLog(@"failed to load CocoaWindow nib");
  }
  sharedController = self;

  origSize = [vPanel frame].size;
  candIndex = -1;

#if DEBUG_CANDIDATE_WINDOW
  printf("origsize: width=%f height=%f\n",
         origSize.width, origSize.height);
#endif

  [vPanel setFrame:NSMakeRect([vPanel frame].origin.x,
                              [vPanel frame].origin.y,
                              origSize.width, 37)
          display:NO];

  realModeTipsPanel = [[ModeTipsPanel alloc]
                        initWithContentRect:[[modeTipsPanel contentView] frame]
                        styleMask:NSBorderlessWindowMask
                        backing:[modeTipsPanel backingType]
                        defer:NO];

  [realModeTipsPanel initView];

  [realModeTipsPanel setBackgroundColor:[NSColor whiteColor]];
  [realModeTipsPanel setHasShadow:YES];
  [realModeTipsPanel setBecomesKeyOnlyIfNeeded:NO];
  //[realModeTipsPanel setdelegate:self];

#if 0 // XXX
  {
    NSView *content = [[modeTipsPanel contentView] retain];
    [content removeFromSuperview];
    //[content setFrame:NSMakeSize(30, 30)];
    [realModeTipsPanel setContentView:content];
    [content release];
  }
#endif

  //[realModeTipsPanel setContentSize:NSMakeSize(30, 30)];

  modeTipsTimer = nil;
  lastLabel = nil;

  return self;
}

/**
 * Set a callback for a CocoaWinController
 */
- (void)setCallBack:(CallBackType)callBack
{
  _callBack = callBack;
}

/**
 * Show a CocoaWinController and make it activate
 */
- (void)showWindow:(int)qdX:(int)qdY:(int)height
{
  lineHeight = height;

  [vPanel setAutodisplay:NO];
  [self replaceWindow:qdX:qdY];

  if (candIndex >= 0) {
    NSIndexSet *indexSet =
      [[NSIndexSet alloc] initWithIndex:candIndex];
    [vTable selectRowIndexes:indexSet byExtendingSelection:nil];
    [indexSet release];
  }

  if ([vPanel isVisible] == NO) {
    [vPanel makeFirstResponder:vTable];
    [vPanel orderFront:nil];
    [vPanel setLevel:NSFloatingWindowLevel];
    //[vPanel setAlphaValue:0.8];
  }

  [vPanel setAutodisplay:YES];

}

/**
 * Hide a candidates-window
 */
- (void)hideWindow
{
  NSPoint origin = [vPanel frame].origin;

  if ([vPanel isVisible] == NO)
    return;

#if DEBUG_CANDIDATE_WINDOW
  printf("CocoaWinController::hideWindow\n");
#endif

  [vPanel orderOut:nil];

  [[vTable tableColumnWithIdentifier:@"candidate"] setWidth:36];
  [vPanel setFrame:NSMakeRect(origin.x, origin.y,
                              origSize.width, 37)
          display:NO];
  
  //[realModeTipsPanel orderOut:nil];
}

/**
 * Returns YES if the candidate window is visible
 */
- (BOOL)isVisible
{
  return [vPanel isVisible];
}

/**
 * Request the NSTableView to reload candidates
 */
- (void)reloadData
{
  [vTable reloadData];
}

/**
 * Initialize
 */
- (void)awakeFromNib
{
  headArray = [[NSMutableArray alloc] init];
  candArray = [[NSMutableArray alloc] init];
}

/**
 * Get a number of rows in the TableView
 */
- (int)numberOfRowsInTableView:(NSTableView *)aTableView
{
  return [candArray count];
}

/**
 * Get data
 */
- (id)tableView:(NSTableView *)tableView
  objectValueForTableColumn:(NSTableColumn *)tableColumn
            row:(int)rowIndex
{
  id colID = [tableColumn identifier];

  if ([colID isEqual:@"head"])
    return [headArray objectAtIndex:rowIndex];
  else if ([colID isEqual:@"candidate"])
    return [candArray objectAtIndex:rowIndex];

  return nil;
}

- (void)tableView:(NSTableView *)tableView
  willDisplayCell:(id)cell
   forTableColumn:(NSTableColumn *)tableColumn
              row:(int)row
{
  [cell setDrawsBackground:YES];
  if (row % 2)
    [cell setBackgroundColor:[NSColor colorWithCalibratedWhite:0.95
                                      alpha:1.0]];
  else
    [cell setBackgroundColor:[NSColor whiteColor]];
}

- (UniCharPtr)getCandidate:(int)index
{
  return nil;
}

- (void)addCandidate:(UniCharPtr)head:(int)headLen
                    :(UniCharPtr)cand:(int)candLen
{
  NSString *headStr;
  NSString *candStr;

  if (head && headLen > 0)
    headStr = [[NSString alloc] initWithCharacters:head + headLen - 1
                                length:1];
  else
    headStr = [[NSString alloc] initWithString:@""];;

  if (cand && candLen > 0)
    candStr = [[NSString alloc] initWithCharacters:cand
                                length:candLen];
  else
    candStr = [[NSString alloc] initWithString:@""];;

  [headArray addObject:headStr];
  [candArray addObject:candStr];
}

/**
 * Clear candidates
 */
- (void)clearCandidate
{
#if DEBUG_CANDIDATE_WINDOW
  printf("CocoaWinController::clearCandidate\n");
#endif

  [headArray removeAllObjects];
  [candArray removeAllObjects];
}

/**
 * Select a candidate
 */
- (void)selectCandidate:(int)index
{
  NSIndexSet *indexSet;

  indexSet = [[NSIndexSet alloc] initWithIndex:index];

  candIndex = index;
  
  [vTable selectRowIndexes:indexSet byExtendingSelection:nil];
  [vTable scrollRowToVisible:index];

  [indexSet release];
}

/**
 * deselect a candidate
 */
- (void)deselectCandidate
{
  [vTable deselectAll:nil];

  candIndex = -1;
}

/**
 * Set a page label
 */
- (void)setPage:(int)index:(int)max
{
  NSString *label;

  if (index > 0)
    label = [NSString stringWithFormat:@"%d / %d",
                      index, max];
  else
    label = [NSString stringWithFormat:@"- / %d", max];

  [hLabel setStringValue:label];
}

/**
 * Button press action
 */
- (IBAction)candClicked:(id)sender
{
#if DEBUG_CANDIDATE_WINDOW
  printf("CocoaWinController::candClicked()\n");
#endif

  [vPanel orderOut:nil];

  (*_callBack)([sender clickedRow]);
}

- (void)replaceWindow:(int)replyX:(int)replyY
{
  NSTableColumn *col = [vTable tableColumnWithIdentifier:@"candidate"];
  NSSize mainSize = [[NSScreen mainScreen] frame].size;
  NSRect rect = [vPanel frame];
  float columnWidth = 0.0;
  int x, y, i;

  for (i = 0; i < [candArray count]; i++) {
    NSMutableAttributedString *text =
      [[NSAttributedString alloc] initWithString:[candArray objectAtIndex:i]
                                  attributes:[NSDictionary dictionaryWithObjectsAndKeys:
                                                             [[col dataCell] font],
                                                           NSFontAttributeName,
                                                           nil]];
    if (columnWidth < [text size].width)
      columnWidth = [text size].width;
    [text release];
  }

  if (columnWidth > 700)
    columnWidth = 700;
  if (columnWidth + 4 > [col width])
    [col setWidth:columnWidth + 4];

  i = 0;
  while (1) {
    NSString *str;
    if (i == [candArray count])
      break;
    str = [candArray objectAtIndex:i];
    if (!str || [str length] == 0)
      break;
    i++;
  }


  if ([[vPanel contentView] frame].size.height < 17 * i + 20) {
    [vPanel setContentSize:NSMakeSize([vTable frame].size.width + 2,
                                      17 * i + 20)];
  }
  else if ([[vPanel contentView] frame].size.width <
          [vTable frame].size.width + 2) {
    [vPanel setContentSize:NSMakeSize([vTable frame].size.width + 2,
                                      [[vPanel contentView] frame].size.height)];
  }

#if DEBUG_CANDIDATE_WINDOW
  [vPanel setContentSize:NSMakeSize([vTable frame].size.width + 2,
                                    17 * i + 20)];
#endif

  [vTable setFrameSize:NSMakeSize([vTable frame].size.width,
                                 17 * i)];

  rect = [vPanel frame];

  x = replyX;
  y = mainSize.height - replyY - rect.size.height;

  if (x < 0)
    x = 0;
  if (x + rect.size.width > mainSize.width)
    x = mainSize.width - rect.size.width;
  if (mainSize.height - replyY - origSize.height < 0)
    y = mainSize.height - replyY + lineHeight + 3;
  if (y + rect.size.height > mainSize.height)
    y = mainSize.height - rect.size.height;

#if DEBUG_CANDIDATE_WINDOW
  printf("CocoaWinController::replaceWindow: x=%d y=%d origin.x=%d origin.y=%d\n",
         x, y, (int) rect.size.width, (int) rect.size.height);
#endif

  if (x != (int) rect.origin.x || y != (int) rect.origin.y)
    [vPanel setFrameOrigin:NSMakePoint(x, y)];

  /*
  [vPanel setFrame:NSMakeRect(x, y,
                              columnWidth + 30,
                              17 * i + 38)
          display:NO];
  */
}

- (NSArray *)parseLabel:(NSArray *)lines
{
  int i;
  NSString *line;
  NSArray *cols;
  NSMutableArray *labels;

  if (!lines || [lines count] < 2)
    return nil;

  line = [lines objectAtIndex:1];

  labels = [[NSMutableArray alloc] init];

  for (i = 0; i < [lines count] - 1; i++) {
    line = [lines objectAtIndex:i];
    if (!line || [line compare:@""] == NSOrderedSame)
      break;

    cols = [line componentsSeparatedByString:@"\t"];
    if (cols && [cols count] >= 2) {
      NSMutableString *label = [[NSMutableString alloc]
                                 initWithString:[cols objectAtIndex:0]];
      //fprintf(stderr, "label='%s'\n", [label UTF8String]);
      [labels addObject:label];
    }
  }

  return labels;
}

- (void)showModeTips:(int)qdX:(int)qdY:(int)height:(NSArray *)lines
{
  //if ([realModeTipsPanel isVisible] == NO) {
  int x, y;
  NSSize mainSize;
  NSRect rect;
  NSArray *labels;
  
  if (!lines || [lines count] <= 0)
    return;

  if (lastLabel && [lastLabel compare:[lines objectAtIndex:0]] == NSOrderedSame)
    return;
  [lastLabel release];
  lastLabel = [lines objectAtIndex:0];
  [lastLabel retain];

  labels = [self parseLabel:lines];
  //fprintf(stderr, "CocoaWinController::showModeTips() count=%d\n",
  //        [labels count]);
  if (!labels) return;
  [realModeTipsPanel showLabels:labels];
  [labels release];

  mainSize = [[NSScreen mainScreen] frame].size;
  rect = [realModeTipsPanel frame];

  x = qdX;
  y = mainSize.height - qdY - rect.size.height;
  [realModeTipsPanel setFrameOrigin:NSMakePoint(x, y)];

  [realModeTipsPanel orderFront:nil];
  [realModeTipsPanel setLevel:NSFloatingWindowLevel];
  //[realModeTipsPanel setAlphaValue:0.8];

  if (modeTipsTimer) {
    [modeTipsTimer invalidate];
    //[modeTipsTimer release];
  }

  modeTipsTimer = [NSTimer scheduledTimerWithTimeInterval:1.0
                           target:self
                           selector:@selector(modeTipsTimeout:)
                           userInfo:nil
                           repeats:NO];
  
  //fprintf(stderr, "CocoaWinController::showModeTips() modeTipsTimer=%p\n",
  //        modeTipsTimer);
}

- (void)hideModeTips
{
  [realModeTipsPanel orderOut:nil];
}

- (void)modeTipsTimeout:(NSTimer *)timer
{
  //fprintf(stderr, "CocoaWinController::modeTipsTimeout()\n");

  [self hideModeTips];

  modeTipsTimer = nil;
}

@end

/**
 * Carbon entry point and C-callable wrapper functions
 */
OSStatus
initializeBundle(OSStatus (*callBack)(int))
{
  CocoaWinController *candWin;
  NSAutoreleasePool *localPool;

  localPool = [[NSAutoreleasePool alloc] init];        

  candWin = [[CocoaWinController alloc] init];
  [candWin setCallBack:callBack];

  [localPool release];

  return noErr;
}

/**
 * Move candidates-window to front
 * This function called by a Carbon function.
 */
OSStatus
orderWindowFront(SInt16 inQDX, SInt16 inQDY, SInt16 inLineHeight)
{
  NSAutoreleasePool *localPool;
        
  localPool = [[NSAutoreleasePool alloc] init];        
  [[CocoaWinController sharedController] reloadData];
  [[CocoaWinController sharedController] showWindow:inQDX:inQDY:inLineHeight];
  [localPool release];

  return noErr;
}

/**
 * Move candidates-window to back
 * This function called by a Carbon function.
 */
OSStatus
orderWindowBack()
{
  NSAutoreleasePool *localPool;

  localPool = [[NSAutoreleasePool alloc] init];        
  [[CocoaWinController sharedController] hideWindow];
  [localPool release];

  return noErr;
}

/**
 * Returns YES if the candidate window is visible
 */
Boolean
windowIsVisible()
{
  NSAutoreleasePool *localPool;
  Boolean visible = false;
  
  localPool = [[NSAutoreleasePool alloc] init];
  visible = [[CocoaWinController sharedController] isVisible];
  [localPool release];
  
  return visible;
}

/**
 * Get a candidate string
 * This function called by a Carbon function.
 */
UniCharPtr
getCandidate(UInt32 inIndex)
{
  NSAutoreleasePool *localPool;
  UniCharPtr str = nil;

  localPool = [[NSAutoreleasePool alloc] init];        
  str = [[CocoaWinController sharedController] getCandidate:inIndex];
  [localPool release];

  return str;
}

/**
 * Add a candidate
 * This function called by a Carbon function.
 */
OSStatus
addCandidate(UniCharPtr inHead, int inHeadLen,
             UniCharPtr inCand, int inCandLen)
{
  NSAutoreleasePool *localPool;

  localPool = [[NSAutoreleasePool alloc] init];        
  [[CocoaWinController sharedController]
    addCandidate:inHead:inHeadLen:inCand:inCandLen];
  [localPool release];

  return noErr;
}

/**
 * Clear candidates
 * This function called by a Carbon function.
 */
OSStatus
clearCandidate()
{
  NSAutoreleasePool *localPool;

  localPool = [[NSAutoreleasePool alloc] init];        
  [[CocoaWinController sharedController] clearCandidate];
  [localPool release];

  return noErr;
}

/**
 * Select a candidate
 * This function called by a Carbon function.
 */
OSStatus
selectCandidate(int inIndex)
{
  NSAutoreleasePool *localPool;

  localPool = [[NSAutoreleasePool alloc] init];        
  [[CocoaWinController sharedController] selectCandidate:inIndex];
  [localPool release];

  return noErr;
}

/**
 * deselect a candidate
 * This function called by a Carbon function.
 */
OSStatus
deselectCandidate(int inIndex)
{
  NSAutoreleasePool *localPool;

  localPool = [[NSAutoreleasePool alloc] init];        
  [[CocoaWinController sharedController] deselectCandidate];
  [localPool release];

  return noErr;
}

/**
 * Set a page label
 * This function called by a Carbon function.
 */
OSStatus
setPage(int inIndex, int inMax)
{
  NSAutoreleasePool *localPool;

  localPool = [[NSAutoreleasePool alloc] init];        
  [[CocoaWinController sharedController] setPage:inIndex:inMax];
  [localPool release];

  return noErr;
}

OSStatus
showModeTips(SInt16 inQDX, SInt16 inQDY, SInt16 inLineHeight, CFArrayRef inLines)
{
  NSAutoreleasePool *localPool;
  
  localPool = [[NSAutoreleasePool alloc] init];        
  [[CocoaWinController sharedController] showModeTips:inQDX:inQDY:inLineHeight:(NSArray *)inLines];
  [localPool release];
  
  return noErr;
}

OSStatus
hideModeTips()
{
  NSAutoreleasePool *localPool;
  
  localPool = [[NSAutoreleasePool alloc] init];        
  [[CocoaWinController sharedController] hideModeTips];
  [localPool release];
  
  return noErr;
}
