/* -*- mode:objc; coding:utf-8; tab-width:8; c-basic-offset:2; indent-tabs-mode:nil -*- */
/*
  Copyright (c) 2004 MacUIM Project
  http://www.digital-genes.com/~yatsu/macuim/

  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
  3. Neither the name of authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
  SUCH DAMAGE.
*/

#import "CandidateController.h"

static CandidateController *sharedController;

@implementation CandidateController
 
+ (CandidateController *)sharedController
{
  return sharedController;
}

/**
 * Initialize Cocoa
 */
- (id)init
{
  self = [super init];
  NSApplicationLoad();
  if (![NSBundle loadNibNamed:@"CandidateWindow" owner:self]) {
    NSLog(@"failed to load CandidateWindow nib");
  }
  sharedController = self;
  return self;
}

/**
 * Set a callback for a CandidateController
 */
- (void)setCallBack:(CallBackType)callBack
{
  _callBack = callBack;
}

/**
 * Show a CandidateController and make it activate
 */
- (void)showWindow:(int)qdX:(int)qdY
{
  NSSize mainSize = [[NSScreen mainScreen] frame].size;
  NSSize winSize = [window frame].size;
  float cgX = qdX;
  float cgY = mainSize.height - qdY - winSize.height - 4;

  printf("CandidateController::showWindow() qdX=%d qdY=%d cgX=%f cgY=%f\n",
         qdX, qdY, cgX, cgY);

  /*
  printf("MainScreen - height=%f width=%f\n",
         mainSize.height, mainSize.width);
  printf("Window - height=%f width=%f\n",
         winSize.height, winSize.width);
  */

  if (cgX < 0)
    cgX = 0;
  if (cgX + winSize.width > mainSize.width)
    cgX = mainSize.width - winSize.width;
  if (cgY < 0)
    cgY = 0;
  if (cgY + winSize.height > mainSize.height)
    cgY = mainSize.height - winSize.height;

  [window setFrameOrigin:NSMakePoint(cgX, cgY)];

  [window makeFirstResponder:table];

  //[window makeKeyAndOrderFront:nil];

  //[window makeKeyWindow];
  [window orderFront:nil];
  [window setLevel:NSFloatingWindowLevel];

  //[window setAlphaValue:0.8];
}

/**
 * Hide a candidates-window
 */
- (void)hideWindow
{
  printf("hideWindow()\n");

  [window orderOut:nil];
}

/**
 * Request the NSTableView to reload candidates
 */
- (void)reloadData
{
  printf("reloadData()\n");

  [table reloadData];
}

/**
 * Initialize
 */
- (void)awakeFromNib
{
  headArray = [[NSMutableArray alloc] init];
  candArray = [[NSMutableArray alloc] init];
}

/**
 * Get a number of rows in the TableView
 */
- (int)numberOfRowsInTableView:(NSTableView *)aTableView
{
  return [headArray count];
}

/**
 * Get data
 */
- (id)tableView:(NSTableView *)aTableView objectValueForTableColumn
               :(NSTableColumn *)aTableColumn row
               :(int)rowIndex
{
  id colID = [aTableColumn identifier];

  if ([colID isEqual:@"head"])
    return [headArray objectAtIndex:rowIndex];
  else if ([colID isEqual:@"candidate"])
    return [candArray objectAtIndex:rowIndex];

  return nil;
}

- (UniCharPtr)getCandidate:(int)index
{
  return nil;
}

- (void)addCandidate:(UniCharPtr)head:(int)headLen
                    :(UniCharPtr)cand:(int)candLen
{
  NSString *headStr;
  NSString *candStr;

  if (head && headLen > 0)
    headStr = [[NSString alloc] initWithCharacters:head
                                length:headLen];
  else
    headStr = [[NSString alloc] initWithString:@""];;

  if (cand && candLen > 0)
    candStr = [[NSString alloc] initWithCharacters:cand
                                length:candLen];
  else
    candStr = [[NSString alloc] initWithString:@""];;

  printf("CandidateController::addCandidate()\n");

  [headArray addObject:headStr];
  [candArray addObject:candStr];
}

/**
 * Clear candidates
 */
- (void)clearCandidate
{
  [headArray removeAllObjects];
  [candArray removeAllObjects];
}

/**
 * Select a candidate
 */
- (void)selectCandidate:(int)index
{
  NSIndexSet *indexSet;

  indexSet = [[NSIndexSet alloc] initWithIndex:index];

  [table selectRowIndexes:indexSet byExtendingSelection:nil];
  [table scrollRowToVisible:index];

  [indexSet release];
}

/**
 * Button press action
 */
- (IBAction)candClicked:(id)sender
{
  printf("CandidateController::candClicked()\n");

  [window orderOut:nil];

  (*_callBack)(kEventCandClicked);
}

@end

/**
 * Carbon entry point and C-callable wrapper functions
 */
OSStatus
initializeBundle(OSStatus (*callBack)(int))
{
  CandidateController *candWin;
  NSAutoreleasePool *localPool;
    
  printf("CandidateController::initializeBundle() callBack=%p\n", callBack);

  localPool = [[NSAutoreleasePool alloc] init];        
  candWin = [[CandidateController alloc] init];
  [candWin setCallBack:callBack];
  [localPool release];

  return noErr;
}

/**
 * Move candidates-window to front
 * This function called by a Carbon function.
 */
OSStatus
orderWindowFront(SInt16 inQDX, SInt16 inQDY)
{
  NSAutoreleasePool *localPool;
        
  printf("orderWindowFront() inQDX=%hd inQDY=%hd\n",
         inQDX, inQDY);

  localPool = [[NSAutoreleasePool alloc] init];        
  [[CandidateController sharedController] reloadData];
  [[CandidateController sharedController] showWindow:inQDX:inQDY];
  [localPool release];

  return noErr;
}

/**
 * Move candidates-window to back
 * This function called by a Carbon function.
 */
OSStatus
orderWindowBack()
{
  NSAutoreleasePool *localPool;

  printf("orderWindowBack()\n");

  localPool = [[NSAutoreleasePool alloc] init];        
  [[CandidateController sharedController] hideWindow];
  [localPool release];

  return noErr;
}

/**
 * Get a candidate string
 * This function called by a Carbon function.
 */
UniCharPtr
getCandidate(UInt32 inIndex)
{
  NSAutoreleasePool *localPool;
  UniCharPtr str = nil;

  printf("getCandidate()\n");

  localPool = [[NSAutoreleasePool alloc] init];        
  str = [[CandidateController sharedController] getCandidate:inIndex];
  [localPool release];

  return str;
}

/**
 * Add a candidate
 * This function called by a Carbon function.
 */
OSStatus
addCandidate(UniCharPtr inHead, int inHeadLen,
             UniCharPtr inCand, int inCandLen)
{
  NSAutoreleasePool *localPool;

  printf("addCandidate()\n");

  localPool = [[NSAutoreleasePool alloc] init];        
  [[CandidateController sharedController]
    addCandidate:inHead:inHeadLen:inCand:inCandLen];
  [localPool release];

  return noErr;
}

/**
 * Clear candidates
 * This function called by a Carbon function.
 */
OSStatus
clearCandidate()
{
  NSAutoreleasePool *localPool;

  printf("addCandidate()\n");

  localPool = [[NSAutoreleasePool alloc] init];        
  [[CandidateController sharedController] clearCandidate];
  [localPool release];

  return noErr;
}

/**
 * Select a candidate
 * This function called by a Carbon function.
 */
OSStatus
selectCandidate(int inIndex)
{
  NSAutoreleasePool *localPool;

  printf("selectCandidate() inIndex=%d\n", inIndex);

  localPool = [[NSAutoreleasePool alloc] init];        
  [[CandidateController sharedController] selectCandidate:inIndex];
  [localPool release];

  return noErr;
}
