/* -*- mode:c; coding:utf-8; tab-width:8; c-basic-offset:2; indent-tabs-mode:nil -*- */
/*
  Copyright (c) 2004 MacUIM Project
  http://www.digital-genes.com/~yatsu/macuim/

  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
  3. Neither the name of authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
  SUCH DAMAGE.
*/

#define TARGET_API_MAC_CARBON 1

#include <Carbon/Carbon.h>

#include "MUIM.h"
#include "MUIMInputEvents.h"


/**
 * Create and sends an "update active input area" event to our client
 * application. We use the Carbon event manager to create a Carbon event,
 * add the appropriate parameters and send it using SendTextInputEvent.
 *
 * @param  inHandle  a reference to the active session
 * @param  inFix     if TRUE, we are fixing the entire input area
 *                   Otherwise we are just updating the contents
 *
 * @return OSStatus  a toolbox error code
 */
OSErr
MUIMUpdateActiveInputArea(MUIMSessionHandle inHandle, Boolean inFix)
{
  OSErr error;
  EventRef event;
  ComponentInstance componentInstance;
  ScriptLanguageRecord scriptLanguageRecord;
  TextRange pinRange;
  TextRangeArrayPtr hiliteRangePtr;
  TextRangeArrayPtr updateRangePtr;
  UniCharPtr str = NULL;
  UInt32 len = 0, cursorPos = 0;
  SInt32 fixLen;

  if (!inFix) {
    GetPreeditString(inHandle, &str, &len, &cursorPos);

    if ((*inHandle)->fOldPreedit != NULL &&
        !memcmp((*inHandle)->fOldPreedit, str, sizeof(UniChar) * len)) {
      free(str);
      return noErr;
    }
    free((*inHandle)->fOldPreedit);
    (*inHandle)->fOldPreedit = str;
    fixLen = 0;
  }
  else {
    len = cursorPos = (*inHandle)->fFixLen;
    fixLen = len * sizeof(UniChar);
    str = (*inHandle)->fFixBuffer;
    /*
    str = (UniCharPtr) malloc(fixLen + sizeof(UniChar));
    memcpy(str, (*inHandle)->fFixBuffer, fixLen);
    str[len] = '\0';
    */
  }

  DEBUG_PRINT("MUIMUpdateActiveInputArea() inFix=%s len=%lu cursorPos=%lu fixLen=%ld\n",
              inFix ? "true" : "false",
              len, cursorPos, fixLen);
  //DumpString("str", (char *) str, len * sizeof(UniChar));

  hiliteRangePtr = nil;
  updateRangePtr = nil;

  // create an event
  error =
    CreateEvent(NULL, kEventClassTextInput,
                kEventTextInputUpdateActiveInputArea,
                GetCurrentEventTime(),
                kEventAttributeUserEvent, &event);

  // set a component instance
  if (error == noErr) {
    componentInstance = (*inHandle)->fComponentInstance;
    error = SetEventParameter(event,
                              kEventParamTextInputSendComponentInstance,
                              typeComponentInstance,
                              sizeof(ComponentInstance),
                              &componentInstance);
  }

  // set a script and language type
  if (error == noErr) {
    scriptLanguageRecord.fScript = smJapanese;
    scriptLanguageRecord.fLanguage = langJapanese;
    error = SetEventParameter(event, kEventParamTextInputSendSLRec,
                              typeIntlWritingCode,
                              sizeof(ScriptLanguageRecord),
                              &scriptLanguageRecord);
  }

  // set the string
  if (error == noErr)
    error = SetEventParameter(event, kEventParamTextInputSendText,
                              typeUnicodeText,
                              len * sizeof(UniChar), str);

  //DEBUG_PRINT("MUIMUpdateActiveInputArea - kEventParamTextInputSendText size=%d\n",
  //            len * sizeof(UniChar));
 
  // set the fix length
  if (error == noErr)
    error = SetEventParameter(event, kEventParamTextInputSendFixLen,
                              typeLongInteger,
                              sizeof(SInt32), &fixLen);

  //DEBUG_PRINT("MUIMUpdateActiveInputArea - kEventParamTextInputSendFixLen size=%d val=%d\n",
  //            sizeof(SInt32), fixLen);
 
  // set the update range
  if (error == noErr) {
    updateRangePtr =
      (TextRangeArrayPtr) NewPtrClear(sizeof(short) +
                                      sizeof(TextRange) * 2);
    if (updateRangePtr) {
      updateRangePtr->fNumOfRanges = 2;
      updateRangePtr->fRange[0].fStart = 0;
      updateRangePtr->fRange[0].fEnd =
        (*inHandle)->fLastUpdateLength * sizeof(UniChar);
      updateRangePtr->fRange[1].fHiliteStyle = 0;
      updateRangePtr->fRange[1].fStart = 0;
      updateRangePtr->fRange[1].fEnd = len * sizeof(UniChar);
      updateRangePtr->fRange[1].fHiliteStyle = 0;

      (*inHandle)->fLastUpdateLength = len;
    }
    else
      error = memFullErr;
  }
  if (error == noErr)
    error = SetEventParameter(event, kEventParamTextInputSendUpdateRng,
                              typeTextRangeArray,
                              sizeof(short) + sizeof(TextRange) * 2,
                              updateRangePtr);

  //DEBUG_PRINT("MUIMUpdateActiveInputArea - fRange[0].fEnd=%d fRange[1].fEnd=%d\n",
  //            updateRangePtr->fRange[0].fEnd,
  //            updateRangePtr->fRange[1].fEnd);

  // set the hilite range
  if (error == noErr) {
    hiliteRangePtr =
      (TextRangeArrayPtr) NewPtrClear(sizeof(short) +
                                      sizeof(TextRange) * 2);
    if (hiliteRangePtr) {
      hiliteRangePtr->fNumOfRanges = 2;
      hiliteRangePtr->fRange[0].fStart = 0;
      hiliteRangePtr->fRange[0].fEnd = len * sizeof(UniChar);
      if (inFix)
        hiliteRangePtr->fRange[0].fHiliteStyle = kConvertedText;
      else
        hiliteRangePtr->fRange[0].fHiliteStyle = kRawText;
      hiliteRangePtr->fRange[1].fStart = cursorPos * sizeof(UniChar);
      hiliteRangePtr->fRange[1].fEnd = cursorPos * sizeof(UniChar);
      hiliteRangePtr->fRange[1].fHiliteStyle = kCaretPosition;
    }
    else
      error = memFullErr;
  }
  if (error == noErr)
    error = SetEventParameter(event, kEventParamTextInputSendHiliteRng,
                              typeTextRangeArray,
                              sizeof(short) + sizeof(TextRange) * 2,
                              hiliteRangePtr);

  //DEBUG_PRINT("MUIMUpdateActiveInputArea - fRange[0].fEnd=%d fRange[1].fStart=%d fRange[1].fEnd=%d\n",
  //            hiliteRangePtr->fRange[0].fEnd,
  //            hiliteRangePtr->fRange[0].fStart,
  //            hiliteRangePtr->fRange[0].fEnd);

  pinRange.fStart = 0;
  pinRange.fEnd = len * sizeof(UniChar);
  if (error == noErr)
    error = SetEventParameter(event, kEventParamTextInputSendPinRng,
                              typeTextRange, sizeof(TextRange), &pinRange);

  //DEBUG_PRINT("MUIMUpdateActiveInputArea - pinRange.fEnd=%d\n",
  //            pinRange.fEnd);

  if (error == noErr)
    error = SendTextInputEvent(event);

  DisposePtr((Ptr) hiliteRangePtr);
  DisposePtr((Ptr) updateRangePtr);

  //free(str);

  return error;
}

/**
 * Get a TSM status
 *
 * @param inHandle   a reference to the active session
 * @param outStat    TSM status
 *
 * @return OSStatus  a toolbox error code
 */
OSErr
MUIMGetTSMStatus(MUIMSessionHandle inHandle, TSMStat *outStat)
{
  OSErr error = noErr;
  ComponentInstance componentInstance;
  EventRef event;

  // create an event
  error = CreateEvent(NULL, kEventClassTextInput,
                      kEventTextInputOffsetToPos,
                      GetCurrentEventTime(),
                      kEventAttributeUserEvent,
                      &event);

  // set a component instance
  if (error == noErr) {
    componentInstance = (*inHandle)->fComponentInstance;
    error = SetEventParameter(event,
                              kEventParamTextInputSendComponentInstance,
                              typeComponentInstance,
                              sizeof(ComponentInstance),
                              &componentInstance);
  }

  if (error == noErr)
    error = SetEventParameter(event,
                              kEventParamTextInputSendRefCon,
                              typeLongInteger,
                              sizeof(long),
                              &outStat->fRefCon);


  if (error == noErr)
    error = SetEventParameter(event,
                              kEventParamTextInputSendTextOffset,
                              typeLongInteger,
                              sizeof(long),
                              &outStat->fTextOffset);


  if (error == noErr)
    error = SetEventParameter(event,
                              kEventParamTextInputSendSLRec,
                              typeIntlWritingCode,
                              sizeof(ScriptLanguageRecord),
                              &outStat->fSLRec);

  if (error == noErr)
    error = SetEventParameter(event,
                              kEventParamTextInputSendLeadingEdge,
                              typeBoolean,
                              sizeof(Boolean),
                              &outStat->fLeadingEdge);

  // send a event
  if (error == noErr)
    error = SendTextInputEvent(event);

  // get a reply point
  if (error == noErr)
    error = GetEventParameter(event,
                              kEventParamTextInputReplyPoint,
                              typeQDPoint,
                              NULL, sizeof(Point), NULL,
                              &outStat->fReplyPoint);
  if (event)
    ReleaseEvent(event);

  if (error)
    DEBUG_PRINT("MUIMGetTSMStatus() error=%ld\n", error);

  return error;
}
