/* -*- mode:c; coding:utf-8; tab-width:8; c-basic-offset:2; indent-tabs-mode:nil -*- */
/*
  Copyright (c) 2004 MacUIM Project
  http://www.digital-genes.com/~yatsu/macuim/

  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
  3. Neither the name of authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
  SUCH DAMAGE.
*/

#include <uim/uim.h>
#include <uim/uim-helper.h>

#include "MUIM.h"
#include "MUIMInputEvents.h"
#include "UIMCallback.h"
#include "CandidateCarbon.h"


static void
AddPreeditSegment(MUIMSessionHandle inHandle, int inAttr,
                  const char *inStr);


void
UIMCommitString(void *ptr, char *str)
{
  DEBUG_PRINT("UIMCommitString() len=%d\n", strlen(str));

  MUIMSessionHandle handle = (MUIMSessionHandle) ptr;
  CFMutableStringRef cf_string;

  cf_string = CFStringCreateMutable(NULL, 0);
  CFStringAppendCString(cf_string, str, kCFStringEncodingUTF8);

  (*handle)->fFixLen = CFStringGetLength(cf_string);
  (*handle)->fFixBuffer =
    (UniCharPtr) malloc(sizeof(UniChar) * ((*handle)->fFixLen + 1));
  CFStringGetCharacters(cf_string, CFRangeMake(0, (*handle)->fFixLen),
                        (*handle)->fFixBuffer);
  CFRelease(cf_string);

  /*
    DEBUG_PRINT("UIMCommitString() fFixLen=%lu\n",
    (*handle)->fFixLen);
    DumpString("utf8", (char *) str, strlen(str));
    DumpString("utf16", (char *) (*handle)->fFixBuffer,
    (*handle)->fFixLen * sizeof(UniChar));
  */

  if ((*handle)->fFixLen > 0) {
    MUIMUpdateActiveInputArea(handle, TRUE);
    free((*handle)->fFixBuffer);
    (*handle)->fFixBuffer = NULL;
    (*handle)->fFixLen = 0;
  }
}

void
UIMMode(void *ptr, int mode)
{
  MUIMSessionHandle handle = (MUIMSessionHandle) ptr;

  DEBUG_PRINT("UIMMode() mode=%d oldmode=%d\n",
              mode, (*handle)->fMode);

  if ((*handle)->fMode == mode)
    return;

  (*handle)->fMode = mode;
}

void
UIMPreeditClear(void *ptr)
{
  MUIMSessionHandle handle = (MUIMSessionHandle) ptr;
  UInt32 i;

  DEBUG_PRINT("UIMPreeditClear()\n");

  for (i = 0; i < (*handle)->fSegmentCount; i++)
    free((*handle)->fSegments[i].fBuffer);

  free((*handle)->fSegments);
  (*handle)->fSegments = NULL;
  (*handle)->fSegmentCount = 0;
}

void
UIMPreeditPushback(void *ptr, int attr, char *str)
{
  MUIMSessionHandle handle = (MUIMSessionHandle) ptr;

#ifdef DEBUG
  {
    char *tmpstr = "";
    char attrstr[50];

    if (attr & UPreeditAttr_None) {
      tmpstr = "None";
      sprintf(attrstr, "%s", tmpstr);
    }
    if (attr & UPreeditAttr_UnderLine) {
      tmpstr = "UnderLine";
      sprintf(attrstr, "%s %s", attrstr, tmpstr);
    }
    if (attr & UPreeditAttr_Reverse) {
      tmpstr = "Reverse";
      sprintf(attrstr, "%s %s", attrstr, tmpstr);
    }
    if (attr & UPreeditAttr_Cursor) {
      tmpstr = "Cursor";
      sprintf(attrstr, "%s %s", attrstr, tmpstr);
    }
    if (attr & UPreeditAttr_Separator) {
      tmpstr = "Separator";
      sprintf(attrstr, "%s %s", attrstr, tmpstr);
    }

    DEBUG_PRINT("UIMPreeditPushback() fSegmentCount=%lu attr=%s len=%lu\n",
                (*handle)->fSegmentCount,
                attrstr, strlen(str));
  }
#endif

  if (!strcmp(str, "")
      && !(attr & (UPreeditAttr_Cursor | UPreeditAttr_Separator | UPreeditAttr_UnderLine)))
    return;

  AddPreeditSegment(handle, attr, str);
}

void
UIMPreeditUpdate(void *ptr)
{
  MUIMSessionHandle handle = (MUIMSessionHandle) ptr;

  DEBUG_PRINT("UIMPreeditUpdate()\n");

  MUIMUpdateActiveInputArea(handle, FALSE);
}

/**
 * Candidate window activate callback
 */
void
UIMCandAcivate(void *inPtr, int inNR, int inLimit)
{
  MUIMSessionHandle handle = (MUIMSessionHandle) inPtr;

  DEBUG_PRINT("UIMCandAcivate() inNR=%d inLimit=%d\n", inNR, inLimit);

  InitCandidateWindow(handle);

  (*handle)->fIsActive = true;
  (*handle)->fCandidateIndex = -1;
  (*handle)->fNRCandidates = inNR;
  (*handle)->fDisplayLimit = inLimit;
  (*handle)->fLayoutBegin = 0;

  LayoutCandidate(handle);
}

/**
 * Candidate window select callback
 */
void
UIMCandSelect(void *inPtr, int inIndex)
{
  MUIMSessionHandle handle = (MUIMSessionHandle) inPtr;

  DEBUG_PRINT("UIMCandSelect() inIndex=%d\n", inIndex);

  (*handle)->fCandidateIndex = inIndex;

  LayoutCandidate(handle);
}

/**
 * Candidate window page shift callback
 */
void
UIMCandShiftPage(void *inPtr, int inDirection)
{
  MUIMSessionHandle handle = (MUIMSessionHandle) inPtr;

  if (inDirection) {
    if ((*handle)->fNRCandidates <
        (*handle)->fCandidateIndex + (*handle)->fDisplayLimit)
      (*handle)->fCandidateIndex = 0;
    else
      (*handle)->fCandidateIndex += (*handle)->fDisplayLimit;
  }
  else {
    if ((*handle)->fCandidateIndex - (*handle)->fDisplayLimit < 0)
      (*handle)->fCandidateIndex = (*handle)->fNRCandidates - 1;
    else
      (*handle)->fCandidateIndex -= (*handle)->fDisplayLimit;
  }

  DEBUG_PRINT("UIMCandShiftPage() inDirection=%d fCandidateIndex=%ld\n",
              inDirection, (*handle)->fCandidateIndex);

  LayoutCandidate(handle);
  uim_set_candidate_index((*handle)->fUC, (*handle)->fCandidateIndex);
}

/**
 * Candidate window deactivate callback
 */
void
UIMCandDeactivate(void *inPtr)
{
  MUIMSessionHandle handle = (MUIMSessionHandle) inPtr;

  DEBUG_PRINT("UIMCandDeactivate()\n");

  HideCandidateWindow(handle);
}

static void
AddPreeditSegment(MUIMSessionHandle inHandle, int inAttr,
                  const char *inStr)
{
  CFMutableStringRef cf_string;
  int len;
  UniCharPtr unistr;

  cf_string = CFStringCreateMutable(NULL, 0);
  CFStringAppendCString(cf_string, inStr, kCFStringEncodingUTF8);

  len = CFStringGetLength(cf_string);
  unistr = (UniCharPtr) malloc(sizeof(UniChar) * (len + 1));
  CFStringGetCharacters(cf_string, CFRangeMake(0, len), unistr);

  //DumpString("unistr", (char *) unistr, len * sizeof(UniChar));

  (*inHandle)->fSegments = realloc((*inHandle)->fSegments,
                                   sizeof(PreeditSegment) *
                                   ((*inHandle)->fSegmentCount + 1));
  (*inHandle)->fSegments[(*inHandle)->fSegmentCount].fBuffer = unistr;
  (*inHandle)->fSegments[(*inHandle)->fSegmentCount].fLength = len;
  (*inHandle)->fSegments[(*inHandle)->fSegmentCount].fAttr = inAttr;
  (*inHandle)->fSegmentCount++;

  CFRelease(cf_string);
}

void
GetPreeditSegment(PreeditSegment *inSegment, UniCharPtr *outStr,
                  UInt32 *outLen)
{
  UniCharPtr tmp = (UniCharPtr) malloc(sizeof(UniChar) * (*outLen + 1));
  memcpy(tmp, *outStr, sizeof(UniChar) * (*outLen));

  DEBUG_PRINT("GetPreeditSegment() outLen=%lu, fLength=%lu\n",
              *outLen, inSegment->fLength);
  //DumpString("tmp", (char *) tmp,
  //           sizeof(UniChar) * (*outLen));

  (*outStr) = (UniCharPtr) realloc(*outStr, sizeof(UniChar) *
                                   ((*outLen) + inSegment->fLength + 1));

  memcpy(*outStr, tmp, sizeof(UniChar) * (*outLen));
  memcpy(&((*outStr)[(*outLen)]),
         inSegment->fBuffer, sizeof(UniChar) * inSegment->fLength);
  //(*outStr)[(*outLen) + inSegment->fLength] = '\0';
  *outLen += inSegment->fLength;

  free(tmp);

  /*
    DumpString("fBuffer", (char *) inSegment->fBuffer,
    sizeof(UniChar) * inSegment->fLength);
    DumpString("outStr", (char *) (*outStr),
    sizeof(UniChar) * (*outLen));
  */
}

void
GetPreeditString(MUIMSessionHandle inHandle, UniCharPtr *outStr,
                 UInt32 *outLen, UInt32 *outCursorPos)
{
  UniCharPtr str;
  UInt32 i, pos = 0, len = 0;

  str = (UniCharPtr) malloc(sizeof(UniChar));
  str[0] = '\0';

  DEBUG_PRINT("GetPreeditString() fSegmentCount=%lu\n",
              (*inHandle)->fSegmentCount);
  for (i = 0; i < (*inHandle)->fSegmentCount; i++) {
    GetPreeditSegment(&((*inHandle)->fSegments[i]), &str, &len);
    if ((*inHandle)->fSegments[i].fAttr & UPreeditAttr_Cursor) {
      pos = len;
    }
    DEBUG_PRINT("GetPreeditString() i=%lu len=%lu\n", i, len);
  }
  DEBUG_PRINT("GetPreeditString() len=%lu pos=%lu\n", len, pos);

  if (outCursorPos)
    *outCursorPos = pos;

  if (outLen)
    *outLen = len;

  if (outStr)
    *outStr = str;
  else
    free(str);
}
