/* -*- mode:c; coding:utf-8; tab-width:8; c-basic-offset:2; indent-tabs-mode:nil -*- */
/*
  Copyright (c) 2004 MacUIM Project
  http://www.digital-genes.com/~yatsu/macuim/

  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
  3. Neither the name of authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
  SUCH DAMAGE.
*/

#include <uim/uim.h>
#include <uim/uim-helper.h>

#include "MUIM.h"
#include "MUIMInputEvents.h"
#include "UIMCallback.h"
#include "CandidateCarbon.h"

extern MUIMSessionHandle gActiveSession;
extern Boolean gDisableFocusedContext;

extern int gNumSession;
extern MUIMSessionHandle *gSessionList;

extern uim_context gUC;

extern int gUimFD;
extern CFSocketRef gUimSock;
extern CFRunLoopSourceRef gUimRun;
extern CFSocketContext gSockContext;


//#define DEBUG_HELPER  1
//#define DEBUG_PREEDIT  1
//#define DEBUG_CANDIDATES  1


static void
AddPreeditSegment(MUIMSessionHandle inHandle, int inAttr,
                  const char *inStr);

static void
HelperRead(CFSocketRef sock, CFSocketCallBackType callbackType, 
           CFDataRef address, const void *data, void *info);

static void
ParseHelperString(const char *str);


void
UIMCommitString(void *ptr, const char *str)
{
#ifdef DEBUG_PREEDIT
  DEBUG_PRINT("UIMCommitString() len=%d\n", strlen(str));
#endif

  //MUIMSessionHandle handle = (MUIMSessionHandle) ptr;
  MUIMSessionHandle handle = gActiveSession;
  CFMutableStringRef cf_string;

  cf_string = CFStringCreateMutable(NULL, 0);
  CFStringAppendCString(cf_string, str, kCFStringEncodingUTF8);

  (*handle)->fFixLen = CFStringGetLength(cf_string);
  (*handle)->fFixBuffer =
    (UniCharPtr) malloc(sizeof(UniChar) * ((*handle)->fFixLen + 1));
  CFStringGetCharacters(cf_string, CFRangeMake(0, (*handle)->fFixLen),
                        (*handle)->fFixBuffer);
  CFRelease(cf_string);

  if ((*handle)->fFixLen > 0) {
    MUIMUpdateActiveInputArea(handle, TRUE);
    free((*handle)->fFixBuffer);
    (*handle)->fFixBuffer = NULL;
    (*handle)->fFixLen = 0;
  }
}


void
UIMPreeditClear(void *ptr)
{
  //MUIMSessionHandle handle = (MUIMSessionHandle) ptr;
  MUIMSessionHandle handle = gActiveSession;
  UInt32 i;

#ifdef DEBUG_PREEDIT
  DEBUG_PRINT("UIMPreeditClear()\n");
#endif

  for (i = 0; i < (*handle)->fSegmentCount; i++)
    free((*handle)->fSegments[i].fBuffer);

  free((*handle)->fSegments);
  (*handle)->fSegments = NULL;
  (*handle)->fSegmentCount = 0;
}

void
UIMPreeditPushback(void *ptr, int attr, const char *str)
{
  //MUIMSessionHandle handle = (MUIMSessionHandle) ptr;
  MUIMSessionHandle handle = gActiveSession;

#ifdef DEBUG
  {
    char *tmpstr = "";
    char attrstr[50];

    if (attr & UPreeditAttr_None) {
      tmpstr = "None";
      sprintf(attrstr, "%s", tmpstr);
    }
    if (attr & UPreeditAttr_UnderLine) {
      tmpstr = "UnderLine";
      sprintf(attrstr, "%s %s", attrstr, tmpstr);
    }
    if (attr & UPreeditAttr_Reverse) {
      tmpstr = "Reverse";
      sprintf(attrstr, "%s %s", attrstr, tmpstr);
    }
    if (attr & UPreeditAttr_Cursor) {
      tmpstr = "Cursor";
      sprintf(attrstr, "%s %s", attrstr, tmpstr);
    }
    if (attr & UPreeditAttr_Separator) {
      tmpstr = "Separator";
      sprintf(attrstr, "%s %s", attrstr, tmpstr);
    }

#ifdef DEBUG_PREEDIT
    DEBUG_PRINT("UIMPreeditPushback() fSegmentCount=%lu attr=%s len=%lu\n",
                (*handle)->fSegmentCount,
                attrstr, strlen(str));
#endif
  }
#endif

  if (!strcmp(str, "")
      && !(attr & (UPreeditAttr_Cursor | UPreeditAttr_Separator | UPreeditAttr_UnderLine)))
    return;

  AddPreeditSegment(handle, attr, str);
}

void
UIMPreeditUpdate(void *ptr)
{
  //MUIMSessionHandle handle = (MUIMSessionHandle) ptr;
  MUIMSessionHandle handle = gActiveSession;

#ifdef DEBUG_PREEDIT
  DEBUG_PRINT("UIMPreeditUpdate()\n");
#endif

  MUIMUpdateActiveInputArea(handle, FALSE);
}

/**
 * Candidate window activate callback
 */
void
UIMCandAcivate(void *inPtr, int inNR, int inLimit)
{
  //MUIMSessionHandle handle = (MUIMSessionHandle) inPtr;
  MUIMSessionHandle handle = gActiveSession;

#ifdef DEBUG_CANDIDATES
  DEBUG_PRINT("UIMCandAcivate() inNR=%d inLimit=%d\n", inNR, inLimit);
#endif

  InitCandidateWindow(handle);

  (*handle)->fIsActive = true;
  (*handle)->fCandidateIndex = -1;
  (*handle)->fNRCandidates = inNR;
  (*handle)->fDisplayLimit = inLimit;
  (*handle)->fLayoutBegin = 0;

  LayoutCandidate(handle);
}

/**
 * Candidate window select callback
 */
void
UIMCandSelect(void *inPtr, int inIndex)
{
  //MUIMSessionHandle handle = (MUIMSessionHandle) inPtr;
  MUIMSessionHandle handle = gActiveSession;

#ifdef DEBUG_CANDIDATES
  DEBUG_PRINT("UIMCandSelect() inIndex=%d\n", inIndex);
#endif

  (*handle)->fCandidateIndex = inIndex;

  LayoutCandidate(handle);
}

/**
 * Candidate window page shift callback
 */
void
UIMCandShiftPage(void *inPtr, int inDirection)
{
  //MUIMSessionHandle handle = (MUIMSessionHandle) inPtr;
  MUIMSessionHandle handle = gActiveSession;

  if (inDirection) {
    if ((*handle)->fNRCandidates <
        (*handle)->fCandidateIndex + (*handle)->fDisplayLimit)
      (*handle)->fCandidateIndex = 0;
    else
      (*handle)->fCandidateIndex += (*handle)->fDisplayLimit;
  }
  else {
    if ((*handle)->fCandidateIndex - (*handle)->fDisplayLimit < 0)
      (*handle)->fCandidateIndex = (*handle)->fNRCandidates - 1;
    else
      (*handle)->fCandidateIndex -= (*handle)->fDisplayLimit;
  }

#ifdef DEBUG_CANDIDATES
  DEBUG_PRINT("UIMCandShiftPage() inDirection=%d fCandidateIndex=%ld\n",
              inDirection, (*handle)->fCandidateIndex);
#endif

  LayoutCandidate(handle);
  uim_set_candidate_index(gUC, (*handle)->fCandidateIndex);
}

/**
 * Candidate window deactivate callback
 */
void
UIMCandDeactivate(void *inPtr)
{
  //MUIMSessionHandle handle = (MUIMSessionHandle) inPtr;
  MUIMSessionHandle handle = gActiveSession;

#ifdef DEBUG_CANDIDATES
  DEBUG_PRINT("UIMCandDeactivate()\n");
#endif

  HideCandidateWindow(handle);
}

void
UIMUpdatePropList(void *inPtr, const char *inStr)
{
  char *tmp;

#ifdef DEBUG_HELPER
  DEBUG_PRINT("UIMUpdatePropList() gActiveSession=%p inStr='%s'\n",
              gActiveSession, inStr);
#endif

  //if (!gActiveSession) return;

  tmp = (char *) malloc(sizeof(char) *
                        (strlen(kPropListUpdate) + 1 +
                         strlen(inStr) + 1));
  if (tmp) {
    if (gUimFD >= 0) {
      snprintf(tmp, strlen(kPropListUpdate) + 1 + strlen(inStr) + 1,
               "%s\n%s",
               kPropListUpdate, inStr);
      uim_helper_send_message(gUimFD, tmp);
    }
    free(tmp);
  }
}

void
UIMUpdatePropLabel(void *inPtr, const char *inStr)
{
  char *tmp;

#ifdef DEBUG_HELPER
  DEBUG_PRINT("UIMUpdatePropLabel() gActiveSession=%p inStr='%s'\n",
              gActiveSession, inStr);
#endif

  //if (!gActiveSession) return;

  tmp = (char *) malloc(sizeof(char) *
                        (strlen(kPropLabelUpdate) + 1 +
                         strlen(inStr) + 1));
  if (tmp) {
    if (gUimFD >= 0) {
      snprintf(tmp, strlen(kPropLabelUpdate) + 1 + strlen(inStr) + 1,
               "%s\n%s",
               kPropLabelUpdate, inStr);
      uim_helper_send_message(gUimFD, tmp);
    }
    free(tmp);
  }
}

void
UIMCheckHelper()
{
  if (gUimFD < 0) {
    gUimFD = uim_helper_init_client_fd(UIMHelperDisconnect);

#ifdef DEBUG_HELPER
    DEBUG_PRINT("UIMCheckHelper() gUimFD=%d(%p)\n", gUimFD, &gUimFD);
#endif

    if (gUimFD >= 0) {
      if (!gUimSock) {
        gSockContext.version = 0;
        //gSockContext.info = inHandle;
        gSockContext.info = NULL;
        gSockContext.retain = NULL;
        gSockContext.release = NULL;
        gSockContext.copyDescription = NULL;

        gUimSock = CFSocketCreateWithNative(kCFAllocatorDefault, gUimFD,
                                            kCFSocketReadCallBack, HelperRead,
                                            &gSockContext);
        if (!gUimSock) return;
      }

      if (!gUimRun) {
        gUimRun = CFSocketCreateRunLoopSource(kCFAllocatorDefault, gUimSock, 0);
        if (!gUimRun) {
          CFRelease(gUimSock);
          gUimSock = NULL;
          return;
        }
        CFRunLoopAddSource(CFRunLoopGetCurrent(), gUimRun, kCFRunLoopDefaultMode);
#ifdef DEBUG_HELPER
        DEBUG_PRINT("UIMCheckHelper() CFRunLoopGetCurrent()\n");
#endif
      }
    }
  }
}

static void
HelperRead(CFSocketRef sock, CFSocketCallBackType callbackType, 
           CFDataRef address, const void *data, void *info)
{
  char *tmp;

#ifdef DEBUG_HELPER
  DEBUG_PRINT("HelperRead()\n");
#endif

  uim_helper_read_proc(CFSocketGetNative(sock));
  while ((tmp = uim_helper_get_message())) {
    ParseHelperString(tmp);
    free(tmp);
  }
}

static void
ParseHelperString(const char *str)
{
  UInt32 i;

#ifdef DEBUG_HELPER
  DEBUG_PRINT("ParseHelperString() str='%s'\n", str);
#endif

  if (gActiveSession && !gDisableFocusedContext) {
    if (strncmp("prop_list_get", str, 13) == 0) {
      uim_prop_list_update(gUC);
    }
    else if (strncmp("prop_label_get", str, 14) == 0) {
      uim_prop_label_update(gUC);
    }
    else if (strncmp("prop_activate", str, 13) == 0) {
      CFMutableStringRef cfstr;
      CFArrayRef array;
      CFStringRef first;

      cfstr = CFStringCreateMutable(NULL, 0);
      CFStringAppendCString(cfstr, str, kCFStringEncodingUTF8);

      array = CFStringCreateArrayBySeparatingStrings(kCFAllocatorDefault,
                                                     cfstr, CFSTR("\n"));

      if (array && CFArrayGetCount(array) > 0 &&
          (first = CFArrayGetValueAtIndex(array, 0))) {
        CFStringRef second = CFArrayGetValueAtIndex(array, 1);
        if (second) {
#ifdef DEBUG_HELPER
          DEBUG_PRINT("ParseHelperString() first='%s' second='%s'\n",
                      CFStringGetCStringPtr(first, kCFStringEncodingUTF8),
                      CFStringGetCStringPtr(second, kCFStringEncodingUTF8));
#endif
          for (i = 0; i < gNumSession; i++) {
            if (*gSessionList[i])
              uim_prop_activate(gUC,
                                CFStringGetCStringPtr(second, kCFStringEncodingUTF8));
          }
          CFRelease(second);
        }
        CFRelease(first);
      }
    }
    else if (strncmp("focus_in", str, 8) == 0) {
      //gDisableFocusedContext = TRUE;
      /* We shouldn't do "focused_context = NULL" here, because some
         window manager has some focus related bugs. */
    }
  }
}

void
UIMHelperDisconnect()
{
#ifdef DEBUG_HELPER
  DEBUG_PRINT("UIMHelperDisconnect()\n");
#endif

  CFRunLoopRemoveSource(CFRunLoopGetCurrent(), gUimRun, kCFRunLoopDefaultMode);

  CFRelease(gUimRun);
  CFRelease(gUimSock);

  gUimRun = NULL;
  gUimSock = NULL;

  gUimFD = -1;
}

void
UIMHelperClose()
{
#ifdef DEBUG_HELPER
  DEBUG_PRINT("UIMHelperClose()\n");
#endif

  if (gUimFD >= 0)
    uim_helper_close_client_fd(gUimFD);
}

static void
AddPreeditSegment(MUIMSessionHandle inHandle, int inAttr,
                  const char *inStr)
{
  CFMutableStringRef cf_string;
  int len;
  UniCharPtr unistr;

  cf_string = CFStringCreateMutable(NULL, 0);
  CFStringAppendCString(cf_string, inStr, kCFStringEncodingUTF8);

  len = CFStringGetLength(cf_string);
  unistr = (UniCharPtr) malloc(sizeof(UniChar) * (len + 1));
  CFStringGetCharacters(cf_string, CFRangeMake(0, len), unistr);

  //DumpString("unistr", (char *) unistr, len * sizeof(UniChar));

  (*inHandle)->fSegments = realloc((*inHandle)->fSegments,
                                   sizeof(PreeditSegment) *
                                   ((*inHandle)->fSegmentCount + 1));
  (*inHandle)->fSegments[(*inHandle)->fSegmentCount].fBuffer = unistr;
  (*inHandle)->fSegments[(*inHandle)->fSegmentCount].fLength = len;
  (*inHandle)->fSegments[(*inHandle)->fSegmentCount].fAttr = inAttr;
  (*inHandle)->fSegmentCount++;

  CFRelease(cf_string);
}

void
GetPreeditSegment(PreeditSegment *inSegment, UniCharPtr *outStr,
                  UInt32 *outLen)
{
  UniCharPtr tmp = (UniCharPtr) malloc(sizeof(UniChar) * (*outLen + 1));
  memcpy(tmp, *outStr, sizeof(UniChar) * (*outLen));

#ifdef DEBUG_PREEDIT
  DEBUG_PRINT("GetPreeditSegment() outLen=%lu, fLength=%lu\n",
              *outLen, inSegment->fLength);
#endif

  (*outStr) = (UniCharPtr) realloc(*outStr, sizeof(UniChar) *
                                   ((*outLen) + inSegment->fLength + 1));

  memcpy(*outStr, tmp, sizeof(UniChar) * (*outLen));
  memcpy(&((*outStr)[(*outLen)]),
         inSegment->fBuffer, sizeof(UniChar) * inSegment->fLength);
  *outLen += inSegment->fLength;

  free(tmp);
}

void
GetPreeditString(MUIMSessionHandle inHandle, UniCharPtr *outStr,
                 UInt32 *outLen, UInt32 *outCursorPos)
{
  UniCharPtr str;
  UInt32 i, pos = 0, len = 0;

  str = (UniCharPtr) malloc(sizeof(UniChar));
  str[0] = '\0';

#ifdef DEBUG_PREEDIT
  DEBUG_PRINT("GetPreeditString() fSegmentCount=%lu\n",
              (*inHandle)->fSegmentCount);
#endif

  for (i = 0; i < (*inHandle)->fSegmentCount; i++) {
    GetPreeditSegment(&((*inHandle)->fSegments[i]), &str, &len);
    if ((*inHandle)->fSegments[i].fAttr & UPreeditAttr_Cursor) {
      pos = len;
    }
#ifdef DEBUG_PREEDIT
    DEBUG_PRINT("GetPreeditString() i=%lu len=%lu\n", i, len);
#endif
  }
#ifdef DEBUG_PREEDIT
  DEBUG_PRINT("GetPreeditString() len=%lu pos=%lu\n", len, pos);
#endif

  if (outCursorPos)
    *outCursorPos = pos;

  if (outLen)
    *outLen = len;

  if (outStr)
    *outStr = str;
  else
    free(str);
}
