<?php
/**
 * DBクラス
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2007 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: bbs_mainMemberDb.php 499 2008-04-14 07:36:58Z fishbone $
 * @link       http://www.magic3.org
 */
require_once($gEnvManager->getDbPath() . '/baseDb.php');

class bbs_mainMemberDb extends BaseDb
{
	/**
	 * 会員総数を取得
	 *
	 * @param int	$type					0=仮会員,1=正会員
	 * @return int							会員総数
	 */
	function getMemberCount($type)
	{
		$queryStr = 'SELECT * FROM bbs_member ';
		$queryStr .=  'WHERE sv_deleted = false ';// 削除されていない
		$queryStr .=     'AND sv_type = ? ';
		return $this->selectRecordCount($queryStr, array($type));
	}
	/**
	 * 会員一覧を取得
	 *
	 * @param int	$type					0=仮会員,1=正会員
	 * @param int		$limit				取得する項目数
	 * @param int		$offset				取得する先頭位置(0～)
	 * @param function	$callback			コールバック関数
	 * @return 			なし
	 */
	function getMemberList($type, $limit, $offset, $callback)
	{
		$queryStr = 'SELECT * FROM bbs_member LEFT JOIN _login_user ON sv_login_user_id = lu_id AND lu_deleted = false LEFT JOIN bbs_post_log ON sv_login_user_id = sl_user_id ';
		$queryStr .=  'WHERE sv_deleted = false ';// 削除されていない
		$queryStr .=     'AND sv_type = ? ';
		$queryStr .=  'ORDER BY sv_id limit ' . $limit . ' offset ' . $offset;
		$this->selectLoop($queryStr, array($type), $callback, null);
	}
	/**
	 * 会員情報をシリアル番号で取得
	 *
	 * @param int		$serial				シリアル番号
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getMemberBySerial($serial, &$row)
	{
		$queryStr  = 'SELECT *,login.lu_account as login_user_account,reg.lu_name as update_user_name FROM bbs_member LEFT JOIN _login_user as login ON sv_login_user_id = login.lu_id AND login.lu_deleted = false ';
		$queryStr .=   'LEFT JOIN _login_user as reg ON sv_create_user_id = reg.lu_id AND reg.lu_deleted = false ';
		$queryStr .=   'WHERE sv_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		return $ret;
	}
	/**
	 * 会員の削除
	 *
	 * @param int $serialNo			シリアルNo
	 * @return						true=成功、false=失敗
	 */
	function delMemberBySerial($serialNo)
	{
		global $gEnvManager;
		$now = date("Y/m/d H:i:s");	// 現在日時
		$userId = $gEnvManager->getCurrentUserId();	// 現在のユーザ
				
		// トランザクション開始
		$this->startTransaction();
		
		// 指定のシリアルNoのレコードが削除状態でないかチェック
		$queryStr  = 'select * from bbs_member ';
		$queryStr .=   'where sv_deleted = false ';		// 未削除
		$queryStr .=     'and sv_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serialNo), $row);
		// 存在しない場合は、既に削除されたとして終了
		if (!$ret){
			$this->endTransaction();
			return false;
		}
		
		// レコードを削除
		$queryStr  = 'UPDATE bbs_member ';
		$queryStr .=   'SET sv_deleted = true, ';	// 削除
		$queryStr .=     'sv_update_user_id = ?, ';
		$queryStr .=     'sv_update_dt = ? ';
		$queryStr .=   'WHERE sv_serial = ?';
		$this->execStatement($queryStr, array($userId, $now, $serialNo));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * 新規ユーザの追加
	 *
	 * @param int  $type			追加するユーザのタイプ(0=仮会員、1=正会員)
	 * @param string  $name			名前
	 * @param string  $account		アカウント
	 * @param string  $password		パスワード
	 * @param string  $widgetId		ウィジェットID
	 * @param int     $userId		更新者ID
	 * @param string  $now			現在日時
	 * @param int     $newId		新規に作成したログインユーザID
	 * @return						true=成功、false=失敗
	 */
	function addUser($type, $name, $account, $password, $widgetId, $userId, $now, &$newId)
	{
		// 新規IDを作成
		$newId = 1;
		$queryStr = 'select max(lu_id) as ms from _login_user ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret) $newId = $row['ms'] + 1;
		
		// ユーザ種別を設定
		if ($type == 0){
			$userType = UserInfo::USER_TYPE_TMP;		// 一時ユーザ
		} else {
			$userType = UserInfo::USER_TYPE_NORMAL;		// 一般ユーザ
		}
		$subject = 'bs,';		// BBS機能(bs)にアクセス可能
		
		// 新規レコードを追加
		$queryStr  = 'INSERT INTO _login_user (';
		$queryStr .=   'lu_id, ';
		$queryStr .=   'lu_history_index, ';
		$queryStr .=   'lu_name, ';
		$queryStr .=   'lu_account, ';
		$queryStr .=   'lu_password, ';
		$queryStr .=   'lu_user_type, ';
		$queryStr .=   'lu_assign, ';
		$queryStr .=   'lu_enable_login, ';
		$queryStr .=   'lu_widget_id, ';
		$queryStr .=   'lu_create_user_id, ';
		$queryStr .=   'lu_create_dt ';
		$queryStr .= ') VALUES (';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   'md5(?), ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?) ';
		$ret = $this->execStatement($queryStr, array($newId, 0, $name, $account, $password, $userType, $subject, 1, $widgetId, $userId, $now));
		return $ret;
	}
	/**
	 * 新規仮会員の追加
	 *
	 * @param string  $lang				言語
	 * @param int     $type				会員タイプ(0=未設定、1=個人、2=法人)
	 * @param int     $companyInfoId	法人情報ID
	 * @param int     $personInfoId		個人情報ID
	 * @param int     $loginUserId		ログインユーザID
	 * @param int     $userId		更新者ID
	 * @param string  $now			現在日時
	 * @return						true=成功、false=失敗
	 */
	function addTmpMember($lang, $type, $companyInfoId, $personInfoId, $loginUserId, $userId, $now)
	{
		// 新規レコードを追加
		$queryStr  = 'INSERT INTO shop_tmp_member (';
		$queryStr .=   'sb_language_id, ';
		$queryStr .=   'sb_type, ';
		$queryStr .=   'sb_company_info_id, ';
		$queryStr .=   'sb_person_info_id, ';
		$queryStr .=   'sb_login_user_id, ';
		$queryStr .=   'sb_create_user_id, ';
		$queryStr .=   'sb_create_dt ';
		$queryStr .= ') VALUES (';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?) ';
		$ret = $this->execStatement($queryStr, array($lang, $type, $companyInfoId, $personInfoId, $loginUserId, $userId, $now));
		return $ret;
	}
	/**
	 * 会員の更新、新規追加
	 *
	 * @param int     $serial			更新する会員情報のシリアル番号(0のときは新規追加)
	 * @param string  $lang				言語
	 * @param int     $type				会員タイプ(0=仮会員、1=正会員)
	 * @param int     $loginUserId		ログインユーザID
	 * @param string  $name				名前
	 * @param string  $group			所属グループ
	 * @param string  $url				URL
	 * @param string  $signature		署名
	 * @param bool    $recvMailnews		新着情報を受信するかどうか
	 * @param int     $userId		更新者ID
	 * @param string  $now			現在日時
	 * @param int     $newSerial	新規シリアル番号
	 * @return						true=成功、false=失敗
	 */
	function updateMember($serial, $lang, $type, $loginUserId, $name, $group, $url, $signature, $recvMailnews, $userId, $now, &$newSerial)
	{
		if ($serial == 0){
			// 新規IDを作成
			$newId = 1;
			$queryStr = 'select max(sv_id) as ms from bbs_member ';
			$ret = $this->selectRecord($queryStr, array(), $maxRow);
			if ($ret) $newId = $maxRow['ms'] + 1;
		
			$historyIndex = 0;
			$registDt = $now;		// 登録日時
		} else {		// データ更新の場合
			// 前レコードの削除状態チェック
			$queryStr = 'SELECT * FROM bbs_member ';
			$queryStr .=  'WHERE sv_serial = ? ';
			$ret = $this->selectRecord($queryStr, array($serial), $row);
			if (!$ret) return;
			
			if ($row['sv_deleted']) return false;		// 削除されていれば終了
			$historyIndex = $row['sv_history_index'] + 1;
			$newId = $row['sv_id'];
			$registDt = $row['sv_regist_dt'];		// 登録日時は更新しない
			
			// レコードを削除
			$queryStr  = 'UPDATE bbs_member ';
			$queryStr .=   'SET sv_deleted = true, ';
			$queryStr .=   'sv_update_user_id = ?, ';
			$queryStr .=   'sv_update_dt = ? ';			
			$queryStr .=   'WHERE sv_serial = ? ';
			$this->execStatement($queryStr, array($userId, $now, $serial));
		}
		// 新規レコードを追加
		$queryStr  = 'INSERT INTO bbs_member (';
		$queryStr .=   'sv_id, ';
		$queryStr .=   'sv_history_index, ';
		$queryStr .=   'sv_language_id, ';
		$queryStr .=   'sv_type, ';
		$queryStr .=   'sv_login_user_id, ';
		$queryStr .=   'sv_name, ';
		$queryStr .=   'sv_group, ';
		$queryStr .=   'sv_url, ';
		$queryStr .=   'sv_signature, ';
		$queryStr .=   'sv_regist_dt, ';
		$queryStr .=   'sv_recv_mailnews, ';
		$queryStr .=   'sv_create_user_id, ';
		$queryStr .=   'sv_create_dt ';
		$queryStr .= ') VALUES (';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?, ';
		$queryStr .=   '?) ';
		$ret = $this->execStatement($queryStr, array($newId, $historyIndex, $lang, $type, $loginUserId, $name, $group, $url, $signature, $registDt, $recvMailnews, $userId, $now));

		// 新規のシリアル番号取得
		$queryStr = 'select max(sv_serial) as ns from bbs_member ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret) $newSerial = $row['ns'];
		return true;
	}
	/**
	 * 会員名が存在するかチェック
	 *
	 * @param string $name		名前
	 * @return					true=存在する、false=存在しない
	 */
	function isExistsMemberName($name)
	{
		$queryStr = 'SELECT * from bbs_member ';
		$queryStr .=  'WHERE sv_name = ? ';
		$queryStr .=    'AND sv_deleted = false';
		return $this->isRecordExists($queryStr, array($name));
	}
	/**
	 * グループ名称を取得
	 *
	 * @param string $groupId			グループID
	 * @param string $langId    	表示言語ID
	 * @return string					グループ名称
	 */
	function getGroupName($groupId, $langId)
	{
		$retValue = '';
		$queryStr = 'SELECT * from bbs_group ';
		$queryStr .=  'WHERE sg_id = ? ';
		$queryStr .=    'AND sg_language_id = ? ';
		$ret = $this->selectRecord($queryStr, array($groupId, $langId), $row);
		if ($ret) $retValue = $row['sg_name'];
		return $retValue;
	}
	/**
	 * ユーザグループ情報を取得
	 *
	 * @param string	$lang				言語
	 * @param array	$rows					取得情報
	 * @return bool							true=取得、false=取得せず
	 */
	function getAllGroupInfo($lang, &$rows)
	{
		$queryStr = 'SELECT * FROM bbs_group ';
		$queryStr .=  'WHERE sg_language_id = ? ';
		$queryStr .=    'AND sg_deleted = false ';		// 削除されていない
		$queryStr .=  'ORDER BY sg_sort_order';
		$retValue = $this->selectRecords($queryStr, array($lang), $rows);
		return $retValue;
	}
	/**
	 * 会員情報をユーザIDで取得
	 *
	 * @param int		$userId				ユーザID
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getMemberByUserId($userId, &$row)
	{
		$queryStr  = 'SELECT *,login.lu_account as login_user_account,reg.lu_name as update_user_name FROM bbs_member LEFT JOIN _login_user as login ON sv_login_user_id = login.lu_id AND login.lu_deleted = false ';
		$queryStr .=   'LEFT JOIN _login_user as reg ON sv_create_user_id = reg.lu_id AND reg.lu_deleted = false ';
		$queryStr .=   'WHERE sv_login_user_id = ? AND sv_deleted = false';
		$ret = $this->selectRecord($queryStr, array($userId), $row);
		return $ret;
	}
}
?>
