#!/bin/env python
# coding: utf-8

"""\
Control Buffalo RemoteStation PC-OP-RS1

This Python script enable you to control RemoteStation, such as recording
signal of remote control equipment and transmitting that signal from
RemoteStation.
"""

__author__ = 'M.Decimal at Specurio Studio.'
__version__ = '0.1'
__date__ = '2009-12-08'
__credits__ = """\
RemoteStation PC-OP-RS1 is a product for trasmitting ir signals made by
Japanese peripheral equipment vender Buffalo,Inc.
portalocker is a module posted at http://code.activestate.com/recipes/65203/)
by Jonathan Feinberg.
I used pexpect module coded by Noah Spurrier (http://www.noah.org/wiki/Pexpect)
for getting info from privilege protected file. 
This script was coded on Pydev, powerful python IDE by Fabio Zadrozny, and now
becoming Apatana open source product.
Of course, Pydev is plug-in for, THE GREATE IDE, Eclipse provided by Eclipse
Foundation.
"""

import logging
import sys
import os
import time
import codecs
import gettext
import optparse
import platform
import re

from ConfigParser import SafeConfigParser, NoSectionError, NoOptionError
from StringIO     import StringIO

import portalocker

import rsdevice
from rsdevice.exception import *

def _acn_dirname(path, gen=0):
    for i in range(0, gen + 1) :
        path = os.path.dirname(path)
    return path
os.path.ancdirname = _acn_dirname

# decision file whether script is in develop environment
IN_DEVELOP_ENV = False
DEVELOP_TOP_DIR = os.path.ancdirname(__file__, 1)
if os.path.exists(os.path.join(DEVELOP_TOP_DIR, 'setup.py')) :
    IN_DEVELOP_ENV = True

FILE_ENCODING = 'utf-8'
TERMINAL_ENCODING = 'utf-8'

#======================================================================
# L10N
#
# setting gettext
if IN_DEVELOP_ENV :
    _ = gettext.translation('managers',
                            os.path.join(DEVELOP_TOP_DIR, 'locale'),
                            ).ugettext
else :
    _ = gettext.translation('managers', fallback=True).ugettext


#======================================================================
# Logging SETTINGS
#
LOGGER = logging.getLogger('managers')
LOGGER.addHandler(logging.StreamHandler())
LOGGER.setLevel(logging.WARNING)

# message level
logging.MESSAGE = logging.CRITICAL + 100
logging.addLevelName(logging.MESSAGE, 'MESSAGE')
LOGGER.message = lambda msg : LOGGER.log(logging.MESSAGE, msg)

# add logging method that use existing writing out funciton
# usage : logging.stream( lvl, func )
#    lvl        log level
#    func       writing function having argment for file (like) object
def _logstream(lvl, func):
    buf = StringIO()
    buf = codecs.getwriter(TERMINAL_ENCODING)(buf)
    func(buf)
    LOGGER.log(lvl, buf.getvalue())
    buf.close()
LOGGER.stream = _logstream


#======================================================================
# FILE SETTINS
#
# system default preference and signal data file name
DEFAULT_PREF_FILE = 'managers.conf'
DEFAULT_SGNL_FILE = 'managers.sig'

# user preference and signal data file name
USER_PREF_FILE = 'user.conf'
USER_SGNL_FILE = 'user.sig'

# default preference and signals data file path
if IN_DEVELOP_ENV :
    SYSTEM_PREF_DIR = os.path.join(DEVELOP_TOP_DIR, 'etc')
else :
    if platform.system().lower() == 'linux' :
        if __file__.startswith('/usr/local') :
            _INSTALL_PREFIX = '/usr/local'
        else :
            _INSTALL_PREFIX = '/usr'
        SYSTEM_PREF_DIR = os.path.join( _INSTALL_PREFIX, 'etc')
    elif platform.system().lower() == 'windows' :
        SYSTEM_PREF_DIR = os.path.dirname( __file__ )

DEFAULT_PREF_PATH = os.path.join(SYSTEM_PREF_DIR, DEFAULT_PREF_FILE)
DEFAULT_SGNL_PATH = os.path.join(SYSTEM_PREF_DIR, DEFAULT_SGNL_FILE)

# user preference and signals data file path
if platform.system().lower() == 'linux' :
    USER_PREF_DIR = os.path.join(os.path.expanduser('~'), '.managers')
elif platform.system().lower() == 'windows' :
    USER_PREF_DIR = os.path.join(os.path.expandvars('$APPDATA'), 'managers')
USER_PREF_PATH = os.path.join(USER_PREF_DIR, USER_PREF_FILE)
USER_SGNL_PATH = os.path.join(USER_PREF_DIR, USER_SGNL_FILE)

# signal data option name and its format in signal data file
SGNL_OPT_PREFIX = 'sig_'
SGNL_OPT_FORMAT = r'%s%%03d' % SGNL_OPT_PREFIX

# signal name and its format in option value
SGNL_UNNAMED_PREFIX = 'untitled'
SGNL_UNNAMED_FORMAT = r'%s%%03d' % SGNL_UNNAMED_PREFIX


#======================================================================
# ConfigParser SETTINGS
#
# Adding force_set method for SafeConfigParser to force to set options value
# usage : inst.force_set( section, option, value )
#    section    section name
#    optoin     options name
#    value      value to set to option
def _forceset(self, section, option, value):
    if not self.has_section(section) :
        self.add_section(section)
    self.set(section, option, unicode(value))
SafeConfigParser.force_set = _forceset
# Adding force_get method for SafeConfigParser to get option value without
# exception even if the section does not have that section
# usage : inst.force_get( section, option, default )
#    section    section name
#    option     options name
#    default    return value if the option does not exist
def _forceget(self, section, option, default=None):
    try :
        return self.get(section, option)
    except (NoSectionError, NoOptionError) :
        return default
SafeConfigParser.force_get = _forceget
# Adding force_items method for SafeConfigParser to get section's items
# without exception even if the section have no items 
# usage : inst.force_get( section, option, default )
#    section    section name
#    option     options name
#    default    return value if the option does not exist
def _forceitems(self, section):
    try :
        return self.items(section)
    except (NoSectionError) :
        return []
SafeConfigParser.force_items = _forceitems

def _write(self, fp):
    if self._defaults:
        fp.write("[%s]\n" % DEFAULTSECT)
        for (key, value) in self._defaults.items():
            fp.write("%s = %s\n" % (key, unicode(value).replace('\n', '\n\t')))
        fp.write("\n")
    for section in self._sections:
        fp.write("[%s]\n" % section)
        for (key, value) in self._sections[section].items():
            if key != "__name__":
                fp.write("%s = %s\n" %
                         (key, unicode(value).replace('\n', '\n\t')))
        fp.write("\n")
SafeConfigParser.write = _write

# utf-8 is default codec for output
sys.stdout = codecs.lookup(TERMINAL_ENCODING)[-1](sys.stdout)
sys.stderr = codecs.lookup(TERMINAL_ENCODING)[-1](sys.stderr)

def parse_option():
    """ Parse all option for this script.  
    """
    
    usage = '''
       %prog find
       %prog list [options] [setname]
       %prog delete [options] setname[.signalname]
       %prog led [options] [repeat [interval]]
       %prog capture [options] [setname[.signalname] [note]]
       %prog emit [options] [channel [setname.signalname]]
       
%prog is software for controling Buffalo RemoteStation PC-OP-RS1(describe as
 RS below).
There are six kind of subcommand.

    find
        search and report all RS connected your computer.  
        result is like this (on Linux).  
        
            0: /dev/ttyUSB0
            1: /dev/ttyUSB5
        
        first is identify number, and second is device name. you can use
        identify number for "auto_select" in config file that specify which
        RS you will control. Device name can also use for -d (--device)
        option.  
        
    list
        lists set name and signal name you can use on this script. Set name
        is group of signal name, like directory groups file. And Signal name
        is actual signal identify name. It looks like this.
        
            TV.switchOn
            TV.channel8
        
        If set name is specified as argument, it lists name only belongs to
        that set name. Otherwise, all name is listed.
        
    delete
        delete specified set or signal. If you specified only for set name,
        managers will delete all signals belongs to the set and set itself.
        
    led
        lights on RS's access LED. You can specify how many time the led light
        on (repeat),and how many seconds the LED turn off (interval).
        Unfortunately, there are no way to specify how many second the LED
        turned on.  
    
    capture
        memorize remote control signal with the name. If signalname was
        omitted, managers decide signal name for captured signal.
        you can memorize notation for this signal with note argument.
        If all arguments is omitted, managers writes out signal to stdout.
        
    emit
        emit specified ir signal from RS on channel. Generally, emitter with
        yellow tape on port A is channel 1, without yellow tape on Port B is
        channel 2, yellow taped on channel is 3, and rest is 4.
        if channel was omitted, managers try to emit signal from all channel.
        And if name is omitted, signal will be taken from stdin.

'''

    #==========================================================================
    # Pre-process
    #==========================================================================
    parser = optparse.OptionParser(usage=usage)
    
    #==========================================================================
    # Parser setting
    #==========================================================================
    parser.add_option('-c', '--config-file', action='store',       type='string', dest='configfile', default=None, help=_('configure file to specify behavior of this command'))
    parser.add_option('-s', '--signal-file', action='store',       type='string', dest='signalsfile', default=None, help=_('signal file storing actual ir signal data'))
    parser.add_option('-d', '--device',      action='store',       type='string', dest='device',     default=None, help=_('device name for RemoteStation'))
    parser.add_option('-v', '--verbose',     action='store_const', const='1',     dest='verbose',    default='0',  help=_('switch for verbose output'))

    (options, args) = parser.parse_args()
    args = [unicode(item, TERMINAL_ENCODING) for item in args]
    if ('configfile' in dir(options)) and (options.configfile is not None) :
        options.configfile = unicode(options.configfile, TERMINAL_ENCODING)
    if ('signalsfile' in dir(options)) and (options.signalsfile is not None) :
        options.signalsfile = unicode(options.signalsfile, TERMINAL_ENCODING)
    if ('device' in dir(options)) and (options.device is not None) :
        options.device = unicode(options.device, TERMINAL_ENCODING)
    if ('verbose' in dir(options)) and (options.verbose is not None) :
        options.verbose = unicode(options.verbose, TERMINAL_ENCODING)
    
    
    #==========================================================================
    # Argument check and set doctype variable for later use.
    #==========================================================================
    args = exlist(args)
    if (args.get(0) is None) or\
       (args.get(0) not in ['find', 'list', 'delete', 'led', 'capture', 'emit']) :
        parser.error(_('this command need to specify subcommand just after command'))
        sys.exit(1)
    else :
        subcommand = args.pop(0)
    argdic = dict()
    
    reg_float = re.compile(r'\d*.?\d+')
    reg_setname_only = re.compile(r'^[\w][\w\+\-]*$')
    reg_setname_and_signalname = re.compile(r'^(?P<setname>[\w][\w\+\-]*)\.(?P<signalname>[\w][\w\+\-]*)$')
    reg_setname_or_signalname = re.compile(r'^(?P<setname>[\w][\w\+\-]*)(\.(?P<signalname>[\w][\w\+\-]*))?$')
    
    # subcommand "find" -------------------------------------------------------
    if subcommand == 'find' :
        pass
    
    # subcommand "list" -------------------------------------------------------
    elif subcommand == 'list' :
        if args.get(0) is None :
            argdic['setname'] = None
        elif reg_setname_only.search(args.get(0)) :
            argdic['setname'] = args.get(0)
        else :
            parser.print_usage(sys.stderr)
            LOGGER.error(_('setname must be consist from alphanumerical and "+" and "-" and "_" characters'))
            sys.exit(1)
    
    # subcommand "delete" -----------------------------------------------------
    elif subcommand == 'delete' :
        if args.get(0) is None :
            parser.print_usage(sys.stderr)
            LOGGER.error(_('at least setname must be specified'))
            sys.exit(1)
        else :
            match = reg_setname_or_signalname.search(args.get(0))
            if match :
                argdic['setname'] = match.group('setname')
                argdic['signalname'] = match.group('signalname')
            else :
                parser.print_usage(sys.stderr)
                LOGGER.error(_('setname and signalname must be consist from alphanumerical and "+" and "-" and "_" characters'))
                sys.exit(1)
            
    # subcommand "led" --------------------------------------------------------
    elif subcommand == 'led' :
        if args.get(0) is None :
            argdic['repeat'] = 1
        elif args.get(0).isdigit() and int(args.get(0)) > 0 :
            argdic['repeat'] = int(args.get(0))
        else :
            parser.print_usage(sys.stderr)
            LOGGER.error(_('repeat must be integer value greater than 1'))
            sys.exit(1)

        if args.get(1) is None :
            argdic['interval'] = 0.5
        elif re.compile(r'^\d*.?\d+$').search(args.get(1)) :
            argdic['interval'] = float(args.get(1))
        else :
            parser.print_usage(sys.stderr)
            LOGGER.error(_('interval must be float value greater than 0'))
            sys.exit(1)
    
    # subcommand "capture" ----------------------------------------------------
    elif subcommand == 'capture' :
        if args.get(0) is None :
            argdic['setname'] = None
            argdic['signalname'] = None
        else :
            match = reg_setname_or_signalname.search(args.get(0))
            if match :
                argdic['setname'] = match.group('setname')
                argdic['signalname'] = match.group('signalname')
            else :
                parser.print_usage(sys.stderr)
                LOGGER.error(_('setname and signalname must be consist from alphanumerical and "+" and "-" and "_" characters'))
                sys.exit(1)

        argdic['note'] = args.get(1)
    
    # subcommand "emit" -------------------------------------------------------
    elif subcommand == 'emit' :
        if args.get(0) is None :
            argdic['channel'] = 1
        elif args.get(0).isdigit() and\
             int(args.get(0)) >= 0 and int(args.get(0)) <= 4 :
            argdic['channel'] = int(args.get(0))
        else :
            parser.print_usage(sys.stderr)
            LOGGER.error(_('channel must be integer between 0 - 4'))
            sys.exit(1)

        if args.get(1) is None :
            argdic['setname'] = None
            argdic['signalname'] = None
        else :
            match = reg_setname_and_signalname.search(args.get(1))
            if match :
                argdic['setname'] = match.group('setname')
                argdic['signalname'] = match.group('signalname')
            else :
                parser.print_usage(sys.stderr)
                LOGGER.error(_('setname and signalname must be consist from alphanumerical and "+" and "-" and "_" characters'))
                sys.exit(1)
    
    
    return (subcommand, options, argdic)


def build_config( default, user=None, override=None ):
    ''' Build configure data deciding this scripts behavior
    
    build_config read 3 types of configuration file, and override that setting
    with options data.
    
    Arguments :
        default (str)
            Default configuration file path.see get_config()
            
        user (str or None, or can be omitted)
            User configuration file path. see get_config()
            
        override (object described below, or None and can be omitted )
            data for overriding configuration from file. this object must be
            first element which got by OptionParser.parse_args().
            
    Return :
        ConfigParser.SafeConfigParser object
    '''
    
    # Getting configration from file
    conf = get_config( default, user, override.configfile )
    
    # Overriding with options
    if override:
        if ('configfile' in dir(override)) and override.configfile:
            conf.force_set('managers', 'configfile', override.configfile)
        if 'signalsfile' in dir(override) and override.signalsfile:
            conf.force_set('managers', 'signalsfile', override.signalsfile)
        if 'verbose' in dir(override) and override.verbose:
            conf.force_set('managers', 'verbose', override.verbose)
        if 'device' in dir(override) and override.device:
            conf.force_set('device', 'name', override.device)

    return conf


def build_signals( default, user=None, onetime=None ):
    ''' Build signals data including all ir signals
    
    build_signals read 3 types of signal file. Actually signals data file is
    same format with configration file, so this function call same function
    as "build_config()" for reading signals dataf file.
    
    Arguments :
        default (str)
            Default signal file path. see get_config()
            
        user (str or None, or can be omitted)
            User configuration file path. see get_config()
            
        onetime (str or None, or can be omitted)
            One time configuration file path. see get_config()
            
    Return :
        ConfigParser.SafeConfigParser object
    '''

    # Getting configration from file
    signals = get_config(default, user, onetime)

    return signals


def save_signals(signals, path):
    ''' Writing out signals ConfigParser object to file
    
    Arguments :
        signals ( ConfigParser )
            ConfigParser object to write out.
            
        path ( str )
            file path to output
    
    Return :
        None
    '''
    
    # Pre-process for if the path is not exists.
    LOGGER.info(_('Saving signals to file %s') % path )
    fs = file(path, 'a')
    fs.close()
    
    # Writing out.
    fs = file(path, 'rb+')
    portalocker.lock(fs, portalocker.LOCK_EX)
    fs = codecs.getwriter(FILE_ENCODING)(fs)
    fs.seek(0)
    fs.truncate(0)
    signals.write(fs)
    fs.close()

def add_signal(signals, signal, section, name=None, note=None) :
    ''' Adding one signal data to signals 
    
    Arguments :
        signals (ConfigParser)
            ConfigParser object to write out.
            
        section (str)
            Section name which the signal will belongs to
            
        name (str)
            Signal name for this signal
        
        signal (str)
            Actual signal data (hex string in length 240)
        
        note (unicode)
            Notation for signal
    
    Return :
        Signals data(SafeConfigParser object)
    '''
    
    # Add section if it does not exists
    if not signals.has_section(section) :
        signals.add_section(section)
    
    option = get_option_by_name(signals, section, name)
    if name and option :
        signals.remove_option(section, option)
        
    # Deciding signal name if it was not specified
    if not name :
        num = get_minimum_unnamed_signal_number(signals, section)
        name = SGNL_UNNAMED_FORMAT % num
    
    # Deciding config option name
    num = get_minimum_option_number(signals, section)
    option = SGNL_OPT_FORMAT % num
    
    if not note :
        note = ''
        
    # Data string for signal in option
    data = '%s,%f,%s,%s' % (name, time.time(), signal, note)

    signals.set(section, option, data)
    return signals

def delete_signal(signals, section, name=None):
    '''Delete one signal data from signals
    
    if name is None, this function will delete section itself.
    Argument :
        signals (ConfigParser)
            ConfigPaser object including all singlas
        
        section (str)
            Section name which the signal belongs to
        
        name (str)
            Signal name you want to delete
    
    Return :
        Signals data(SafeConfigParser object)
    '''
    
    if name :
        option = get_option_by_name(signals, section, name)
        signals.remove_option(section, option)
    else :
        signals.remove_section(section)
    
    return signals

def get_signal_by_name(signals, section, name) :
    '''Getting single signal data from signals by signal name 
    
    Argument :
        signals (ConfigParser)
            ConfigPaser object including all singlas
        
        section (str)
            Section name which the signal belongs to
        
        name (str)
            Signal name you want to get
    
    Return :
        exlist including name, added timestamp, signal, and notation as
        its element value.
    '''
    
    result = exlist()
    
    option = get_option_by_name(signals, section, name)
    return get_signal_by_option(signals, section, option)


def get_signal_by_option(signals, section, option):
    '''Getting single signal data from signals by option name
    
    Argument :
        signals (ConfigParser)
            ConfigPaser object including all singlas
        
        section (str)
            Section name which the option belongs to
        
        option (str)
            Option name you want to get
    
    Return :
        exlist including name, added timestamp, signal, and notation as
        its element value.
    '''
    
    return exstr(signals.get(section, option)).split_to(',', 3)[2]


def get_option_by_name( signals, section, name ):
    '''Getting options name having named signal.

    Argument :
        signals (ConfigParser)
            ConfigPaser object including all singlas
        
        section (str)
            Section name which the signal belongs to
        
        name (str)
            Signal name you want to get
    
    Return :
        exlist including name, added timestamp, signal, and notation as
        its element value.
    '''
    
    result = ''
    
    for option in signals.options(section):
        data = exstr(signals.get(section, option))
        (inname, timestamp, signal, note) = data.split_to(',', 3, None)
        if inname == name :
            result = option
            break

    return result

def get_minimum_unnamed_signal_number(signals, section):
    '''Getting unused number in signal name

    Argument :
        signals (ConfigParser)
            ConfigPaser object including all singlas
        
        section (str)
            Section name to search
    
    Return :
        Integer value
    '''
    
    number = 0
    
    # Get all singal names in section
    options = get_option_list_only_for_signal(signals, section)
    names = [exstr(signals.get(section, option)).split_to(',', 1)[0] for option in options]
    
    # Search unused number for signal name
    for i in range(0, len(names) + 1):
        if SGNL_UNNAMED_FORMAT % i not in names :
            number = i
            break

    return number

def get_minimum_option_number(signals, section):
    '''Getting unused number in option name

    Argument :
        signals (ConfigParser)
            ConfigPaser object including all singlas
        
        section (str)
            Section name to search
    
    Return :
        Integer value
    '''
    
    number = 0
    
    # Get all option names only for signal data in section
    options = get_option_list_only_for_signal(signals, section)
    
    # Search unused number for option name
    for i in range(0, len(options) + 1) :
        if SGNL_OPT_FORMAT % i not in options :
            number = i
            break ;

    return number

def get_option_list_only_for_signal(signals, section):
    '''Getting option name list only for signal data

    Argument :
        signals (ConfigParser)
            ConfigPaser object including all singlas
        
        section (str)
            Section name to search
    
    Return :
        exlist of str option name
    '''
    
    return exlist([item for item in signals.options(section) if item.startswith(SGNL_OPT_PREFIX)])

def get_config(default, user=None, onetime=None):
    ''' Read 3 types of configuration file
    
    get_config read application or user or one-time configuration file, and
    returns SafeConfigParser object.
    
    Arguments :
        default (str)
            Application configuration file path.This is for default settings.
            
        user (str or None, or can be omitted)
            User configuration file path. If "onetime" is not passed,and if
            the path for argument really exists, this funciton reads
            configuration from this file
            
        onetime (str or None, or can be omitted)
            One time configuration file path. if this argument is passed, and
            the path for argument really exists, this funciton ignore user"
            argument, and reads configuration from this file.
    
    Return :
        ConfigParser.SafeConfigParser object
    '''
    
    conf = SafeConfigParser()
    
    #===========================================================================
    # Application Default configration file
    #===========================================================================
    if os.path.exists(default):
        LOGGER.info(_('Reading application default configuration...'))
        try :
            conf = read_config(conf, default)
        except IOError :
            LOGGER.warning(_('Can not read application default configuration file.'))
    
    #===========================================================================
    # User configuration and Default configuration file
    #===========================================================================
    if onetime:
        LOGGER.info(_('Reading one-time configuration...'))
        conf = read_config(conf, onetime)
    else :
        LOGGER.info(_('Reading user configuration...'))
        try :
            conf = read_config(conf, user)
        except IOError :
            LOGGER.warning(_('Can not read user configuration file.'))

    return conf

def read_config(config, path):
    fp = file(path, 'rb')
    fp = codecs.getreader(FILE_ENCODING)(fp)
    config.readfp(fp)
    fp.close()
    
    return config

def items_to_dict(items):
    '''Convert items type data to dictionary
    
    Arguments :
        items (list)
            List of (key, value) pair value.
            
    Return :
        Dictionary converted.
    '''
    
    data = dict()
    for (key, value) in items :
        data[key] = value

    return data

def led(rs, repeat=1, interval=0.5):
    '''Lights the RS access LED
    
    Arguments :
        rs (rsdevice.Device object)
            Device object for RS.
        
        repeat (int, or can be omitted)
            How many time LED light out. Defaults to 1
            
        interval (float, or can be omitted)
            How many second it there between LED light out
    
    Return :
        None
    '''
    if (not isinstance(repeat, int)) or (repeat < 1) :
        repeat = 1
    if (not isinstance(interval, float)) or (interval < 0) :
        interval = 0.5
    rs.led(repeat, interval)

def capture(rs):
    '''Captures signals on RS ir reciever
    
    Arguments :
        rs (rsdevice.Device object)
            Device object for RS.
                
    Return :
        240 bytes hex string
    '''
    
    rs.led(repeat=1)
    capsignal = rs.capture()
    rs.led(repeat=3, interval=0.5)

    return capsignal

def emit(rs, signal, channel=None):
    '''Captures signals on RS ir reciever
    
    Arguments :
        rs (rsdevice.Device object)
            Device object for RS.
            
        signal (str)
            Actual signal to emit, this must be string in 240 bytes hex string.
        
        channel (int or None)
            Channel number to emit signal. it should be 1 - 4 integer value,
            but it also set to 0 that will emit signal on all channel 
                
    Return :
        None
    '''
    
    rs.emit(signal, channel)


def main() :
    '''Main function for script execution
    '''
    # Script argument parsing
    (subcommand, options, argdic) = parse_option()

    # Configuration and signals data
    conf = build_config(DEFAULT_PREF_PATH, USER_PREF_PATH, options )
    signals = build_signals(DEFAULT_SGNL_PATH,
                            USER_SGNL_PATH,
                            conf.force_get('managers','signalsfile'))
    
    if conf.force_get('managers', 'signalsfile') is None :
        conf.force_set('managers', 'signalsfile', USER_SGNL_PATH)
    if conf.force_get('managers', 'verbose', '').lower() in ['1', 'on', 'true'] :
        LOGGER.setLevel(logging.DEBUG)
    
    LOGGER.info(_('Application preference file is set to %s') % DEFAULT_PREF_PATH )
    LOGGER.info(_('Application signals data file is set to %s') % DEFAULT_SGNL_PATH )
    LOGGER.info(_('User preference file is set to %s') % USER_PREF_PATH)
    LOGGER.info(_('User signals data file is set to %s') % USER_SGNL_PATH)
    
    LOGGER.debug(_('Final configuration data ------------------------------'))
    LOGGER.stream(logging.DEBUG, conf.write)
    LOGGER.debug(_('Final signals data ------------------------------------'))
    LOGGER.stream(logging.DEBUG, signals.write)

    #===========================================================================
    # Processing "find" subcommand
    #===========================================================================
    if subcommand == 'find' :
        LOGGER.info( _('Processing device find mode ...') )
        result = rsdevice.Device.get_devices()
        if len(result) :
            for i in range(0, len(result)) :
                print('%d: %s' % (i, result[i]))
        else :
            LOGGER.warning(_('RemoteStation was not found.'))
        sys.exit(0)

    #===========================================================================
    # Processing "list" subcommand
    #===========================================================================
    if subcommand == 'list' :
        LOGGER.info(_('Processing signal list mode ...'))
        LOGGER.debug('setname: %s' % argdic.get('setname'))
        #-----------------------------------------------------------------------
        # Getting set name list to display
        #     if set name is supplied in argument, set name list will have only 
        #     one elemet with the set name.otherwise, it will have all set name
        #     in signals data as section name.
        #-----------------------------------------------------------------------
        # Argument for "setname" was supplied.
        sections = exlist()
        if argdic.get( 'setname' ) :
            # Signals data have the section same as "setname".
            if signals.has_section(argdic.get('setname')) :
                sections = exlist([argdic.get('setname')])
            
            # Signals data does not have the section.
            else :
                LOGGER.error(_('There are no set name "%s"' % argdic.get('setname')))
                sys.exit(1)
        
        # No "setname"
        else :
            LOGGER.debug(_('setname was not specified'))
            sections = signals.sections()

        LOGGER.debug(_('Setname list to display: %s') % str(sections))
        
        #-----------------------------------------------------------------------
        # Print all set name and signal name for set name list.
        #-----------------------------------------------------------------------
        for section in sections :
            sys.stdout.write('[%s]\n' % section)
            for option in signals.options(section) :
                option_data = exstr(signals.get(section, option))
                (name, note) = option_data.split_to(',', 3, None).multi_get(0, 3)
                sys.stdout.write('\t%12s.%-12s : %s\n' % (section, name, note))
        sys.exit(0)

    #===========================================================================
    # Processing "delete" subcommand
    #===========================================================================
    if subcommand == 'delete' :
        LOGGER.info(_('Processing signal delete mode ...'))
        LOGGER.debug('setname: %s' % argdic.get('setname'))
        LOGGER.debug('signalname: %s' % argdic.get('signalname'))
        
        #-----------------------------------------------------------------------
        # Delete signal from signals data
        #-----------------------------------------------------------------------
        try :
            delete_signal(signals, argdic.get('setname'), argdic.get('signalname'))
        except NoSectionError, inst :
            LOGGER.error(_('Error: Set name "%s" was not found in signal data') % argdic['setname'])
            sys.exit(1)
        except NoOptionError, inst :
            LOGGER.error(_('Error: Signal name "%s" was not found in signal data') % argdic['signalname'])
            sys.exit(1)
        
        #-----------------------------------------------------------------------
        # Save signals data
        #-----------------------------------------------------------------------
        try :    
            save_signals(signals, conf.force_get('managers', 'signalsfile'))
        except IOError, inst :
            LOGGER.error(_('Error: Can not write out to signals data file (%s)') % unicode(inst))
            sys.exit(1)
            
        sys.exit(0)
        
    #===========================================================================
    # Getting device object for RemoteStation
    #===========================================================================
    target_device = conf.force_get('device', 'name', '')
    auto_select = int(conf.force_get('device', 'autoselect', '0'))
    serial_conf = items_to_dict(conf.force_items('serial'))
    protocol_conf = items_to_dict(conf.force_items('protocol'))
    try :
        rs = rsdevice.Device(target_device, auto_select, serial_conf, protocol_conf)
    except SetupDeviceError, inst :
        LOGGER.error(_('Error: Can not setting up device (%s)') % unicode(inst))
        sys.exit(1)
    except Exception, inst :
        raise
        sys.exit(1)

    #===========================================================================
    # Processing "led" subcommand
    #===========================================================================
    if subcommand == 'led' :
        LOGGER.info(_('Processing LED light on mode ...'))
        try :
            led(rs, argdic.get('repeat'), argdic.get('interval'))
        except TimeoutError, inst :
            LOGGER.error(_('Error: Access timeouted. RemoteStation did not reply for certain term'))
            sys.exit(1)
        except UnexpectedReplyError, inst :
            LOGGER.error(_('Error: RemoteStation replied unexpected value (%s)' % unicode(inst)))
            sys.exit(1)
        except :
            raise

    #===========================================================================
    # Processing "capture" subcommand
    #===========================================================================
    elif subcommand == 'capture' :
        LOGGER.info(_('Processing capture signal mode ...'))
        LOGGER.debug('setname: %s' % argdic.get('setname'))
        LOGGER.debug('signalname: %s' % argdic.get('signalname'))
        LOGGER.debug('note: %s' % argdic.get('note'))
        
        #----------------------------------------------------------------------- 
        # Capturing signal
        #----------------------------------------------------------------------- 
        LOGGER.message(_('Push your remote control equipment button toward RemoteStation\'s IR reciever.'))
        try :
            captured_signal = capture(rs)
        except TimeoutError, inst :
            LOGGER.error(_('Error: Access timeouted. RemoteStation did not reply for certain term'))
            sys.exit(1)
        except UnexpectedReplyError, inst :
            LOGGER.error(_('Error: RemoteStation replied unexpected value (%s)' % unicode(inst)))
            sys.exit(1)
        except :
            raise
        LOGGER.message(_('ManageRS got signal.'))
        
        #----------------------------------------------------------------------- 
        # If set name was specified, save signal to the signals file.
        #-----------------------------------------------------------------------
        if argdic.get('setname') :
            signals = add_signal(signals, captured_signal, argdic.get('setname'),
                                 argdic.get('signalname'), argdic.get('note'))
            try :
                save_signals(signals, conf.force_get('managers', 'signalsfile'))
            except IOError, inst :
                LOGGER.error(_('Error: Can not write out to signals data file (%s)') % unicode(inst))
                
        #----------------------------------------------------------------------- 
        # If set name was not specified, output captured signal to stdout
        #----------------------------------------------------------------------- 
        else :
            print(captured_signal)

    #===========================================================================
    # Processing "emit" subcommand
    #===========================================================================
    elif subcommand == 'emit' :
        LOGGER.info(_('Processing emit signal mode ...'))
        LOGGER.debug('channel: %s' % argdic.get('channel'))
        LOGGER.debug('setname: %s' % argdic.get('setname'))
        LOGGER.debug('signalname: %s' % argdic.get('signalname'))

        #----------------------------------------------------------------------- 
        # Get signal from signals data if setname and signal name was specified 
        #-----------------------------------------------------------------------
        if argdic.get('setname') :
            try :
                signal = get_signal_by_name(signals, argdic.get('setname'), argdic.get('signalname'))
            except (NoSectionError, NoOptionError), inst :
                LOGGER.error(_('Error: Can not find signal "%s.%s" in signals data') % (argdic.get('setname'), argdic.get('signalname')))
        
        #----------------------------------------------------------------------- 
        # Get signal from stdin if setname and signal name was not specified 
        #-----------------------------------------------------------------------
        else :
            signal = sys.stdin.read().strip()
            if not re.compile(r'^[a-f0-9]{480}$', re.I) :
                LOGGER.error(_('Error: Input signal must be 240 pair of hexadecimal string.'))
                sys.exit(1)
        
        #----------------------------------------------------------------------- 
        # Emit signal
        #-----------------------------------------------------------------------
        try :
            emit(rs, signal, int( argdic['channel'] ))
        except TimeoutError, inst :
            LOGGER.error(_('Error: Access timeouted. RemoteStation did not reply for certain term'))
            sys.exit(1)
        except UnexpectedReplyError, inst :
            LOGGER.error(_('Error: RemoteStation replied unexpected value (%s)' % unicode(inst)))
            sys.exit(1)
        except :
            raise


class exlist( list ):
    '''Expanding build-in list class.
    '''
    
    def get(self, index, default=None ):
        '''Safe value getter for list
        
        This function can get value if the specified index element does not
        exists. if so, just return the default value(None)
        
        Argument :
            index (int)
                Element index to get.
                
            default (any, and can be omitted)
                Default value for non existance.
        
        Return :
            The element value if the index exists.
            None if the index does not exists
        '''
        
        # Checking index type
        if not str( index ).isdigit() :
            raise IndexError( 'invalid index' )
        
        # Selecting return value.
        if index >= len( self ) :
            return default
        else :
            return self[index]
        
    def multi_get(self, *args ):
        '''Safe multiple value getter for list
        
        This method can get multiple value in list, and it uses get method
        in this class, so you can get default value if you specified non-
        
        existance element.
        
        Argument :
            args (all int)
                index number list you want to get from list.
        
        Return :
            list of all element value requested, including None for non-
            existance element.
        '''
        args = exlist( args )
        return [self.get(i) for i in args]


class exstr( unicode ):
    '''Expanding build-in str class
    '''
    
    def split_to( self, sep=None, number=0, fill=None ):
        '''Expand split method with filler feature.
        
        split method for build in str class can get list separating sting with
        character. But with this method, when you need to get element with
        certain index, you will be disappointed for the result. For examle,

            >>> value = 'some string'.split(' ', 2)
            >>> value[3]
            Traceback (most recent call last):
              File "<stdin>", line 1, in ?
            IndexError: list index out of range
        
        you need to code for when the index does not exists.
        
            >>> if len(value) < 3:
            ...    value.append(3, '')
            >>> value[3]
            ''
        
        split_to method can split string to explicit length of list, with
        filling default value for element can not get with split. so you
        can code like this.
        
            >>> value = 'some string'.split(' ', 2, '')
            >>> value[3]
            ''
        
        Argument :
            sep (str, or can be omitted)
                character to separate string. Same as sep argument of split.
            
            number (int, or can be omitted)
                list length after split string.Defaults to 0.
            
            fill (any or can be omitted)
                filler value for element can not get with split method.
        '''
        arr = exlist(self.split(sep, number))
        narr = exlist()
        for i in range(0, number + 1) :
            narr.append(arr.get(i, fill))
        return narr


if __name__ == '__main__' :
    main()
