# coding: utf-8
'''\
Core module for RemoteStation device
'''

__author__ = 'M.Decimal at Specurio Studio.'
__version__ = '0.1'
__date__ = '2009-12-08'
__credits__ = """\
RemoteStation PC-OP-RS1 is a product for trasmitting ir signals made by
Japanese peripheral equipment vender Buffalo,Inc.
This script was coded on Pydev, powerful python IDE by Fabio Zadrozny, and now
becoming Apatana open source product.
Of course, Pydev is plug-in for, THE GREATE IDE, Eclipse provided by Eclipse
Foundation.
"""

import logging
import os
import gettext
import re
import serial
import binascii
import time

from exception import *


def _acn_dirname(path, gen=0):
    for i in range(0, gen + 1) :
        path = os.path.dirname(path)
    return path
os.path.ancdirname = _acn_dirname

# decision file whether script is in develop environment
IN_DEVELOP_ENV = False
DEVELOP_TOP_DIR = os.path.ancdirname(__file__, 2)
if os.path.exists(os.path.join(DEVELOP_TOP_DIR, 'setup.py')) :
    IN_DEVELOP_ENV = True


#======================================================================
# Logging SETTINGS
#
LOGGER = logging.getLogger('managers.device')


#======================================================================
# L10N
#
# setting gettext
if IN_DEVELOP_ENV :
    _ = gettext.translation('managers.rsdevice',
                            os.path.join(DEVELOP_TOP_DIR, 'locale'),
                            ).ugettext
else :
    _ = gettext.translation('managers.rsdevice', fallback=True).ugettext


#===============================================================================
# Device default settings
#===============================================================================
# Default for serial port
serial_default = {
    'baudrate':         115200,
    'bytesize':         8,
    'parity':           None,
    'stopbits':         1,
    'timeout':          60,
}

# Default for protocol against RS
protocol_default = {
    'order_led':          '69',
    'order_capture':      '72',
    'order_emit':         '74',
    'report_wait':        '59',
    'report_cont':        '53',
    'report_check':       '4f',
    'report_end':         '45',
    'data_length':        240,
}


class DeviceBase(object):
    '''\
    Abstract Base class for independent platform device class.
    
    Derrive from this class for making platform device class, with name
    "Device" in module file named as platform name(this name can get by 
    platform.system()).
    And Platform independent class need to override get_devices and is_exists
    method in those own way.
    
        class Device( DeviceBase ):
            @staticmethod
            def get_devices()
                # Code for get device name for RS connected your machine and
                # must return list of device name that could fuound.
                
            @staticmethod
            def is_exists(name)
                # Code for find whether passed device name is exists in your
                # machine and must return boolean depend on that result.
    '''


    def __init__(self, name=None, auto_select=0, serial_conf=serial_default, protocol_conf=protocol_default):
        '''Constructor for DeviceBase instances
        
        Argument :
            name (str or None or can be omitted)
                Device name for RS. If this argument was omitted, this
                constructor find that automatically with get_devices().
                
            auto_select (int or can be omitted)
                If constructor have to find device automatically (name argument
                did not supplied), this value decide which device should be
                selected for this instance, because get_devices() returns list
                including one or more elements.
            
            serial_conf (dic or can be omitted)
                Dictionary for Serial port settings. see the part of module
                variables settings on the top of this file.
            
            protocol_conf (dic or can be omitted)
                Dictionary for protocol settings. see the part of module
                variables settings on top of this file.
        '''

        #=======================================================================
        # If name was not supplied, find device name automatically
        #=======================================================================
        if name is None or name == '' :
            LOGGER.info(_('finding RemoteStation device automatically'))
            devices = self.get_devices()

            # Could not find any RS device
            if len(devices) == 0 :
                raise SetupDeviceError(_('can not find any RemoteStation Device'))

            # Select device depend on auto_select value
            if auto_select < 0 or len(devices) < auto_select + 1 :
                LOGGER.warning('auto_select illegal value was ignored, using it as 0')
                auto_select = 0
            name = devices[auto_select]

        # Set instance value "name" to the specified, or founded device name
        # But before that, check existance of the device
        if self.is_exists(name) :
            self.name = name
        else :
            raise DeviceNotFoundError('device %s does not exists' % name)

        LOGGER.info('RemoteStation device name = %s', self.name)

        #=======================================================================
        # Serial
        #=======================================================================
        LOGGER.info(_('Making serial device...'))
        serial_conf = normalize_serial_conf(serial_conf)
        self.serial = serial.Serial(baudrate=serial_conf['baudrate'],
                                     bytesize=serial_conf['bytesize'],
                                     parity=serial_conf['parity'],
                                     stopbits=serial_conf['stopbits'],
                                     timeout=serial_conf['timeout'])
        #=======================================================================
        # Protocol
        #=======================================================================
        protocol_conf = normalize_protocol_conf(protocol_conf)
        self.protocol = protocol_conf
        self.SIGN_ORDER_LED = protocol_conf['order_led']
        self.SIGN_ORDER_CAPTURE = protocol_conf['order_capture']
        self.SIGN_ORDER_EMIT = protocol_conf['order_emit']
        self.SIGN_REPORT_WAIT = protocol_conf['report_wait']
        self.SIGN_REPORT_CONT = protocol_conf['report_cont']
        self.SIGN_REPORT_CHECK = protocol_conf['report_check']
        self.SIGN_REPORT_END = protocol_conf['report_end']
        self.DATA_LENGTH = protocol_conf['data_length']

        # Set Serial class method to this class method for convinience.
        self.open = self.serial.open
        self.close = self.serial.close


    @staticmethod
    def get_devices():
        '''Static method for finding device name
           override this method individually for platform
        '''
        pass

    @staticmethod
    def is_exists(name):
        '''Static method for check whether device.name is really exists
           override this method individually for platform
        '''
        pass

    def initialize(self):
        '''Initialize RS
        '''

        self.serial.port = self.name
        self.open()
        self.write('00')
        self.close()

    def led(self, repeat=1, interval=0.2):
        '''Light on RS access LED
        
        Argument :
            repeat (int or can be omitted)
                How many time you want to light on the LED
            
            interval (float or can be omitted)
                How long the LED light out between light on in second.
        '''
        self.initialize()
        self.open()

        LOGGER.info(_('LED lights %d times with interval %f' % (repeat, interval)))
        for i in range(0, repeat) :
            self.write(self.SIGN_ORDER_LED)
            self.read(1, [self.SIGN_REPORT_CHECK, self.SIGN_REPORT_WAIT])
            time.sleep(interval)

        self.close()

    def capture(self):
        '''Capture the signal on ir reciever on RS        
        '''

        self.initialize()
        self.open()

        LOGGER.info(_('reciving signal'))
        self.write(self.SIGN_ORDER_CAPTURE)
        self.read(1, self.SIGN_REPORT_WAIT)
        self.read(1, self.SIGN_REPORT_CONT)
        data = self.read(self.DATA_LENGTH)
        self.read(1,self.SIGN_REPORT_END)

        self.close()
        return data

    def emit(self, data, chan) :
        '''Emit ir signal on RS ir emitter
        
        Argument :
            data (str)
                Actual signal data. This must be 480 length of string
                representing hexdicimal characters.
            
            chan (int)
                Channel number to emit. It must be 1 - 4 integer number. On RS,
                yellow taped emitter on connector "A" is channnel 1, emitter
                without tape on same connector is 2, and same on connector B
                3 and 4.
                chan can be 0, it means signal will be emitted on all channel.
        '''
        if (not isinstance(chan, int)) or chan < 0 or chan > 4 :
            raise InvalidChannelError('Invalid channel value : %s', unicode(chan))

        self.initialize()
        self.open()

        LOGGER.info(_('casting signal'))
        LOGGER.debug(_('  channel     : %d' % chan))
        LOGGER.debug(_('  signal byte : %s' % len(data)))
        if chan :
            self.write(self.SIGN_ORDER_EMIT)
            self.read(1, self.SIGN_REPORT_WAIT)
            self.write(str(chan + 30))
            self.read(1, self.SIGN_REPORT_WAIT)
            self.write(data)
            self.read(1, self.SIGN_REPORT_END)
        else :
            for i in range(1, 5) :
                self.write(self.SIGN_ORDER_EMIT)
                self.read(1, self.SIGN_REPORT_WAIT)
                self.write(str(i + 30))
                self.read(1, self.SIGN_REPORT_WAIT)
                self.write(data)
                self.read(1, self.SIGN_REPORT_END)

        self.close()
        
    def write(self, value):
        LOGGER.debug('> send %s' % value)
        self.serial.write(binascii.unhexlify(value))
        
    def read(self, length, expected=None):
        '''Check value returned from RS is match as you expected.
        
        This method check return value and value you expected, if it is not
        matched, raise exception.
        
        Argument:
            exstrs (str)
                Value you expected
                
            gotstr (str)
                Value RS actually return.
        '''
        
        got = binascii.hexlify(self.serial.read(length))
        LOGGER.debug('< got %s' % got)
        if got == '' :
            raise TimeoutError('')
        if expected is not None :
            if (isinstance(expected, str) and expected != got) or\
               (isinstance(expected, list) and not (got in expected)) :
                self.close()
                raise UnexpectedReplyError(_('replied %s (expected %s)') % (got, str(expected)))
        
        return got

def normalize_serial_conf(conf):
    '''Normalizing dictionary for serial setting
    '''
    
    if conf.get('baudrate', None) is None or not str(conf['baudrate']).isdigit() :
        conf['baudrate'] = 115200
    conf['baudrate'] = int(conf['baudrate'])

    if conf.get('bytesize', None) is None or not str(conf['bytesize']).isdigit():
        conf['baudrate'] = 8
    conf['bytesize'] = int(conf.get('bytesize', 0))
    if conf['bytesize'] == 5 :
        conf['bytesize'] = serial.FIVEBITS
    elif conf['bytesize'] == 6 :
        conf['bytesize'] = serial.SIXBITS
    elif conf['bytesize'] == 7 :
        conf['bytesize'] = serial.SEVENBITS
    elif conf['bytesize'] == 8 :
        conf['bytesize'] = serial.EIGHTBITS
    else :
        conf['bytesize'] = serial.EIGHTBITS

    conf['parity'] = str(conf.get('parity', 0))
    if (conf['parity'] == '0') or (conf['parity'] == '') or\
       (re.match(r'no.*', conf['parity'], re.I)) :
        conf['parity'] = serial.PARITY_NONE
    elif re.match('even', conf['parity'], re.I) :
        conf['parity'] = serial.PARITY_EVEN
    elif re.match('odd', conf['parity'], re.I) :
        conf['parity'] = serial.PARITY_ODD
    elif re.match('mark', conf['parity'], re.I) :
        conf['parity'] = serial.PARITY_MARK
    elif re.match('space', conf['parity'], re.I) :
        conf['parity'] = serial.PARITY_SPACE

    if conf.get('stopbits', None) is None or\
       not re.match(r'^\d+\.?\d*$', str(conf['stopbits'])):
        conf['stopbits'] = 1.0
    conf['stopbits'] = float(conf['stopbits'])
    if conf['stopbits'] == 1.0 :
        conf['stopbits'] = serial.STOPBITS_ONE
    elif conf['stopbits'] == 1.5 :
        conf['stopbits'] = serial.STOPBITS_ONE_POINT_FIVE
    elif conf['stopbits'] == 2.0 :
        conf['stopbits'] = serial.STOPBITS_TWO

    if conf.get('timeout', None) is None or\
       not re.match(r'^\d+\.?\d*$', str(conf['timeout'])):
        conf['timeout'] = 30
    conf['timeout'] = float(conf['timeout'])

    return conf

def normalize_protocol_conf(conf):
    '''Normalizing dictionary for protocool setting
    '''
    
    for (key, value) in protocol_default.items() :
        if conf.get(key, None) is None or not str(conf[key]).isdigit() :
            conf[key] = value
        conf[key] = str(conf[key])
        
    conf['data_length'] = int(conf['data_length'])
    
    return conf

