/*
 * Copyright (c) 2003 Shinji Kashihara. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies
 * this distribution, and is available at cpl-v10.html.
 */
package mergedoc.ui;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;

import mergedoc.MergeDocException;
import mergedoc.core.Preference;
import mergedoc.xml.ConfigManager;
import mergedoc.xml.Persister;
import mergedoc.xml.ReplaceEntry;

/**
 * 設定パネルです。
 * @author Shinji Kashihara
 */
public class PreferencePanel extends JPanel {

    /** 基本設定パネル：API ドキュメントディレクトリ ファイル選択フィールド */
    private FileChooserField docField = new FileChooserField();

    /** 基本設定パネル：入力ソースアーカイブファイル ファイル選択フィールド */
    private FileChooserField srcField = new FileChooserField();
    
    /** 基本設定パネル：出力ソースアーカイブファイル ファイル選択フィールド */
    private FileChooserField outField = new FileChooserField();


    /** 詳細設定パネル：プレビュー スクロールペイン */
    private PreviewScrollPane previewScrollPane;

    /** 詳細設定パネル：スプリットペイン */
    private JSplitPane splitPane = new JSplitPane();

    /** 詳細設定パネル：置換エントリのチェックリスト */
    private List entryCheckList = new ArrayList();


    /** メッセージダイアログ */
    private MessageDialog dialog;


    /**
     * コンストラクタです。 
     * @param dialog メッセージダイアログ
     * @throws MergeDocException 設定ファイルが取得できない場合
     */
    public PreferencePanel(MessageDialog dialog) throws MergeDocException {

        this.dialog = dialog;

        // レイアウト設定
        setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
        setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
        setMaximumSize(ComponentFactory.createMaxDimension());

        // 上下パネルをこのコンテナに追加
        add(createUpperPanel());
        add(ComponentFactory.createSpacer(0, 7));
        add(createLowerPanel());
    }

    /**
     * 上部の基本設定パネルを作成します。
     * @return 上部の基本設定パネル
     * @throws MergeDocException 設定ファイルが取得できない場合
     */
    private JComponent createUpperPanel() throws MergeDocException {

        // 入力フィールドの設定
        docField.getLabel().setText("API ドキュメントディレクトリ");
        srcField.getLabel().setText("入力ソースアーカイブファイル");
        outField.getLabel().setText("出力ソースアーカイブファイル");
        docField.setDirectoryOnly(true);
        docField.setCharset( Charset.forName("ISO-2022-JP") );
        
        // API ドキュメントディレクトリ選択時のリスナー設定
        docField.setChooseListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                resolveFilePath();
            }
        });
        
        // 上部パネル作成
        JPanel panel = new TitledPanel("基本設定");
        panel.add(docField);
        panel.add(srcField);
        panel.add(outField);

        // ターゲットディレクトリの取得
        final String OPTION_KEY = "target.directory";
        String targetStr = System.getProperty(OPTION_KEY);
        if (targetStr != null) {
        
            // API ドキュメントディレクトリの設定
            File targetDir = new File(targetStr);
            if (!targetDir.exists() || targetDir.isFile()) {
                throw new MergeDocException(
                    "オプション " + OPTION_KEY + " に指定された値 " + targetStr +
                    " は\n存在しないかディレクトリではありません。");
            }
            File docDir = searchDocDirectory(targetDir);
            if (docDir != null) {
                docField.setFile(docDir);
            }

            // 入出力ソースアーカイブファイルの設定
            File[] fs = targetDir.listFiles();
            for (int i = 0; i < fs.length; i++) {
                String name = fs[i].getName();
                if (name.equals("src.zip") || name.equals("src.jar")) {
                    String inName = fs[i].getPath();
                    String outName = inName.replaceFirst("src....$", "srcja.zip");
                    srcField.setFile(new File(inName));
                    outField.setFile(new File(outName));
                }
            }
        }

        // 設定ファイル読み込み
        loadPersister(docField, Persister.DOC_DIR, Persister.DOC_ENC);
        loadPersister(srcField, Persister.IN_FILE, Persister.IN_ENC);
        loadPersister(outField, Persister.OUT_FILE, Persister.OUT_ENC);

        return panel;
    }

    /**
     * 下部の詳細設定パネルを作成します。
     * @return 下部の詳細設定パネル
     * @throws MergeDocException 設定ファイルが取得できない場合
     */
    private JComponent createLowerPanel() throws MergeDocException {
        
        // チェックボックス配置ペインを作成
        JPanel checkPanel = new JPanel();
        checkPanel.setLayout(new BoxLayout(checkPanel, BoxLayout.Y_AXIS));
        List list = ConfigManager.getInstance().getGlobalEntries();
        for (Iterator it = list.iterator(); it.hasNext();) {
            ReplaceEntry entry = (ReplaceEntry) it.next();
            EntryCheckBox cb = new EntryCheckBox(entry);
            cb.addActionListener(new ActionListener(){
                public void actionPerformed(ActionEvent e) {
                    previewScrollPane.updatePreview(entryCheckList);
                }
            });
            checkPanel.add(cb);
            entryCheckList.add(cb);
        }
        JScrollPane checkScrollPane = ComponentFactory.createScrollPane(checkPanel);
        checkScrollPane.getVerticalScrollBar().setUnitIncrement(10);
        
        // プレビューペインを作成
        previewScrollPane = new PreviewScrollPane(dialog);
        
        // スプリットペインに配置
        splitPane.setBorder(BorderFactory.createEmptyBorder());
        splitPane.setOrientation(JSplitPane.VERTICAL_SPLIT);
        splitPane.setContinuousLayout(true);
        splitPane.setTopComponent(checkScrollPane);
        splitPane.setBottomComponent(previewScrollPane);
        JPanel splitPanel = new JPanel();
        splitPanel.setLayout(new BoxLayout(splitPanel, BoxLayout.X_AXIS));
        splitPanel.add(splitPane);

        // 下部パネル作成
        JPanel panel = new TitledPanel("詳細設定");
        panel.setMaximumSize(ComponentFactory.createMaxDimension());
        panel.add(splitPanel);

        // 設定ファイルからスプリットペイン分割位置を取得
        Persister psst = Persister.getInstance();
        int loc = psst.getInt(Persister.DETAIL_PANEL_HEIGHT, 88);
        splitPane.setDividerLocation(loc);

        // 設定ファイルからチェック有無を取得
        String[] descs = psst.getStrings(Persister.REPLACE_DESCRIPTION_ARRAY);
        for (Iterator it = entryCheckList.iterator(); it.hasNext();) {
            EntryCheckBox ecb = (EntryCheckBox) it.next();
            ReplaceEntry entry = ecb.getReplaceEntry();
            String desc = entry.getDescription();
            for (int i = 0; i < descs.length; i++) {
                if (desc.equals(descs[i])) {
                    ecb.setSelected(true);
                    break;
                }
            }
        }
        previewScrollPane.updatePreview(entryCheckList);

        return panel;
    }
    
    /**
     * ファイル選択フィールドに設定ファイルから取得した値をセットします。
     * @param field ファイル選択フィールド
     * @param pathKey パスを示す設定ファイルのキー
     * @param charKey 文字セットを示す設定ファイルのキー
     * @throws MergeDocException 設定ファイルが取得できない場合
     */
    private void loadPersister(FileChooserField field, Persister.Key pathKey,
        Persister.Key charKey) throws MergeDocException
    {
        Persister psst = Persister.getInstance();
        if (field.getFile().getPath().equals("")) {
            String path = psst.getString(pathKey, "");
            if (path.length() > 0) {
                field.setFile(new File(path));
            }
        }
        try {
            String enc = psst.getString(charKey);
            field.setCharset( Charset.forName(enc) );
        } catch (IllegalArgumentException e) {
        }
    }

    /**
     * API ドキュメントディレクトリを元に予想される入力ソースアーカイブファイルと
     * 出力ソースアーカイブファイルをセットします。
     */
    private void resolveFilePath() {

        String src = srcField.getFile().getPath();
        String out = outField.getFile().getPath();
        if (!src.equals("") || !out.equals("")) {
            return;
        }

        File dir = docField.getFile();
        while (dir != null) {

            String path = dir + File.separator + "src.";
            File zip = new File(path + "zip");
            if (zip.exists()) {
                srcField.setFile(zip);
                break;
            } else {
                File jar = new File(path + "jar");
                if (jar.exists()) {
                    srcField.setFile(jar);
                    break;
                }
            }
            dir = dir.getParentFile();
        }
        
        src = srcField.getFile().getPath();
        if (!src.equals("")) {
            String outName = src.replaceFirst("src....$", "srcja.zip");
            outField.setFile(new File(outName));
        }
    }

    /**
     * 基本設定パネルの各ラベル横幅を最大のものと同じになるようにレイアウト
     * し直します。ただしラベルが可視状態でない場合は正常に動作しません。<br>
     * このメソッドはスレッドに対して安全です。
     */
    public void revalidateLabel() {

        int maxWidth = 0;
        JLabel[] labels = {
            docField.getLabel(),
            srcField.getLabel(),
            outField.getLabel(),
        };
        for (int i = 0; i < labels.length; i++) {
            int width = (int) labels[i].getSize().getWidth();
            if (width > maxWidth ) {
                maxWidth = width;
            }
        }
        for (int i = 0; i < labels.length; i++) {
            JLabel label = labels[i];
            ComponentFactory.ensureSize(label, maxWidth, label.getHeight());
            label.revalidate();
        }
    }

    /**
     * Javadoc API ディレクトリを再帰的に探します。
     * @param file 探すディレクトリ
     * @return Javadoc API ディレクトリパス
     */
    private File searchDocDirectory(File file) {
        File dir = null;
        File[] fs = file.listFiles();
        for (int i = 0; i < fs.length; i++) {
            File f = fs[i];
            if (f.isFile()) {
                if (f.getName().equals("package-list")) {
                    dir = file;
                }
            } else {
                dir = searchDocDirectory(f); //再帰
            }
            if (dir != null) {
                break;
            }
        }
        return dir;
    }

    /**
     * 選択された置換エントリの配列を取得します。
     * @return 選択された置換エントリの配列
     */
    public ReplaceEntry[] getSelectedEntries() {
        List enables = new ArrayList();
        for (Iterator it = entryCheckList.iterator(); it.hasNext();) {
            EntryCheckBox cb = (EntryCheckBox) it.next();
            if (cb.isSelected()) {
                ReplaceEntry entry = cb.getReplaceEntry();
                enables.add(entry);
            }
        }
        ReplaceEntry[] entries = (ReplaceEntry[])
            enables.toArray(new ReplaceEntry[enables.size()]);
        return entries;
    }

    /**
     * マージ設定を取得します。
     * @return マージ設定
     */
    public Preference getPreference() {

        return new Preference() {

            private File docDir  = docField.getFile();
            private File srcFile = srcField.getFile();
            private File outFile = outField.getFile();
            private Charset docEnc = docField.getCharset();
            private Charset srcEnc = srcField.getCharset();
            private Charset outEnc = outField.getCharset();
            private ReplaceEntry[] entries = getSelectedEntries();

            public File getDocDirectory()  {return docDir;}
            public File getInputArchive()  {return srcFile;}
            public File getOutputArchive() {return outFile;}
            public Charset getDocEncoding()    {return docEnc;}
            public Charset getInputEncoding()  {return srcEnc;}
            public Charset getOutputEncoding() {return outEnc;}
            public ReplaceEntry[] getGlobalEntries() {return entries;}
        };
    }
    
    /**
     * このパネルの設定内容を Persister にセットします。
     * @throws Persister にセット出来なかった場合
     */
    public void persistent() throws MergeDocException {

        Preference pref = getPreference();
        Persister psst = Persister.getInstance();

        psst.setString(Persister.DOC_DIR, pref.getDocDirectory().getPath());
        psst.setString(Persister.IN_FILE, pref.getInputArchive().getPath());
        psst.setString(Persister.OUT_FILE, pref.getOutputArchive().getPath());
        psst.setString(Persister.DOC_ENC, pref.getDocEncoding().toString());
        psst.setString(Persister.IN_ENC, pref.getInputEncoding().toString());
        psst.setString(Persister.OUT_ENC, pref.getOutputEncoding().toString());
        psst.setInt(Persister.DETAIL_PANEL_HEIGHT, splitPane.getDividerLocation());

        ReplaceEntry[] entries = pref.getGlobalEntries();
        List descList = new ArrayList();
        for (int i = 0; i < entries.length; i++) {
            descList.add(entries[i].getDescription());
        }
        String[] descs = (String[]) descList.toArray(new String[descList.size()]);
        psst.setStrings(Persister.REPLACE_DESCRIPTION_ARRAY, descs);
    }
}
