/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.aspect;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import jp.sourceforge.mergedoc.pleiades.log.Logger;
import jp.sourceforge.mergedoc.pleiades.util.FastProperties;
import jp.sourceforge.mergedoc.pleiades.util.Files;

import org.apache.commons.io.FileUtils;

/**
 * metadata デフォルト・イニシャライザーです。
 * <p>
 * @author cypher256
 */
public class MetadataDefaultInitializer {

	/** ロガー */
	private static Logger log = Logger
			.getLogger(MetadataDefaultInitializer.class);

	/** 除外リスト */
	@SuppressWarnings("serial")
	private static final List<String> EXCLUDES = Collections.unmodifiableList(new LinkedList<String>() {
		
		{
			// Ant デフォルト除外
			add(".*~");
			add("#.*#");
			add(".#.*");
			add("%.*%");
			add("\\._.*");
			add("CVS");
			add("\\.cvsignore");
			add("SCCS");
			add("vssver\\.scc");
			add("\\.svn");
			add("\\.DS_Store");

			// その他 [#13834]
			add("\\.bzr");			// Bazaar
			add("\\.cdv");			// Codeville
			add("\\.git");			// Git
			add("\\.hg");			// Mercurial
			add("\\.pc");			// quilt
			add("RCS");				// RCS
			add("_darcs");			// darcs
			add("_sgbak");			// Vault/Fortress
			
			// その他 [#13834] (miau)
			add("vssver2\\.scc");	// VSS2005
			add("_svn");			// SVN オプション
		}
		
		@Override
		public boolean contains(Object name) {
			for (String pattern : this) {
				if (((String) name).matches(pattern)) {
					return true;
				}
			}
			return false;
		}
	});

	/** Eclipse ホーム親ディレクトリー */
	private final String ECLIPSE_HOME_PARENT;

	/** metadata デフォルト・ディレクトリー */
	private File metadataDefaultDir;

	/** metadata コピー元ルート・ディレクトリー */
	private File srcRootDir;

	/** metadata コピー先ルート・ディレクトリー */
	private File dstRootDir;

	/** コピー数 */
	private int copyCount;

	/**
	 * metadata デフォルトで初期化します。
	 */
	public static void init() {
		try {
			new MetadataDefaultInitializer();
		} catch (Exception e) {
			log.error("metadata デフォルト設定の初期化に失敗しました。", e);
		}
	}

	/**
	 * コンストラクタ。
	 * @throws URISyntaxException 
	 * @throws IOException 
	 */
	private MetadataDefaultInitializer() throws URISyntaxException, IOException {

		File eclipseHomeParent = Pleiades.getEclipseHome().getParentFile();
		ECLIPSE_HOME_PARENT = (eclipseHomeParent != null) ? eclipseHomeParent
				.getPath().replace("\\", "/") : null;
		if (ECLIPSE_HOME_PARENT == null) {
			return;
		}
		metadataDefaultDir = new File(ECLIPSE_HOME_PARENT, ".metadata.default");
		if (!metadataDefaultDir.exists()) {
			return;
		}
		srcRootDir = new File(metadataDefaultDir, ".plugins");
		dstRootDir = new File(new File(getWorkspace(), ".metadata"), ".plugins");

		copyMetadata(srcRootDir);
		
		if (copyCount > 0) {
			log.info(".metadata.default から " + copyCount
				+ " 個のファイルをコピーしました。" + dstRootDir.getParentFile());
		}
	}

	/**
	 * metadata を再帰的にコピーします。
	 * @param srcDir コピー元ディレクトリー
	 * @throws IOException
	 */
	private void copyMetadata(File srcDir) throws IOException {

		for (File src : srcDir.listFiles()) {
			
			if (EXCLUDES.contains(src.getName())) {
				continue;
			}
			if (src.isDirectory()) {
				copyMetadata(src); // 再帰
			} else {
				File dst = new File(dstRootDir, Files.relativePath(srcRootDir,
						src));
				if (!dst.exists()) {
					String s = FileUtils.readFileToString(src);
					s = s.replace("%ECLIPSE_HOME_PARENT%", ECLIPSE_HOME_PARENT);
					FileUtils.writeStringToFile(dst, s);
					copyCount++;
				}
			}
		}
	}

	/**
	 * ワークスペースを取得します。
	 * @return ワークスペース
	 * @throws URISyntaxException
	 */
	private File getWorkspace() throws URISyntaxException {

		String workspacePath = System.getProperty("osgi.instance.area");
		if (workspacePath != null) {
			return new File(new URI(workspacePath));
		}

		File configuration = Pleiades.getConfigurationPath().getParentFile();
		File idePrefs = new File(configuration,
				".settings/org.eclipse.ui.ide.prefs");
		String value = new FastProperties(idePrefs).get("RECENT_WORKSPACES");
		workspacePath = value.replaceFirst("\n.*", "");
		return new File(workspacePath);
	}
}
