/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.aspect;

import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import jp.sourceforge.mergedoc.pleiades.log.Logger;

/**
 * 安全なタスク実行サービスです。
 * <p>
 * JDK デフォルトの ThreadPoolExecutor はタスクを実行用に受け入れることが
 * できない場合、RejectedExecutionException をスローしますが、
 * このクラスはこの例外が発生した場合、非同期実行せずに同期実行に切り替えて
 * 実行します。
 * <p>
 * @author cypher256
 */
public class SafetyExecutorService extends ThreadPoolExecutor {

	/** ロガー */
	private static Logger log = Logger.getLogger(SafetyExecutorService.class);
	
	/**
	 * コンストラクタ。
	 * <p>
	 * 設定されるスレッド数やタイムアウト時間は Executors.newCachedThreadPool()
	 * と同じです。
	 */
	public SafetyExecutorService() {
		super(0, Integer.MAX_VALUE,
                60L, TimeUnit.SECONDS,
                new SynchronousQueue<Runnable>());
	}

	@Override
	public void execute(Runnable command) {
		try {
			super.execute(command);
		} catch (RejectedExecutionException e) {
			log.debug("RejectedExecutionException が発生したため、同期実行します。");
			command.run();
		}
	}
}
