/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.aspect;

import java.io.IOException;
import java.util.List;
import java.util.regex.Matcher;

import javassist.CannotCompileException;
import javassist.CtBehavior;
import javassist.CtClass;
import javassist.CtMethod;
import javassist.NotFoundException;
import javassist.expr.ExprEditor;
import javassist.expr.MethodCall;
import jp.sourceforge.mergedoc.pleiades.aspect.advice.AspectMapping;
import jp.sourceforge.mergedoc.pleiades.aspect.advice.JointPoint;
import jp.sourceforge.mergedoc.pleiades.aspect.advice.PointCut;
import jp.sourceforge.mergedoc.pleiades.aspect.advice.JointPoint.EditPoint;
import jp.sourceforge.mergedoc.pleiades.aspect.advice.PointCut.Timing;
import jp.sourceforge.mergedoc.pleiades.log.Logger;

/**
 * 翻訳処理をクラス・オブジェクトに埋め込むエディターです。
 * <p>
 * 対象となるジョイント・ポイントやアドバイスは
 * {@link jp.sourceforge.mergedoc.pleiades.aspect.advice.AspectMapping}
 * から取得します。
 * <p>
 * @author cypher256
 */
public class TranslationEditor extends ExprEditor {

	/** ロガー */
	private static final Logger log = Logger.getLogger(TranslationEditor.class);

	/** 編集対象となる CtClass オブジェクト */
	private final CtClass ctClass;

	/** 編集済みの場合は true */
	private boolean isEdited;

	/**
	 * 翻訳エディターを構築します。
	 * <p>
	 * @param ctClass CtClass オブジェクト
	 */
	public TranslationEditor(CtClass ctClass) {
		this.ctClass = ctClass;
	}

	/**
	 * コンストラクタまたはメソッド呼び出しを編集します。
	 * editPoint が call の場合に呼び出されます。
	 */
	@Override
	public void edit(MethodCall methodCall) throws CannotCompileException {

		//-------------------------------------
		// 呼び出し先の情報を取得
		//-------------------------------------
		String calleeClassName = methodCall.getClassName();
		String calleeMethodName = methodCall.getMethodName();
		AspectMapping mapping = AspectMapping.getInstance();
		if (!mapping.containesMethodCall(calleeClassName, calleeMethodName)) {
			return;
		}

		JointPoint calleeJP = new JointPoint();
		calleeJP.setEditPoint(EditPoint.CALL);
		calleeJP.setClassName(calleeClassName);
		calleeJP.setMethodName(calleeMethodName);
		try {
			CtMethod calleeMethod = methodCall.getMethod();
			calleeJP.setDescriptor(calleeMethod.getMethodInfo().getDescriptor());
		} catch (NotFoundException e) {
			// クラスパスに呼び出し先が含まれていない場合は無視
			log.warn("クラスパスなし。" + calleeClassName + " ← " + ctClass.getName());
			return;
		}
		PointCut pointCut = mapping.getPointCut(calleeJP);
		if (pointCut == null) {
			return;
		}
		if (pointCut.getExcludeTrace().size() > 0 || pointCut.getIncludeTrace().size() > 0) {
			throw new IllegalStateException(
				"editPoint が call の場合、*cludeTrace 属性を指定することはできません。" +
				pointCut);
		}

		//-------------------------------------
		// 呼び出し元の情報を取得
		//-------------------------------------
		CtBehavior callerMethod = methodCall.where();

		// 呼び出し元の場所による除外
		if (contains(pointCut.getExcludeWheres(), callerMethod)) {
			return;
		}

		// 呼び出し元の場所による限定
		List<JointPoint> includeWheres = pointCut.getIncludeWheres();
		if (includeWheres.size() > 0 && !contains(includeWheres, callerMethod)) {
			return;
		}

		String advice = pointCut.getAdvice();
		if (advice.contains("?{JOINT_POINT}")) {
			JointPoint jointPoint = new JointPoint();
			jointPoint.setEditPoint(EditPoint.CALL);
			jointPoint.setClassName(callerMethod.getDeclaringClass().getName());
			jointPoint.setMethodName(callerMethod.getName());
			jointPoint.setDescriptor(callerMethod.getMethodInfo().getDescriptor());
			advice = replaceJointPoint(advice, jointPoint);
		}

		methodCall.replace(advice);
		isEdited = true;
	}

	/**
	 * ジョイント・ポイントのリストにクラス名とメソッド名が含まれるか判定します。
	 *
	 * @param jointPointList ジョイント・ポイントのリスト
	 * @param callerMethod 呼び出し元メソッド
	 * @return 含まれる場合は true
	 */
	private boolean contains(List<JointPoint> jointPointList, CtBehavior callerMethod) {

		// 呼び出し元の情報
		String callerClassName = callerMethod.getDeclaringClass().getName();
		String callerMethodName = callerMethod.getName();

		// 現状は include、exclude の modefier、descriptor は未サポート
		for (JointPoint jointPoint : jointPointList) {

			// クラス名は前方一致
			if (callerClassName.startsWith(jointPoint.getClassName())) {

				// メソッド名は完全一致
				String mName = jointPoint.getMethodName();
				if (mName == null || callerMethodName.equals(mName)) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * アドバイス文字列に含まれる ${JOINT_POINT} を new JointPont(...) に
	 * 置き換えます。
	 *
	 * @param advice アドバイス
	 * @param jointPoint ジョイント・ポイント
	 * @return 置換後のアドバイス文字列
	 */
	private String replaceJointPoint(String advice, JointPoint jointPoint) {

		String replacement = Matcher.quoteReplacement(
				"new " + JointPoint.class.getName() + "(" +
				EditPoint.class.getName() + "." + jointPoint.getEditPoint().name() +
				",\"" + jointPoint.getClassName() + "\"" +
				",\"" + jointPoint.getMethodName() + "\"" +
				",\"" + jointPoint.getDescriptor() + "\"" +
				")");
		return advice.replaceAll("\\?\\{JOINT_POINT\\}", replacement);
	}

	/**
	 * コンストラクタまたはメソッドを編集します。
	 * editPoint が execution の場合に呼び出されます。
	 * <p>
	 * @param ctBehavior コンストラクタまたはメソッド・オブジェクト
	 * @throws CannotCompileException コンパイルできない場合
	 */
	public void editBehavior(CtBehavior ctBehavior) throws CannotCompileException {

		String className = ctClass.getName();
		String methodName = ctBehavior.getName();

		JointPoint jointPoint = new JointPoint();
		jointPoint.setEditPoint(EditPoint.EXECUTION);
		jointPoint.setClassName(className);
		jointPoint.setMethodName(methodName);
		jointPoint.setDescriptor(ctBehavior.getMethodInfo().getDescriptor());

		AspectMapping mapping = AspectMapping.getInstance();
		PointCut pointCut = mapping.getPointCut(jointPoint);
		if (pointCut == null) {
			return;
		}
		if (pointCut.getExcludeWheres().size() > 0 || pointCut.getIncludeWheres().size() > 0) {
			throw new IllegalStateException(
				"editPoint が execution の場合、*cludeWhere 属性を指定することはできません。" +
				pointCut);
		}

		Timing timing = pointCut.getTiming();
		String advice = pointCut.getAdvice();

		if (advice.contains("?{JOINT_POINT}")) {
			advice = replaceJointPoint(advice, jointPoint);
		} else {
			if (pointCut.getExcludeTrace().size() > 0 || pointCut.getIncludeTrace().size() > 0) {
				throw new IllegalStateException(
					"*cludeTrace を指定している場合は Advice に ?{JOINT_POINT} が" +
					"含まれている必要があります。" + jointPoint);
			}
		}

		if (timing == Timing.BEFORE) {
			ctBehavior.insertBefore(advice);
		} else if (timing == Timing.AFTER) {
			ctBehavior.insertAfter(advice);
		} else {
			throw new IllegalStateException(
			"編集ポイントが execution の場合、timing は before または after " +
			"である必要があります。" + jointPoint);
		}
		isEdited = true;
	}

	/**
	 * 編集結果をバイトコードで取得します。
	 * <p>
	 * @return バイトコード。未編集の場合は null。
	 * @throws IOException 入出力例外が発生した場合
	 * @throws CannotCompileException コンパイルできない場合
	 */
	public byte[] toBytecode() throws IOException, CannotCompileException {
		return isEdited ? ctClass.toBytecode() : null;
	}
}
