/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.aspect.resource;

import java.io.File;
import java.util.List;
import java.util.Set;

import jp.sourceforge.mergedoc.pleiades.aspect.Pleiades;
import jp.sourceforge.mergedoc.pleiades.aspect.advice.AspectMapping;
import jp.sourceforge.mergedoc.pleiades.aspect.advice.JointPoint;
import jp.sourceforge.mergedoc.pleiades.aspect.advice.PointCut;
import jp.sourceforge.mergedoc.pleiades.log.Logger;
import jp.sourceforge.mergedoc.pleiades.resource.AbstractTranslationDictionary;
import jp.sourceforge.mergedoc.pleiades.util.FastProperties;
import jp.sourceforge.mergedoc.pleiades.util.UnMnemonicProperties;

/**
 * キャッシュ機構および実行時除外機構を持つ翻訳辞書クラスです。
 * Eclipse 実行時の動的翻訳に利用されるクラスです。
 * <p>
 * @author cypher256
 */
public class DynamicTranslationDictionary extends AbstractTranslationDictionary {

	//////////////////////////////////////////////////////////////////////////
	// スタティック

	/** ロガー */
	private static final Logger log = Logger.getLogger(DynamicTranslationDictionary.class);

	/** 翻訳キャッシュ・プロパティー・ファイル */
	private static final File cacheFile = new File(
			Pleiades.getConfigurationPath(), "translation-cached.properties");

	/** このクラスのシングルトン・インスタンス */
	private static final DynamicTranslationDictionary singleton;
	static {
		if (log.isDebugEnabled()) {
			singleton = new DynamicTranslationLoggingDictionary();
		} else {
			singleton = new DynamicTranslationDictionary();
		}
	}

	/**
	 * 翻訳辞書インスタンスを取得します。
	 * <p>
	 * @return 翻訳辞書インスタンス
	 */
	public static DynamicTranslationDictionary getInstance() {
		return singleton;
	}

	//////////////////////////////////////////////////////////////////////////
	// インスタンス・フィールド

	/** ロード時の辞書サイズ */
	private int loadedSize;

	//////////////////////////////////////////////////////////////////////////
	// オーバーライド

	@Override
	protected String getValue(String enNoMnemonic) {

		// 親の同メソッド呼び出し
		String result = super.getValue(enNoMnemonic);

		// DEBUG
		//if (enNoMnemonic.equals("Example")) {
		//	String s = System.currentTimeMillis() + " [" + enNoMnemonic + "] → [" + result + "]";
		//	log.debug("デバッグ翻訳追跡スタックトレース", new Exception(s));
		//	return s;
		//}

		// 次回の起動用にキャッシュする
		getProperties().put(enNoMnemonic, result);

		return result;
	}

	@Override
	protected String editMnemonicEnToJa(String en, String enNoMnemonic,
			String jaNoMnemonic) {

		// ニーモニック処理なしの指定がある場合は何もしない
		if (Pleiades.getPleiadesOption().isNoMnemonic()) {
			return jaNoMnemonic;
		}
		// 親の同メソッド呼び出し
		return super.editMnemonicEnToJa(en, enNoMnemonic, jaNoMnemonic);
	}

	@Override
	protected void load() {

		// -clean の場合、翻訳辞書をロード
		if (Pleiades.getPleiadesOption().isClean() || !cacheFile.exists()) {
			
			super.load();
		}
		// -clean でない場合、翻訳辞書キャッシュをロード
		else {
			FastProperties properties = getProperties();
			properties.load(cacheFile);
			log.info("翻訳辞書キャッシュをロードしました。" + properties.size());
		}

		// ロード時の辞書サイズを保存
		loadedSize = getProperties().size();
	}

	//////////////////////////////////////////////////////////////////////////
	// 公開メソッド

	/**
	 * 翻訳プロパティーをキャッシュとして永続化します。
	 */
	public void store() {

		FastProperties prop = getProperties();

		if (prop.size() > loadedSize + 100) {
			prop.store(cacheFile, "翻訳辞書キャッシュ・プロパティー");
			log.info("翻訳辞書キャッシュを保管しました。" +
					loadedSize + " -> " + prop.size());
		}
	}

	/**
	 * 指定した英語リソース文字列から日本語リソースを探します。
	 * ニーモニックは日本用に変換されます。
	 * <p>
	 * @param en 英語リソース文字列
	 * @param jointPoint ジョイント・ポイント
	 * @return 日本語リソース文字列
	 */
	public String lookup(String en, JointPoint jointPoint) {

		// (&A) のような日本用のニーモニックが既に含まれている場合は
		// 翻訳済みであるため、何もしない
		if (UnMnemonicProperties.hasJaMnemonic(en)) {
			return en;
		}

		// 英語リソース文字列からニーモニック制御文字 & を除去
		String enNoMnemonic = UnMnemonicProperties.removeEnMnemonic(en);

		// 翻訳不要な場合
		if (isNoTranslation(enNoMnemonic, jointPoint)) {
			
			// 日本用ニーモニックをに変換はしない 2008.10.24
			// ロードが共通の場合はトレースで不要判断が不可能なため除外するが、
			// 代わりに UI 表示時に再度、翻訳するため。
			//String enMnemonicJa = editMnemonicEnToJa(en, enNoMnemonic, enNoMnemonic);
			//return enMnemonicJa;
			return en;
		}

		// 翻訳プロパティーから日本語訳を取得
		String jaNoMnemonic = getValue(enNoMnemonic);

		// ニーモニックを日本用に変換。
		// この変換は訳語が見つからなかった場合も行う。
		String ja = editMnemonicEnToJa(en, enNoMnemonic, jaNoMnemonic);

		return ja;
	}

	/**
	 * 指定した英語リソース文字列から日本語リソースを探します。
	 * ニーモニックは処理されません。
	 * <p>
	 * @param en 英語リソース文字列
	 * @param jointPoint ジョイント・ポイント
	 * @return 日本語リソース文字列
	 */
	public String lookupIgnoreMnemonic(String en, JointPoint jointPoint) {

		// (&A) のような日本用のニーモニックが既に含まれている場合は
		// 翻訳済みであるため、何もしない
		if (UnMnemonicProperties.hasJaMnemonic(en)) {
			return en;
		}

		// 翻訳不要な場合は、そのまま返す
		if (isNoTranslation(en, jointPoint)) {
			return en;
		}

		// 翻訳プロパティーから日本語訳を取得
		String ja = getValue(en);

		return ja;
	}

	//////////////////////////////////////////////////////////////////////////
	// ローカル・メソッド

	/**
	 * 翻訳が不要か判定します。
	 * 呼び出し元のパッケージやクラスを元に判定されます。
	 * <p>
	 * @param enNoMnemonic 英語リソース文字列（ニーモニック無し）
	 * @param jointPoint ジョイント・ポイント
	 * @return 翻訳が不要な場合は true
	 */
	protected boolean isNoTranslation(String enNoMnemonic, JointPoint jointPoint) {

		final int TRACE_MAX = 30;
		StackTraceElement[] stes = null;

		// xml [exclludeTrace] 呼び出し元トレースによる除外（再帰的に遡る）
		if (jointPoint != null) {

			PointCut pointCut = AspectMapping.getInstance().getPointCut(jointPoint);
			if (pointCut != null) {

				List<JointPoint> excludeTrace = pointCut.getExcludeTrace();
				if (excludeTrace.size() > 0) {

					stes = Thread.currentThread().getStackTrace();
					for (JointPoint jp : excludeTrace) {

						for (int i = 0; i < TRACE_MAX && i < stes.length; i++) {

							StackTraceElement ste = stes[i];
							String className = jp.getClassName();
							if (className.equals(ste.getClassName())) {

								String methodName = jp.getMethodName();
								if (methodName == null || methodName.equals(ste.getMethodName())) {
									return true;
								}
							}
						}
					}
				}
			}
		}

		// properties [%EXCULUDE%] 呼び出し元による除外（訳語とパッケージ単位）
		Set<String> noTransPathEntries = TranslationExcludeProperties
				.getInstance().getPathEntries(enNoMnemonic);

		if (noTransPathEntries != null) {
			if (stes == null) {
				stes = Thread.currentThread().getStackTrace();
			}
			for (int i = 0; i < TRACE_MAX && i < stes.length; i++) {

				StackTraceElement ste = stes[i];
				String className = ste.getClassName();

				for (String noTransPath : noTransPathEntries) {
					if (className.startsWith(noTransPath)) {
						return true;
					}
				}
			}
		}
		return false;
	}
}
