/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.generator;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Map.Entry;

import jp.sourceforge.mergedoc.pleiades.aspect.resource.RegexDictionary;
import jp.sourceforge.mergedoc.pleiades.aspect.resource.TranslationExcludeProperties;
import jp.sourceforge.mergedoc.pleiades.log.Logger;
import jp.sourceforge.mergedoc.pleiades.resource.AbstractTranslationDictionary;
import jp.sourceforge.mergedoc.pleiades.util.FastProperties;
import jp.sourceforge.mergedoc.pleiades.util.Files;
import jp.sourceforge.mergedoc.pleiades.util.UnMnemonicProperties;

/**
 * 複数のプロパティー・ファイルをマージし、1 つの
 * プロパティー・ファイル (translation.properties) を作成するための
 * ジェネレーターです。
 * <p>
 * @author cypher256
 */
public class Generator {

	/** ロガー */
	private static final Logger log = Logger.getLogger(Generator.class);

	/** eclipse.org 言語パックから抽出・カスタマイズ済みの辞書プロパティー (既存訳) のファイル名 */
	private static final String IN_NLS_CUSTOMIZED_PROP = "props/nls-customized.properties";

	/** 追加辞書プロパティー・ファイル名 */
	private static final String OUT_ADDITIONS_ALL_PROP = "props/additions-all.properties";

	/** 正規表現キープレフィックス */
	private static final String OUT_REGEX_KEY_PREFIX = "%REGEX%";

	/** 翻訳除外キープレフィックス */
	private static final String OUT_EXCLUDE_KEY_PREFIX = "%EXCLUDE%";
	
	/**
	 * ジェネレーターを開始するための main メソッドです。
	 * <p>
	 * @param args 起動引数
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static void main(String... args) throws IOException {
		new Generator().run();
	}

	/**
	 * ジェネレーターを実行します。
	 * <p>
	 * @throws IOException 入出力例外が発生した場合
	 */
	private void run() throws IOException {
		
		// カスタマイズ済み言語パック辞書プロパティー (既存訳)
		// 英文から & 除去。ニーモニックでなくても除去。&1 など。
		// これは実際に翻訳されるときに lookup メソッドで英分からまず & が除去されるため。
		// translation.properties の英文は & が付かない形式で格納されることに注意。
		// 日本語のほうはかっこがない限り除去されることはない。
		FastProperties inNlsCustomizedProp = new UnMnemonicProperties(IN_NLS_CUSTOMIZED_PROP);
		log.info("Loaded " + IN_NLS_CUSTOMIZED_PROP + " " + inNlsCustomizedProp.size());
		
		// 追加辞書プロパティー
		FastProperties outAdditionsAllProp = new FastProperties();
		// 正規表現辞書プロパティー
		FastProperties outRegexProp = new FastProperties();
		// 除外プロパティー
		FastProperties outExcludeProp = new FastProperties();
		
		// バリデーター
		Validator validator = new Validator(inNlsCustomizedProp);

		// 追加辞書の振り分け
		File[] files = Files.getResourceFile("props/additions").listFiles();
		Arrays.sort(files); // 昇順

		for (File file : files) {
			if (file.isFile() && file.getName().endsWith(".properties")) {

				FastProperties prop = new UnMnemonicProperties(file);
				log.debug("Loaded " + file.getName() + " " + prop.size());

				// % で始まる特殊プロパティを抽出し、別プロパティへ振り分け
				for (Entry<String, String> entry : prop.entrySet()) {

					String key = entry.getKey();
					String value = entry.getValue();

					if (key.startsWith(OUT_REGEX_KEY_PREFIX)) {

						// 正規表現プロパティ
						String newKey = key.replaceFirst("^" + OUT_REGEX_KEY_PREFIX, "");
						outRegexProp.put(newKey, value);

					} else if (key.startsWith(OUT_EXCLUDE_KEY_PREFIX)) {

						// 翻訳除外プロパティ
						String newKey = key.replaceFirst("^" + OUT_EXCLUDE_KEY_PREFIX, "");
						Object existsValue = outExcludeProp.get(newKey);
						if (existsValue != null) {
							value = existsValue + "," + value;
						}
						outExcludeProp.put(newKey, value);

					} else {

						// 訳語の検証
						validator.validate(file.getName(),
								key, value);

						// 翻訳辞書
						outAdditionsAllProp.put(key, value);
					}
				}
			}
		}
		
		// エラーがある場合は終了
		validator.abortHasError();
		
		// 追加辞書を保管 (参照用)
		outAdditionsAllProp.store(OUT_ADDITIONS_ALL_PROP,
			"追加辞書プロパティー\n\n" +
			"  目視参照用にプラグイン別の追加辞書をマージしたものです。\n" +
			"  システム的には不要なファイルです。\n" +
			"  入力元ファイル：conf/props/addtions/*.properties");

		// マージ (重複がないことをバリデーターでチェック済み)
		FastProperties resultProp = new FastProperties();
		resultProp.putAll(inNlsCustomizedProp);
		resultProp.putAll(outAdditionsAllProp);

		// 保管
		resultProp.store(AbstractTranslationDictionary.PROP_FILE_NAME,
			"翻訳辞書プロパティー\n\n" +
			"  言語パック辞書と全追加辞書をマージしたもので、Pleiades が実行時に参照します。\n" +
			"  入力元ファイル：nls-customized.properties、conf/props/addtions/*.properties");
		outRegexProp.store(RegexDictionary.PROP_FILE_NAME,
			"正規表現辞書プロパティー\n\n" +
			"  正規表現で翻訳するための辞書で、Pleiades が実行時に参照します。\n" +
			"  入力元ファイル：conf/props/addtions/*.properties のキー先頭に " + OUT_REGEX_KEY_PREFIX + " がある項目");
		outExcludeProp.store(TranslationExcludeProperties.PROP_FILE_NAME,
			"翻訳除外プロパティー\n\n" +
			"  翻訳を Java パッケージ単位で除外訳するための辞書で、Pleiades が実行時に参照します。\n" +
			"  入力元ファイル：conf/props/addtions/*.properties のキー先頭に " + OUT_EXCLUDE_KEY_PREFIX + " がある項目");

		// ログ出力
		String msg = "%-35s %5d";
		log.info(String.format(msg, OUT_ADDITIONS_ALL_PROP,
				outAdditionsAllProp.size()));
		log.info(String.format(msg, IN_NLS_CUSTOMIZED_PROP,
				inNlsCustomizedProp.size()));
		log.info(String.format(msg, AbstractTranslationDictionary.PROP_FILE_NAME,
				resultProp.size()));
		log.info(String.format(msg, RegexDictionary.PROP_FILE_NAME,
				outRegexProp.size()));
		log.info(String.format(msg, TranslationExcludeProperties.PROP_FILE_NAME,
				outExcludeProp.size()));
	}
}
