/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.generator;

import java.io.File;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import jp.sourceforge.mergedoc.pleiades.log.FileLogger;
import jp.sourceforge.mergedoc.pleiades.log.Logger;
import jp.sourceforge.mergedoc.pleiades.util.FastProperties;
import jp.sourceforge.mergedoc.pleiades.util.Files;
import jp.sourceforge.mergedoc.pleiades.util.UnMnemonicProperties;

/**
 * 翻訳バリデーター・クラスです。
 * 訳語を検証し、結果をログ出力します。
 * <p>
 * @author cypher256
 */
public class Validator {

	/** 検証無視セット */
	private static final Set<String> vIgnoreSet =
		new FastProperties("props/validation-ignore.properties").keySet();

	/** 翻訳禁止セット */
	private static final Set<String> vProhibitionSet =
		new FastProperties("props/validation-prohibition.properties").keySet();

	/** 用語検証プロパティー */
	private static final FastProperties vTermProp =
		new FastProperties("props/validation-term.properties");

	/** 対訳正規表現セット */
	private static final FastProperties vTranslationProp =
		new FastProperties("props/validation-translation.properties");

	/** 検証フラグ (true:検証する) */
	private static final boolean isValidate = true;

	/** 既存訳 nlsCustomizedProp 検証フラグ (true:検証する) */
	private static final boolean isValidateNlsCustomizedProp = false;

	/** 検証済みのプロパティー */
	private static class ValidatedItem {
		/** ファイル名などの識別子 */
		public String name;
		/** 日本語文字列 */
		public String ja;
	}

	/** ロガー */
	private Logger log = Logger.getLogger(Validator.class);

	/** eclipse.org 言語パックから抽出・カスタマイズ済みの辞書プロパティー (既存訳) */
	private final FastProperties nlsCustomizedProp;

	/** 検証済みのプロパティーのマップ (キー：英語文字列) */
	private final Map<String, ValidatedItem> validatedMap = new HashMap<String, ValidatedItem>();

	/** エラー数 */
	private int errorCount;

	/**
	 * コンストラクタ。
	 * @param nlsCustomizedProp
	 * 	eclipse.org 言語パックから抽出・カスタマイズ済みの辞書プロパティー (既存訳)
	 */
	public Validator(FastProperties nlsCustomizedProp) {

		this.nlsCustomizedProp = nlsCustomizedProp;

		if (isValidateNlsCustomizedProp) {

			// ロガーをファイル出力に変更して初期化
			File logFile = Files.getResourceFile("props/temp/nls-customized_validation.log");
			System.setProperty(FileLogger.LOG_FILE_NAME, logFile.getAbsolutePath());
			System.setProperty(Logger.LOGGER_CLASS_NAME, FileLogger.class.getName());
			System.setProperty(Logger.LOG_LEVEL, Logger.Level.DEBUG.toString());
			log = Logger.getLogger(Validator.class);

			// 言語パック全プロパティーの検証
			for (Entry<String, String> entry : nlsCustomizedProp.entrySet()) {

				String en = entry.getKey();
				String ja = entry.getValue();

				// 訳語を検証
				validateInternal("nls-customized.properties", en, ja);
			}
		}
	}

	/**
	 * 訳語を検証します。
	 * @param name ファイル名などの識別子
	 * @param en 英語文字列
	 * @param ja 日本語文字列
	 */
	public void validate(String name, String en, String ja) {

		if (!isValidate) {
			return;
		}
		StringBuilder message = new StringBuilder();

		// 言語パック辞書重複チェック
		String nlsJa = nlsCustomizedProp.get(en);
		if (nlsJa != null) {

			if (nlsJa.equals(ja)) {

				message.append(
					"既存の校正済み言語パック辞書にまったく同一の訳が存在します。" +
					"このエントリーを削除してください。\n");
			} else {

				message.append(
					"既: " + nlsJa + "\n" +
					"既存の校正済み言語パック辞書に既に訳が存在します。" +
					"既存訳を確認してください。\n");
			}
			// エラーログの出力
			error(name, en, ja, message);
			return;
		}

		// 追加辞書同士の重複チェック
		ValidatedItem validatedItem = validatedMap.get(en);
		if (validatedItem != null) {

			if (ja.equals(validatedItem.ja)) {

				message.append(
					"既存の追加辞書 " + validatedItem.name + " にまったく同じ訳が既に存在します。" +
					"このエントリーは不要なため削除してください。\n");
			} else {

				message.append(
					"既: " + validatedItem.ja + "\n" +
					"既存の追加辞書に異なる訳が既に存在します。" + validatedItem.name + "\n");
			}
			// エラーログの出力
			error(name, en, ja, message);
			return;

		} else {

			validatedItem = new ValidatedItem();
			validatedItem.name = name;
			validatedItem.ja = ja;
			validatedMap.put(en, validatedItem);
		}

		// 訳語を検証
		validateInternal(name, en, ja);
	}

	/**
	 * 訳語を検証します。
	 * @param name ファイル名などの識別子
	 * @param en 英語文字列
	 * @param ja 日本語文字列
	 */
	protected void validateInternal(String name, String en, String ja) {

		// 検証除外セットに含まれる場合は何もしない
		if (vIgnoreSet.contains(en)) {
			return;
		}

		String e = en.trim();
		String j = ja.trim();
		StringBuilder message = new StringBuilder();

		//---------------------------------------------------------------------
		// 翻訳忘れチェック
		//---------------------------------------------------------------------

		if (j.equals("") || j.equals(en)) {
			message.append("翻訳されていません。\n");
		}

		//---------------------------------------------------------------------
		// 末尾の同一性チェック
		//---------------------------------------------------------------------

		// 末尾の「...」チェック
		else if (e.endsWith("...") && !j.endsWith("...")) {
			// 「...」がその他複数を示す場合は「、、、」があれば OK
			if (!j.contains("、、、")) {
				message.append("原文の末尾に「...」がある場合は、訳文の末尾にも「...」が必要です。\n");
			}
		}

		// 末尾の「..」チェック
		else if (e.endsWith("..") && !j.endsWith("..")) {
			message.append("原文の末尾に「..」がある場合は、訳文の末尾にも「..」が必要です。\n");
		}

		// 末尾の「.」チェック
		else if (!e.endsWith("..") && e.endsWith(".") && !j.endsWith("。")) {

			// ・1 単語の場合はチェックしない
			// ・途中に「:」が含まれる場合はチェックしない
			// 例）Incompatible\ file\ format.\ Workspace\ was\ saved\ with\ an\ incompatible\ version\:\ {0}.
			if (e.contains(" ") && !e.contains(":")) {
				message.append("原文末尾にある「.」が句点を表す場合は、訳文の末尾にも「。」が必要です。\n");
			}
		}

		// 末尾の「:」チェック
		else if (e.endsWith(":") && !j.endsWith(":")) {
			message.append("原文の末尾に「:」がある場合は、訳文の末尾に「:」が必要です。\n");
		}

		//---------------------------------------------------------------------
		// 英数字隣接空白チェック
		//---------------------------------------------------------------------

		final String IGNORE_CHAR = "\\p{ASCII}（）「」、。…";
		Matcher mat = Pattern.compile("[^" + IGNORE_CHAR + "\\(]\\p{Alnum}").matcher(j);
		while (mat.find()) {
			message.append("英数字に隣接する全角文字の間に半角スペースが必要です。→"
					+ mat.group() + "\n");
		}
		mat = Pattern.compile("\\p{Alnum}[^" + IGNORE_CHAR + "\\)]").matcher(j);
		while (mat.find()) {
			message.append("英数字に隣接する全角文字の間に半角スペースが必要です。→"
					+ mat.group() + "\n");
		}

		//---------------------------------------------------------------------
		// 埋め込み引数の出現数チェック
		//---------------------------------------------------------------------

		// {0} などの埋め込み数チェック
		mat = Pattern.compile("\\{[0-9]\\}").matcher(e);
		while (mat.find()) {
			String group = mat.group();
			if (!j.contains(group)) {
				message.append("原文に含まれる " + group + " が訳文にも必要です。\n");
			}
		}

		// 改行数チェック
		int eCount = en.length() - en.replace("\n", "").length();
		int jCount = ja.length() - ja.replace("\n", "").length();
		if (eCount != jCount && eCount < 50) {
			message.append("改行文字 \\n の数が異なります。原文:" + eCount + " 訳文:" + jCount + "\n");
		}

		//---------------------------------------------------------------------
		// 二重クォート数チェック
		//---------------------------------------------------------------------
		
		if (j.contains("''")) {
			int jqc = (" " + j + " ").split("'").length - 1;
			int eqc = (" " + e + " ").split("'").length - 1;
			if (jqc % 2 == 1 && jqc != eqc) {
				message.append("シングル・クォートの数を確認してください。\n");
			}
		}
		
		if (j.contains("\"\"")) {
			int jqc = (" " + j + " ").split("\"").length - 1;
			int eqc = (" " + e + " ").split("\"").length - 1;
			if (jqc % 2 == 1 && jqc != eqc) {
				message.append("ダブル・クォートの数を確認してください。\n");
			}
		}

		//---------------------------------------------------------------------
		// 用語チェック
		//---------------------------------------------------------------------

		for (Entry<String, String> term : vTermProp.entrySet()) {

			String ng = term.getKey();
			String ok = term.getValue();
			
			if (ja.matches(".*?" + ng + ".*")) {
				if (ok.equals("")) {
					message.append("「" + ng + "」は使用禁止です。\n");
				} else {
					message.append("「" + ng + "」は「" + ok + "」である必要があります。\n");
				}
			}
		}
		
		//---------------------------------------------------------------------
		// 対訳正規表現チェック
		//---------------------------------------------------------------------

		for (Entry<String, String> trans : vTranslationProp.entrySet()) {

			String enPart = trans.getKey();
			String jaPart = trans.getValue();

			if (en.matches(enPart) && !ja.matches(jaPart)) {
				message.append("「" + enPart + "」の訳は「" + jaPart + "」である必要があります。\n");
			}
		}

		//---------------------------------------------------------------------
		// 翻訳禁止チェック
		//---------------------------------------------------------------------

		if (vProhibitionSet.contains(en)) {

			message.append(
				"この訳は Eclipse の動作に問題が発生するため、翻訳が許可されていません。" +
				"このエントリーを削除してください。\n");
		}

		// エラーログの出力
		error(name, en, ja, message);
	}

	/**
	 * エラーログを出力します。
	 * @param name ファイル名などの識別子
	 * @param en 英語文字列
	 * @param ja 日本語文字列
	 * @param message メッセージ
	 */
	protected void error(String name, String en, String ja, StringBuilder message) {

		if (message.length() == 0) {
			return;
		}
		String m =
			name + "\n" +
			"英: " + UnMnemonicProperties.toPropertyKey(en) + "\n" +
			"日: " + UnMnemonicProperties.toPropertyValue(ja) + "\n" +
			message;

		String indent = "      ";
		m = m.replace("\n", "\n" + indent);

		log.error(m);
		errorCount++;
	}

	/**
	 * エラーがあればエラー件数をログ出力し、終了します。
	 */
	public void abortHasError() {

		if (errorCount > 0) {
			log.error("辞書の検証で " + errorCount + " 件のエラーが検出されました。");
			System.exit(-1);
		}
	}
}
