/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.generator.nls;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import jp.sourceforge.mergedoc.pleiades.log.Logger;
import jp.sourceforge.mergedoc.pleiades.util.FastProperties;
import jp.sourceforge.mergedoc.pleiades.util.Files;
import jp.sourceforge.mergedoc.pleiades.util.UnMnemonicProperties;

import org.apache.commons.lang.StringUtils;

/**
 * Eclipse 3.2 まで eclipse.org から配布されてきた言語パックから、
 * 日本語訳を抽出し、Pleiades 形式のプロパティー・ファイルに出力します。
 * <p>
 * @author cypher256
 */
public class Extractor {

	/** ロガー */
	private static final Logger log = Logger.getLogger(Extractor.class);

	/** NLS 取得元プラグイン確認デバッグ・ログ出力用の英語リソース文字列 */
	private static String debugEnValue;

	/**
	 * 抽出を開始するための main メソッドです。
	 * <p>
	 * @param args NLS 取得元プラグイン確認デバッグ・ログ出力用の英語リソース文字列
	 * @throws IOException 入出力例外が発生した場合
	 */
	public static void main(String... args) throws IOException {
		
		if (args.length != 0) {
			debugEnValue = args[0].toLowerCase();
			log.info("引数に指定された [" + debugEnValue +
					"] の NLS 取得元プラグインを確認します。プロパティーは出力されません。");
		}
		new Extractor().run();
	}

	/** 言語パックとそのプラグイン本体が格納されたフォルダー */
	private final File inFolder = Files.getResourceFile("../nls");
	
	/** 出力先フォルダー */
	private final File outFolder = Files.getResourceFile("props/nls");

	/** NLS マージ時に日本語プロパティーが無いエントリーを格納するプロパティー */
	private final FastProperties nlsMissingProp = new UnMnemonicProperties();

	/** エントリー数 */
	private int count;
	
	/**
	 * 抽出を実行します。
	 * <p>
	 * @throws IOException 入出力例外が発生した場合
	 */
	private void run() throws IOException {
		
		// デバッグ文字列が設定されていない場合
		if (StringUtils.isBlank(debugEnValue)) {
			
			// 出力先フォルダーのプロパティー・ファイルをすべて削除
			// (.svn を削除しないように注意)
			File[] files = outFolder.listFiles();
			for (File file : files) {
				if (file.getPath().endsWith(".properties")) {
					file.delete();
				}
			}
		}
		
		// 言語パックからプロパティーを抽出し、出力先フォルダーに出力
		generate(inFolder);
		
		// 抽出したプロパティーをロードし、翻訳ルールを適用
		FastProperties nlsAllProp = new UnMnemonicProperties("props/nls");
		Optimizer optimizer = new Optimizer();
		nlsAllProp = optimizer.apply(nlsAllProp);

		// 出力するプロパティー・ファイル名
		String nlsAllPropPath = "props/temp/nls-all.properties";
		String nlsMissingPropPath = "props/temp/nls-all-missing.properties";
		
		// デバッグ文字列が設定されていない場合は出力
		if (StringUtils.isBlank(debugEnValue)) {

			// 1 つのプロパティー・ファイルとして保管
			nlsAllProp.store(nlsAllPropPath,
				"eclipse.org 言語パック抽出プロパティー\n\n" +
				"  プラグイン別の言語パック辞書をマージしたものです。\n" +
				"  入力元ファイル：conf/props/nls/*.properties");
			
			// 訳が見つからなかったプロパティーを保管
			nlsMissingProp.store(nlsMissingPropPath,
				"eclipse.org 言語パック抽出未翻訳プロパティー\n\n" +
				"  言語パック抽出時に訳が見つからなかった項目のプロパティーです。\n" +
				"  入力元ファイル：nls/*/*/*");
		}
		
		log.info(nlsAllPropPath + " " + nlsAllProp.size() + " (" + count + ")");
		log.info(nlsMissingPropPath + " " + nlsMissingProp.size());
	}

	/**
	 * 指定されたフォルダーに存在する言語パックを再帰的にロードし、
	 * プロパティーを生成します。
	 * <p>
	 * @param nlsFolder NLS フォルダー
	 * @throws IOException 入出力例外が発生した場合
	 */
	private void generate(File nlsFolder) throws IOException {

		File[] files = nlsFolder.listFiles();
		Arrays.sort(files); // 昇順

		for (File file : files) {

			if (file.isFile()) continue;
			String name = file.getName();
			if (name.equals("plugins") || name.equals("features")) {
				generateProperties(file);
			} else {
				generate(file); // 再帰
			}
		}
	}

	/**
	 * 指定されたプラグイン・フォルダーに存在する言語パックをロードし、
	 * プロパティーを生成します。
	 * <p>
	 * @param pluginFolder プラグイン・フォルダー
	 * @throws IOException 入出力例外が発生した場合
	 */
	private void generateProperties(File pluginFolder) throws IOException {

		Map<String, PropertiesUnit> pluginMap =
			new HashMap<String, PropertiesUnit>();

		File[] files = pluginFolder.listFiles();
		Arrays.sort(files); // 昇順

		// 日本語と英語のプロパティーをロード
		for (File file : files) {

			Plugin plugin = (pluginFolder.getName().equals("plugins"))
				? new Plugin(file) : new Feature(file);

			String msg = "%-46s en:%4d ja:%4d %s";
			log.debug(String.format(msg,
					plugin.getId(),
					plugin.getEnProperties().size(),
					plugin.getJaProperties().size(),
					Files.relativePath(inFolder, file)));

			String pluginId = plugin.getId();
			PropertiesUnit propUnit = pluginMap.get(pluginId);
			if (propUnit == null) {
				propUnit = new PropertiesUnit();
				pluginMap.put(pluginId, propUnit);
			}
			propUnit.getEnProperties().putAll(plugin.getEnProperties());
			propUnit.getJaProperties().putAll(plugin.getJaProperties());
		}

		String pluginPath = Files.relativePath(inFolder, pluginFolder);

		// 日本語と英語のプロパティーでキーが一致するものからプロパティー作成
		for (Entry<String, PropertiesUnit> entry : pluginMap.entrySet()) {

			PropertiesUnit propUnit = entry.getValue();
			FastProperties enProp = propUnit.getEnProperties();
			FastProperties jaProp = propUnit.getJaProperties();
			FastProperties outProp = new UnMnemonicProperties();

			for (Entry<?, ?> jaEntry : jaProp.entrySet()) {

				String key = (String) jaEntry.getKey();
				String enValue = (String) enProp.get(key);
				String jaValue = (String) jaEntry.getValue();

				if (StringUtils.isNotEmpty(enValue) && !enValue.equals(jaValue)) {

					if (jaValue.matches("\\p{ASCII}+")) {
						continue;
					}
					// 原文にない NLS 固有の訳注が含まれているものは除外
					if (jaValue.contains("の翻訳版です") || jaValue.contains("のためのドイツ語")) {
						continue;
					}
					outProp.put(enValue, jaValue);

					String v = UnMnemonicProperties.removeEnMnemonic(enValue).toLowerCase();
					if (StringUtils.isNotBlank(debugEnValue) && v.contains(debugEnValue)) {
						
						String pluginBaseName = pluginFolder.getPath().replace("\\", "/");
						pluginBaseName = pluginBaseName.replaceFirst(".+/([^/]+?/[^/]+?)", "$1");
						log.info("取得元プラグイン: " +
								pluginBaseName + "/" + key + "=" + jaValue);
					}
				}
			}

			// プロパティー・ファイルへ出力
			int outPropSize = outProp.size();
			if (outPropSize > 0) {
				
				count += outPropSize;
				
				// デバッグ文字列が設定されていない場合は出力
				if (StringUtils.isBlank(debugEnValue)) {
					
					String pluginId = entry.getKey();
					String outFileName = pluginPath.replace('\\', '@') + "@" + pluginId + ".properties";
					File outFile = new File(outFolder, outFileName);
					outProp.store(outFile, "eclipse.org 言語パック抽出プロパティー (プラグイン別)");
					log.info(outFileName + " " + outProp.size());
				}
			}

			// 日本語プロパティが存在しない場合のプロパティー作成
			for (Entry<?, ?> enEntry : enProp.entrySet()) {

				String jaValue = (String) jaProp.get(enEntry.getKey());
				String enValue = enEntry.getValue().toString();

				if (jaValue == null &&
					enValue.matches(".*[a-zA-Z].*") &&
					enValue.length() > 1) {

					nlsMissingProp.put(enValue, "");
				}
			}
		}
	}
}
