/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.generator.nls;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.jar.Attributes;
import java.util.jar.Manifest;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import org.apache.commons.io.IOUtils;

/**
 * プラグイン・リソースを保持するクラスです。
 * <p>
 * @author cypher256
 */
public class Plugin extends PropertiesUnit {

	/** プラグイン・ルート */
	private final File root;

	/** プラグイン ID */
	private String id;

	/**
	 * プラグインを構築します。
	 * <p>
	 * @param pluginFile プラグイン・フォルダーまたはファイル
	 */
	public Plugin(File pluginFile) {

		this.root = pluginFile;
		if (pluginFile.isDirectory()) {
			loadFolder(pluginFile);
		} else {
			loadJar(pluginFile);
		}

		if (id == null) {
			throw new NullPointerException(
					"プラグイン ID が取得できませんでした。"
					+ pluginFile.getName());
		}
	}

	/**
	 * 指定されたフォルダーのリソースをロードします。
	 * <p>
	 * @param pluginFolder プラグイン・フォルダー
	 */
	protected void loadFolder(File pluginFolder) {

		InputStream is = null;
		try {

			File manifest = new File(pluginFolder, "META-INF/MANIFEST.MF");
			if (manifest.exists()) {

				// マニフェスト・ファイルがあれば、それからプラグイン ID を取得
				is = new BufferedInputStream(new FileInputStream(manifest));
				setIdInManifestFile(is);

			} else {

				File fragmentXML = new File(pluginFolder, "fragment.xml");
				if (fragmentXML.exists()) {

					// フラグメント XML があれば、それからプラグイン ID を取得
					is = new BufferedInputStream(new FileInputStream(fragmentXML));
					setIdInFragmentXML(is);

				} else {

					// フォルダー名からプラグイン ID を取得
					setIdInFileName(pluginFolder);
				}
			}

		} catch (IOException e) {
			throw new IllegalArgumentException(e);

		} finally {
			IOUtils.closeQuietly(is);
		}

		getEnProperties().load(new File(pluginFolder, "plugin.properties"));
		getJaProperties().load(new File(pluginFolder, "plugin_ja.properties"));
		getJaProperties().load(new File(pluginFolder, "plugin_ja_JP.properties"));

		File[] files = pluginFolder.listFiles();
		for (File file : files) {
			if (file.isFile() && file.getName().endsWith(".jar")) {
				loadJar(file);
			}
		}
	}

	/**
	 * 指定された jar ファイルをロードします。
	 * <p>
	 * @param jar jar ファイル
	 */
	private void loadJar(File jar) {

		// jar ファイル名からプラグイン ID を取得
		setIdInFileName(jar);

		ZipInputStream in = null;
		try {
			in = new ZipInputStream(new BufferedInputStream(new FileInputStream(jar)));

            for (ZipEntry inEntry = null; (inEntry = in.getNextEntry()) != null;) {

                String entryName = inEntry.getName();

                if (entryName.equals("META-INF/MANIFEST.MF")) {
                	// プラグイン ID のセット（マニフェストファイル）
                	setIdInManifestFile(in);

                } else if (entryName.equals("fragment.xml")) {
                	// プラグイン ID のセット（フラグメント XML）
                	setIdInFragmentXML(in);

                } else if (entryName.endsWith("_ja.properties") ||
                		   entryName.endsWith("_ja_JP.properties")) {
                	// 日本語プロパティ
                    getJaProperties().load(in);

                } else if (entryName.matches("^.*_\\w{2}\\.properties$")) {
                	// 英語・日本語以外のプロパティ

                } else if (entryName.endsWith(".properties")) {
                	// 英語プロパティ
                	String rootName = root.getName();
                	if (!rootName.matches("^.*?(\\.nls1_|\\.nl1_).*$")) {
                		// ja で上書きされていた場合の対応（_ja があることが前提）
                        getEnProperties().load(in);
                	}
                }
            }

		} catch (IOException e) {
			throw new IllegalStateException(e);

		} finally {

			IOUtils.closeQuietly(in);
		}
	}

	/**
	 * マニフェスト・ファイル入力ストリームからプラグイン ID をセットします。
	 * <p>
	 * @param is マニフェスト・ファイル入力ストリーム
	 * @throws IOException 入出力例外が発生した場合
	 */
	private void setIdInManifestFile(InputStream is) throws IOException {

		if (getId() != null) return;
		Attributes attributes = new Manifest(is).getMainAttributes();
		String value = attributes.getValue("Fragment-Host");
		if (value == null) {
			value = attributes.getValue("Bundle-SymbolicName");
		}
		if (value != null) {
			setId(value.split(";")[0].trim());
		}
	}

	/**
	 * フラグメント XML 入力ストリームからプラグイン ID をセットします。
	 * <p>
	 * @param is フラグメント XML ファイル入力ストリーム
	 * @throws IOException 入出力例外が発生した場合
	 */
	private void setIdInFragmentXML(InputStream is) throws IOException {

		if (getId() != null) return;
		String fragment = IOUtils.toString(is);
		setId(fragment.replaceFirst("(?s)^.*plugin-id=\"(.*?)\".*$", "$1"));
	}

	/**
	 * プラグイン・フォルダーまたはファイル名からプラグイン ID をセットします。
	 * <p>
	 * @param pluginFolder プラグイン・フォルダーまたはファイル名
	 */
	protected void setIdInFileName(File pluginFolder) {
		if (getId() != null) return;
		String name = pluginFolder.getName();
		name = name.replaceFirst("(\\.nls1|\\.nl1)", "");
		name = name.replaceFirst("_[0-9]+\\.[0-9]+\\.[0-9]+.*$", "");
		setId(name);
	}

	/**
	 * プラグイン ID をセットします。
	 * <p>
	 * @param id プラグイン ID
	 */
	public void setId(String id) {
		this.id = id;
	}

	/**
	 * プラグイン ID を取得します。
	 * <p>
	 * @return プラグイン ID
	 */
	public String getId() {
		return id;
	}
}
