/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.log;

import java.io.PrintStream;
import java.lang.reflect.Constructor;
import java.util.Date;

/**
 * Log4J ライクなインタフェースを持つ抽象ロガーです。
 * <p>
 * @author cypher256
 */
public abstract class Logger {

	/** ロガー・クラス名を示すシステム・プロパティーのキー */
	public static final String LOGGER_CLASS_NAME = "pleiades.logger.class.name";

	/** ログ・レベルを示すシステム・プロパティーのキー */
	public static final String LOG_LEVEL = "pleiades.log.level";

	/** ログ・レベル列挙型 */
	public static enum Level {
		/** デバッグ */
		DEBUG,
		/** 情報 */
		INFO,
		/** 警告 */
		WARN,
		/** エラー */
		ERROR,
		/** 致命的 */
		FATAL
	}

	/** 最小出力ログ・レベル */
	private static volatile Level minLevel;

	/** このロガーを生成したクラス名 */
	private final String className;

	/**
	 * ロガーを取得します。
	 * <p>
	 * @param clazz クラス
	 * @return ロガー
	 */
	public static Logger getLogger(Class<?> clazz) {

		synchronized (Logger.class) {

			if (minLevel == null) {
				String logLevel = System.getProperty(LOG_LEVEL, Level.INFO.toString());
				try {
					minLevel = Level.valueOf(logLevel.toUpperCase());
				} catch (IllegalStateException e) {
					throw new IllegalStateException(
							"システムプロパティ " + LOG_LEVEL + "=" + logLevel +
							" に指定する値は " + Level.class.getName() +
							"で定義された型である必要があります。");
				}
				System.out.println(LOG_LEVEL + "=" + logLevel);
			}
		}

		String loggerClassName =
			System.getProperty(LOGGER_CLASS_NAME, SystemOutLogger.class.getName());
		try {

			Constructor<?> cons = Class.forName(loggerClassName).getDeclaredConstructor(Class.class);
			cons.setAccessible(true);
			return (Logger) cons.newInstance(clazz);

		} catch (Exception e) {
			throw new IllegalStateException(
					"システムプロパティ " + LOGGER_CLASS_NAME + "=" +
					loggerClassName +
					" に指定されたロガークラスがロードできませんでした。", e);
		}
	}

	/**
	 * ロガーを構築します。
	 * <p>
	 * @param clazz クラス
	 */
	protected Logger(Class<?> clazz) {
		this.className = clazz.getName().replaceFirst("(\\w+\\.)+", "");
	}

	/**
	 * 出力先となる PrintStream を取得します。
	 * <p>
	 * @return 出力先となる PrintStream
	 */
	abstract protected PrintStream getOut();

	/**
	 * メッセージを出力します。
	 * <p>
	 * @param level ログ・レベル
	 * @param message メッセージ
	 * @param t 例外。null の場合は message のみを出力。
	 */
	protected void println(Level level, String message, Throwable t) {

		Date date = new Date();
		String threadName = Thread.currentThread().getName();
		String s = String.format(
				"%-5s %tT.%tL [%s] (%s) %s",
				level, date, date, threadName, className, message);

		PrintStream out = getOut();
		out.println(s);
		if (t != null) {
			t.printStackTrace(out);
		}
	}

	/**
	 * デバッグ・ログを出力します。
	 * <p>
	 * @param message メッセージ
	 */
	public void debug(String message) {
		debug(message, null);
	}

	/**
	 * デバッグ・ログを出力します。
	 * <p>
	 * @param message メッセージ
	 * @param e 例外
	 */
	public void debug(String message, Throwable e) {
		if (isDebugEnabled()) println(Level.DEBUG, message, e);
	}

	/**
	 * 情報ログを出力します。
	 * <p>
	 * @param message メッセージ
	 */
	public void info(String message) {
		info(message, null);
	}

	/**
	 * 情報ログを出力します。
	 * <p>
	 * @param message メッセージ
	 * @param e 例外
	 */
	public void info(String message, Throwable e) {
		if (isInfoEnabled()) println(Level.INFO, message, e);
	}

	/**
	 * 警告ログを出力します。
	 * <p>
	 * @param message メッセージ
	 */
	public void warn(String message) {
		warn(message, null);
	}

	/**
	 * 警告ログを出力します。
	 * <p>
	 * @param message メッセージ
	 * @param e 例外
	 */
	public void warn(String message, Throwable e) {
		if (isWarnEnabled()) println(Level.WARN, message, e);
	}

	/**
	 * エラー・ログを出力します。
	 * <p>
	 * @param message メッセージ
	 */
	public void error(String message) {
		error(message, null);
	}

	/**
	 * エラー・ログを出力します。
	 * <p>
	 * @param message メッセージ
	 * @param e 例外
	 */
	public void error(String message, Throwable e) {
		if (isErrorEnabled()) println(Level.ERROR, message, e);
	}

	/**
	 * 致命的エラー・ログを出力します。
	 * <p>
	 * @param message メッセージ
	 */
	public void fatal(String message) {
		fatal(message, null);
	}

	/**
	 * 致命的エラー・ログを出力します。
	 * <p>
	 * @param message メッセージ
	 * @param e 例外
	 */
	public void fatal(String message, Throwable e) {
		if (isFatalEnabled()) println(Level.FATAL, message, e);
	}

	/**
	 * デバッグ・ログが有効か判定します。
	 * <p>
	 * @return 有効な場合は true
	 */
	public boolean isDebugEnabled() {
		return	minLevel == Level.DEBUG;
	}

	/**
	 * 情報ログが有効か判定します。
	 * <p>
	 * @return 有効な場合は true
	 */
	public boolean isInfoEnabled() {
		return	minLevel == Level.DEBUG || minLevel == Level.INFO ;
	}

	/**
	 * 警告ログが有効か判定します。
	 * <p>
	 * @return 有効な場合は true
	 */
	public boolean isWarnEnabled() {
		return	minLevel == Level.DEBUG || minLevel == Level.INFO  ||
				minLevel == Level.WARN ;
	}

	/**
	 * エラー・ログが有効か判定します。
	 * <p>
	 * @return 有効な場合は true
	 */
	public boolean isErrorEnabled() {
		return	minLevel == Level.DEBUG || minLevel == Level.INFO  ||
				minLevel == Level.WARN  || minLevel == Level.ERROR;
	}

	/**
	 * 致命的エラー・ログが有効か判定します。
	 * <p>
	 * @return 有効な場合は true
	 */
	public boolean isFatalEnabled() {
		return true;
	}
}
