/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.resource;

import java.io.File;
import java.io.FileNotFoundException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import jp.sourceforge.mergedoc.pleiades.aspect.Pleiades;
import jp.sourceforge.mergedoc.pleiades.aspect.resource.RegexDictionary;
import jp.sourceforge.mergedoc.pleiades.log.Logger;
import jp.sourceforge.mergedoc.pleiades.util.FastProperties;
import jp.sourceforge.mergedoc.pleiades.util.Files;
import jp.sourceforge.mergedoc.pleiades.util.UnMnemonicProperties;

/**
 * 抽象翻訳辞書クラスです。
 * <p>
 * @author cypher256
 */
abstract public class AbstractTranslationDictionary {

	/** ロガー */
	private static final Logger log = Logger.getLogger(AbstractTranslationDictionary.class);

	/** デフォルト翻訳プロパティー・ファイル名 */
	public static final String PROP_FILE_NAME = "translation.properties";

	/** 翻訳プロパティー追加ディレクトリー */
	private static final String ADDITIONS_DIRECTORY = "additions";

	/** 変換プロパティー・ファイル名 */
	public static final String CONVERTER_PROPERTIES_FILE_NAME = "translation-converter.properties";

	/** 特殊接尾文字 */
	@SuppressWarnings("serial")
	private static final Map<String, String> SPECIAL_SUFFIX_MAP =
		Collections.unmodifiableMap(new HashMap<String, String>(){
			{
				put("(Incubation)", "(インキュベーション)");
				put("(Experimental)", "(実験用)");
			}
		}
	);

	/** 翻訳プロパティー */
	private final FastProperties properties = new FastProperties();
	
	/**
	 * デフォルト辞書を構築します。
	 */
	protected AbstractTranslationDictionary() {
		load();
	}

	/**
	 * 辞書ファイルをロードします。
	 */
	protected void load() {

		File propFile = Files.getResourceFile(PROP_FILE_NAME);
		if (!propFile.exists()) {
			Exception e = new FileNotFoundException(propFile.getPath());
			Exception ise = new IllegalStateException("翻訳辞書が見つかりません。", e);
			Pleiades.abort(ise);
		}
		properties.load(propFile);

		File additions = Files.getResourceFile(ADDITIONS_DIRECTORY);
		if (additions.exists()) {

			File[] files = additions.listFiles();
			Arrays.sort(files); // 昇順

			for (File file : files) {
				if (file.isFile() && file.getName().endsWith(".properties")) {
					properties.load(file);
				}
			}
		}
		log.info("翻訳辞書をロードしました。" + properties.size());

		// 変換プロパティーを適用
		applyConverter();
	}

	/**
	 * 変換プロパティーを適用します。
	 */
	protected void applyConverter() {

		File convFile = Files.getResourceFile(CONVERTER_PROPERTIES_FILE_NAME);
		if (convFile.exists()) {

			FastProperties convMap = new FastProperties();
			convMap.load(convFile);

			for (Entry<String, String> entry : properties.entrySet()) {

				String key = entry.getKey();
				String value = entry.getValue();
				String resultValue = value;

				// 置換（例：本当に→ほんまに）
				for (Entry<String, String> e : convMap.entrySet()) {
					String convKey = e.getKey();
					String convValue = e.getValue();
					resultValue = resultValue.replaceAll(convKey, convValue);
				}
				if (!value.equals(resultValue)) {
					properties.put(key, resultValue);
				}
			}
		}
	}

	/**
	 * 翻訳プロパティーを保持するマップを取得します。
	 * @return 翻訳プロパティーを保持するマップ
	 */
	protected FastProperties getProperties() {
		return properties;
	}

	/**
	 * 日本語訳を取得します。
	 * <p>
	 * @param enNoMnemonic 英語リソース文字列（ニーモニック無し）
	 * @return 日本語リソース文字列（ニーモニック無し）。翻訳できない場合は enNoMnemonic。
	 */
	protected String getValue(String enNoMnemonic) {

		// 翻訳プロパティーから日本語訳を取得
		String result = properties.get(enNoMnemonic);

		// 翻訳プロパティーから取得できない場合
		if (result == null) {

			// trim して再度取得
			result = getValueToTrim(enNoMnemonic);
		}

		// 見つからない場合はそのまま返す
		if (result == null) {
			result = enNoMnemonic;
		}

		return result;
	}

	/**
	 * trim して日本語リソース文字列を取得します。
	 * <p>
	 * @param enNoMnemonic 英語リソース文字列（ニーモニック無し）
	 * @return 日本語リソース文字列（ニーモニック無し）。翻訳できない場合は null。
	 */
	protected String getValueToTrim(String enNoMnemonic) {

		// 先頭スペースを退避
		StringBuilder leading = new StringBuilder();
		char[] cArray = enNoMnemonic.toCharArray();
		for (int i = 0; i < cArray.length; i++) {
			char c = cArray[i];
			if (c == ' ' || c == '\t' || c == '\r' || c == '\n') {
				leading.append(c);
			} else {
				break;
			}
		}
		if (leading.length() == enNoMnemonic.length()) {
			return null;
		}

		// 後方スペース、:、... を退避
		StringBuilder trailing = new StringBuilder();
		for (int i = cArray.length - 1; i > 0; i--) {
			char c = cArray[i];
			if (c == ' ' || c == '\t' || c == '\r' || c == '\n' || c == ':') {
				trailing.insert(0, c);
			} else if (c == '.' && i > 2 && cArray[i - 1] == '.' && cArray[i - 2] == '.') {
				trailing.insert(0, "...");
				i -= 2;
			} else {
				break;
			}
		}

		// 前後スペースを除去
		String enTrimmed = enNoMnemonic.substring(
				leading.length(), enNoMnemonic.length() - trailing.length());

		// 再度、翻訳プロパティーから日本語訳を取得
		String result = properties.get(enTrimmed);

		// 正規表現プロパティーから日本語訳を取得
		if (result == null) {
			result = getValueByRegex(enTrimmed);
		}

		// 末尾が (Incubation) などの特殊文字列の場合は除去して辞書参照
		if (result == null && enTrimmed.endsWith(")")) {
			for (Entry<String, String> entry : SPECIAL_SUFFIX_MAP.entrySet()) {
				String en = entry.getKey();
				String ja = entry.getValue();
				if (enTrimmed.endsWith(en)) {
					String s = enTrimmed.replace(en, "").trim();
					result = properties.get(s);
					if (result != null) {
						result += " " + ja;
					}
				}
			}
		}
		
		// 前後スペースを復元
		if (result != null) {
			result = leading + result + trailing;
		}
		// 訳が見つからない場合のログ出力
		else {
			TranslationNotFoundProperties.getInstance().println(enNoMnemonic);
		}

		return result;
	}

	/**
	 * 正規表現辞書から日本語リソース文字列を取得します。
	 * <p>
	 * @param enTrimmed 英語リソース文字列（ニーモニック無し）
	 * @return 日本語リソース文字列（ニーモニック無し）。翻訳できない場合は null。
	 */
	protected String getValueByRegex(String enTrimmed) {
		return RegexDictionary.getInstance().lookup(enTrimmed);
	}

	/**
	 * ニーモニックを英語用から日本用に編集します。<br>
	 * 例）&x → (&X)
	 * <p>
	 * @param en				英語リソース文字列  （ニーモニック有り）
	 * @param enNoMnemonic		英語リソース文字列  （ニーモニック無し）
	 * @param jaNoMnemonic		日本語リソース文字列（ニーモニック無し）
	 * @return					日本語リソース文字列（ニーモニック有り）
	 */
	protected String editMnemonicEnToJa(String en,
			String enNoMnemonic, String jaNoMnemonic) {

		if (en.equals(enNoMnemonic)) {
			return jaNoMnemonic;
		}
		
		// 日本語リソースにニーモニック "&" が含まれる場合は英語側もニーモニックではないと判断。
		// "&&" はニーモニックではなく、"&" そのものを示すため除外。
		// そのまま処理してしまうと "K&R" が "K&R(&R)" のようになる。
		// translation.properties 上は英語リソース文字列に & はない。例："KR"
		if (jaNoMnemonic.contains("&") && !jaNoMnemonic.contains("&&")) {
			return jaNoMnemonic;
		}
		
		String mnemonicChar = en.replaceFirst(
				"(?s)^.*?\\&(" + UnMnemonicProperties.MNEMONIC_CHARS + ").*$",
				"$1");

		if (mnemonicChar.length() != 1) {
			log.error("Mnemonic invalid length: " + mnemonicChar.length());
			log.error(" enValue:            " + en);
			log.error(" enValueNonMnemonic: " + enNoMnemonic);
			log.error(" mnemonicChar:       " + mnemonicChar);
			return en;
		}
		String mnemonicJa = "(&" + mnemonicChar.toUpperCase() + ")";

		// 例）
		// hoge @Ctrl+X		-> hoge(&H) @Ctrl+X
		// '@Override'		-> '@Override'(&O)

		// modified 2008.03.29 &Left -> 左 のように訳語が 1 文字の場合に対応
		//String ja = jaNoMnemonic.replaceFirst("(?s)^(.{2,}?)(" +
		String ja = jaNoMnemonic.replaceFirst("(?s)^(.+?)(" +
				"(\\s*\\.{3,4}|)@\\p{ASCII}+|" +	// 例）末尾が @Ctrl+Shift+X
				"\\.{3,4}(</a>|)\\s*|" +			// 例）末尾が ... or ....
				"\\s*|" +							// 例）末尾が 空白
				":\\s*|" +							// 例）末尾が :空白
				":\\s+\\{[0-9]\\}\\s*|" +			// 例）末尾が :{数字}
				"(:|)\\s+\\(.+\\)\\s*|" +			// 例）末尾が :(文字列)
				"\\s+(-*|)>+\\s*" +					// 例）末尾が > or >> or -->
				")$", "$1" + mnemonicJa + "$2");

		return ja;
	}
}
