//******************************************************************************
//
// MIDITrail / MTFileViewCtrl
//
// ファイルビュー制御クラス
//
// Copyright (C) 2012-2018 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "YNBaseLib.h"
#import "MTFileViewCtrl.h"


//******************************************************************************
// プライベートメソッド定義
//******************************************************************************
@interface MTFileViewCtrl ()

//ファイル一覧生成
- (int)makeFileList;

//通知送信処理
- (int)postNotificationWithName:(NSString*)pName;

//テーブルセル作成：ファイル
- (UITableViewCell*)makeFileCellForIndexPath:(NSIndexPath*)indexPath;

//テーブルセル選択イベント：ファイル一覧
- (void)onSelectFileCellForIndexPath:(NSIndexPath*)indexPath;

//ファイル削除
- (int)removeFileWithPath:(NSString*)pPath;

//ファイルリネーム
- (int)renameFileWithPath:(NSString*)pSrcPath toPath:(NSString*)pDestPath;

@end


@implementation MTFileViewCtrl

//******************************************************************************
// 生成
//******************************************************************************
- (id)initWithNibName:(NSString *)nibNameOrNil bundle:(NSBundle *)nibBundleOrNil
{
    NSString* pFileEditViewNibName = nil;
	
	self = [super initWithNibName:nibNameOrNil bundle:nibBundleOrNil];
    if (self) {
		//ビュー設定
		self.title = @"MIDI Files";
		self.tabBarItem.image = [UIImage imageNamed:@"img/TabIcon"];
		
		//ファイル一覧生成
		[self makeFileList];
		
		//選択行番号
		m_SelectedFileIndex = 0;
		
		//ファイル編集ビュー制御の生成
		if ([[UIDevice currentDevice] userInterfaceIdiom] == UIUserInterfaceIdiomPhone) {
			//iPhone / iPod touch
			pFileEditViewNibName = @"MTFileEditView_iPhone";
		}
		else {
			//iPad
			pFileEditViewNibName = @"MTFileEditView_iPad";
		}
		m_pFileEditViewCtrl = [[MTFileEditViewCtrl alloc] initWithNibName:pFileEditViewNibName bundle:nil];
    }
    return self;
}

//******************************************************************************
// ビュー登録完了
//******************************************************************************
- (void)viewDidLoad
{
	NSNotificationCenter* pCenter = nil;
	
    [super viewDidLoad];
	
	// Do any additional setup after loading the view.
	
	//ナビゲーションバー右側に編集ボタンを登録
	self.navigationItem.rightBarButtonItem = [self editButtonItem];
	
	//通知先登録：ファイル名更新
	pCenter = [NSNotificationCenter defaultCenter];
	[pCenter addObserver:self
				selector:@selector(onUpdateFileName:)
					name:@"MTFileViewCtrl_onUpdateFileName"
				  object:nil];
	
	[m_pFileEditViewCtrl setNotificationNameForUpdateFileName:@"MTFileViewCtrl_onUpdateFileName"];
	
	return;
}

//******************************************************************************
// ビュー解除完了
//******************************************************************************
//  iOS6で廃止
//- (void)viewDidUnload
//{
//    [super viewDidUnload];
//
//    // Release any retained subviews of the main view.
//
//	return;
//}

//******************************************************************************
// インターフェース自動回転確認
//******************************************************************************
- (BOOL)shouldAutorotateToInterfaceOrientation:(UIInterfaceOrientation)interfaceOrientation
{
	return YES;
}

//******************************************************************************
// インターフェース自動回転確認（iOS6以降）
//******************************************************************************
- (BOOL)shouldAutorotate
{
	//回転を許可する
	return YES;
}

//******************************************************************************
// インターフェース自動回転確認：回転方向（iOS6以降）
//******************************************************************************
- (NSUInteger)supportedInterfaceOrientations
{
	//全方向に対応する
	return UIInterfaceOrientationMaskAll;
}

//******************************************************************************
// ビュー表示
//******************************************************************************
- (void)viewWillAppear:(BOOL)animated
{
	return;
}

//******************************************************************************
// ビュー非表示
//******************************************************************************
- (void)viewWillDisappear:(BOOL)animated
{
	return;
}

//******************************************************************************
// ファイル一覧生成
//******************************************************************************
- (int)makeFileList
{
	int result = 0;
	int i = 0;
	NSArray* pPathList = nil;
	NSString* pDocDirPath = nil;
	NSString* pPath = nil;
	NSString* pFile = nil;
	NSArray* pFileArrayTmp = nil;
	NSMutableArray* pFileArray = nil;
	BOOL isDir = NO;
	
	pFileArray = [[NSMutableArray alloc] init];
	
	//Documentsディレクトリパスを取得
	pPathList = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
	pDocDirPath = [pPathList objectAtIndex:0];
	
	//ディレクトリ配下のファイル一覧を取得
	pFileArrayTmp = [[NSFileManager defaultManager] contentsOfDirectoryAtPath:pDocDirPath error:nil];
	
	//ファイル一覧を作成（ディレクトリと画像ファイルは除外する）
	for (i = 0; i < [pFileArrayTmp count]; i++) {
		pFile = [pFileArrayTmp objectAtIndex:i];
		pPath = [pDocDirPath stringByAppendingPathComponent:pFile];
		[[NSFileManager defaultManager] fileExistsAtPath:pPath isDirectory:&isDir];
		if (!isDir) {
            if (([[pPath pathExtension] caseInsensitiveCompare:@"jpg"] == NSOrderedSame) ||
                ([[pPath pathExtension] caseInsensitiveCompare:@"png"] == NSOrderedSame) ||
				([[pPath pathExtension] caseInsensitiveCompare:@"dls"] == NSOrderedSame) ||
				([[pPath pathExtension] caseInsensitiveCompare:@"sf2"] == NSOrderedSame)) {
                //画像ファイルとWavetableファイルは無視する
                //NSLog(@"%@", pFile);
            }
            else {
                [pFileArray addObject:pFile];
            }
		}
	}
	
	//ファイル名ソート：大文字小文字を区別しない
	m_pFileArray = [pFileArray mutableCopy];
	[m_pFileArray sortUsingComparator:^(id str1, id str2) {
		return [((NSString*)str1) compare:((NSString*)str2) options:NSCaseInsensitiveSearch]; }];
	[m_pFileArray retain];
	
	//テスト：ファイルが存在しない場合
	//m_pFileArray = [[NSArray alloc] init];
	
	//ファイルが存在しない場合は選択不可とする
	if ([m_pFileArray count] == 0) {
		m_pTableView.allowsSelection = NO;
	}
	
	[pFileArray release];
	return result;
}

//******************************************************************************
// 通知送信処理
//******************************************************************************
- (int)postNotificationWithName:(NSString*)pName
{
	int result = 0;
	NSNotification* pNotification = nil;
	NSNotificationCenter* pCenter = nil;
	
	//通知オブジェクトを作成
    pNotification = [NSNotification notificationWithName:pName
												  object:self
												userInfo:nil];
	//通知する
	pCenter = [NSNotificationCenter defaultCenter];
	
	//通知に対応する処理を演奏スレッドで処理させる場合
	//[pCenter postNotification:pNotification];
	
	//通知に対応する処理をメインスレッドに処理させる場合
	[pCenter performSelectorOnMainThread:@selector(postNotification:)
							  withObject:pNotification
						   waitUntilDone:NO];
	
	return result;
}

//******************************************************************************
// セクション数
//******************************************************************************
-(NSInteger)numberOfSectionsInTableView:(UITableView*)tableView
{
	//セクション
	// 0. ファイル一覧
    return 1;
} 

//******************************************************************************
// セクションヘッダ
//******************************************************************************
- (NSString*)tableView:(UITableView*)tableView titleForHeaderInSection:(NSInteger)section
{
	NSString* pSectionHeader = nil;
	
	switch(section) {
		case 0:
			//ファイル一覧
			pSectionHeader = @"";
			break;
		default:
			break;
    }
	
    return pSectionHeader;
} 

//******************************************************************************
// セクションごとの項目数
//******************************************************************************
- (NSInteger)tableView:(UITableView *)tableView
 numberOfRowsInSection:(NSInteger)section
{
	NSInteger numOfRows = 0;
	
	//NSLog(@"section %d", section);
	
	switch (section) {
		case 0:
			//ファイル一覧
			numOfRows = [m_pFileArray count];
			
			//ファイルなしの場合はメッセージを表示
			if (numOfRows == 0) {
				numOfRows = 1;
			}
			break;
		default:
			break;
	}
	
	return numOfRows;
}

//******************************************************************************
// 項目表示内容
//******************************************************************************
- (UITableViewCell*)tableView:(UITableView*)tableView
		cellForRowAtIndexPath:(NSIndexPath*)indexPath
{
	UITableViewCell* pCell = nil;
	
	switch (indexPath.section) {
		case 0:
			//ファイル一覧
			pCell = [self makeFileCellForIndexPath:indexPath];
			break;
		default:
			break;
	}
	
    return pCell;
}

//******************************************************************************
// テーブルセル作成：ファイル
//******************************************************************************
- (UITableViewCell*)makeFileCellForIndexPath:(NSIndexPath*)indexPath
{
    static NSString* pCellIdentifier = @"MTFileViewCtrl";
	UITableViewCell* pCell = nil;
	
	//再利用可能セル生成
	pCell = [m_pTableView dequeueReusableCellWithIdentifier:pCellIdentifier];
	if (pCell == nil) {
		pCell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleSubtitle
									   reuseIdentifier:pCellIdentifier];
	}
	
	//ラベル設定
	if ([m_pFileArray count] == 0) {
		//ファイルなしの場合はメッセージを表示
		pCell.textLabel.text = @"File not found.";
		pCell.detailTextLabel.text = @"You can import your MIDI files through iTunes File Sharing.";
		pCell.detailTextLabel.numberOfLines = 0;
		pCell.accessoryType = UITableViewCellAccessoryNone;
		pCell.editingAccessoryType = UITableViewCellAccessoryNone;
	}
	else {
		//ファイル名を表示
		pCell.textLabel.text = [m_pFileArray objectAtIndex:indexPath.row];
		pCell.editingAccessoryType = UITableViewCellAccessoryDisclosureIndicator;
	}
	
	return pCell;
}

//******************************************************************************
// テーブルセル作成：列の高さ
//******************************************************************************
- (CGFloat)tableView:(UITableView*)tableView heightForRowAtIndexPath:(NSIndexPath*)indexPath
{
	CGFloat height = 0.0f;
	
	if ([m_pFileArray count] == 0) {
		//ファイルなしの場合はメッセージ表示のため高さを変更する
		height = 80.0f;
	}
	else {
		//デフォルトの高さを返す
		height = m_pTableView.rowHeight;
	}
	
	return height;
}

//******************************************************************************
// テーブルセル選択イベント
//******************************************************************************
- (void)tableView:(UITableView*)tableView didSelectRowAtIndexPath:(NSIndexPath*)indexPath
{
	//選択状態解除
    [tableView deselectRowAtIndexPath:indexPath animated:YES];
	
	switch (indexPath.section) {
		case 0:
			//ファイル一覧
			[self onSelectFileCellForIndexPath:indexPath];
			break;
		default:
			break;
	}
	
	return;
}

//******************************************************************************
// テーブルセル選択イベント：ファイル一覧
//******************************************************************************
- (void)onSelectFileCellForIndexPath:(NSIndexPath*)indexPath
{
	NSString* pFileName = nil;
	
	//NSLog(@"selected %d", indexPath.row);
	
	//ファイルなしの場合は何もしない
	if ([m_pFileArray count] == 0) goto EXIT;
	
	//選択行番号を保持
	m_SelectedFileIndex = indexPath.row;
	
	//ファイル選択処理
	if (super.editing) {
		//編集モードであればファイル編集ビューを表示
		pFileName = [m_pFileArray objectAtIndex:m_SelectedFileIndex];
		[m_pFileEditViewCtrl setFileName:pFileName];
		[self.navigationController pushViewController:m_pFileEditViewCtrl animated:YES];
	}
	else {
		//編集中でなければファイル選択通知を発行
		[self postNotificationWithName:@"onSelectFile"];
	}
	
EXIT:;
	return;
}

//******************************************************************************
// 選択行ファイルパス取得
//******************************************************************************
- (NSString*)selectedFilePath
{
	NSArray* pPathList = nil;
	NSString* pDocDirPath = nil;
	NSString* pFileName = nil;
	NSString* pFilePath = nil;
	
	//Documentsディレクトリパスを取得
	pPathList = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
	pDocDirPath = [pPathList objectAtIndex:0];
	
	//選択行のファイル名
	pFileName = [m_pFileArray objectAtIndex:m_SelectedFileIndex];
	
	//ファイルパス
	pFilePath = [NSString stringWithFormat:@"%@/%@", pDocDirPath, pFileName];
	
	return pFilePath;
}

//******************************************************************************
// 編集開始終了通知
//******************************************************************************
- (void)setEditing:(BOOL)editing animated:(BOOL)animated
{
	//編集開始終了処理
	[super setEditing:editing animated:animated];
	[m_pTableView setEditing:editing animated:animated];
	
	return;
}

//******************************************************************************
// 行単位削除可否判定
//******************************************************************************
- (BOOL)tableView:(UITableView *)tableView canEditRowAtIndexPath:(NSIndexPath *)indexPath
{
	BOOL isDeletable = YES;
	
	if ([m_pFileArray count] == 0) {
		isDeletable = NO;
	}
	
	return isDeletable;
}

//******************************************************************************
// 行削除イベント
//******************************************************************************
- (void)tableView:(UITableView *)tableView commitEditingStyle:(UITableViewCellEditingStyle)editingStyle forRowAtIndexPath:(NSIndexPath *)indexPath
{
	int result = 0;
	NSString* pFilePath = nil;
	
	//ファイルなしの場合は何もしない
	if ([m_pFileArray count] == 0) goto EXIT;
	
	//選択行番号を保持
	m_SelectedFileIndex = indexPath.row;
	
	//選択行ファイルパス取得
	pFilePath = [self selectedFilePath];
	
	//ファイル削除
	result = [self removeFileWithPath:pFilePath];
	if (result != 0) goto EXIT;
	
	//ファイル一覧の該当データを削除
	[m_pFileArray removeObjectAtIndex:indexPath.row];
	
	//テーブル再読み込み
	[m_pTableView reloadData];
	
EXIT:;
	//if (result != 0) YN_SHOW_ERR();
	return;
}

//******************************************************************************
// ファイル削除
//******************************************************************************
- (int)removeFileWithPath:(NSString*)pPath
{
	int result = 0;
	NSError* pError = nil;
	NSURL* pPathURL = nil;
	BOOL isDeleted = NO;
	UIAlertController* pAlertController = nil;
	UIAlertAction* pAlertAction = nil;
	
	pPathURL = [NSURL fileURLWithPath:pPath];
	isDeleted = [[NSFileManager defaultManager] removeItemAtURL:pPathURL error:&pError];
	if (!isDeleted) {
		//ファイル削除失敗
		result = YN_SET_ERR(@"File deletion failed.", pError.code, 0);
		
		//アラート表示
		pAlertController = [UIAlertController alertControllerWithTitle:@"File deletion failed."
															   message:[pError localizedDescription]
														preferredStyle:UIAlertControllerStyleAlert];
		pAlertAction = [UIAlertAction actionWithTitle:@"OK"
												style:UIAlertActionStyleDefault
											  handler:nil];
		[pAlertController addAction:pAlertAction];
		[self presentViewController:pAlertController animated:YES completion:nil];
		
		goto EXIT;
	}
	
EXIT:;
	return result;
}

//******************************************************************************
// ファイル名更新イベント
//******************************************************************************
- (void)onUpdateFileName:(NSNotification*)pNotification
{
	int result = 0;
	NSString* pNewFileName = nil;
	NSString* pSrcPath = nil;
	NSString* pDestPath = nil;
	
	//変更前ファイルパス
	pSrcPath = [self selectedFilePath];
	
	//変更後ファイルパス
	pNewFileName = [m_pFileEditViewCtrl getNewFileName];
	pDestPath = [[pSrcPath stringByDeletingLastPathComponent] stringByAppendingPathComponent:pNewFileName];
	
	//NSLog(@"SrcFilePath: %@", pSrcPath);
	//NSLog(@"DestFilePath: %@", pDestPath);
	
	//ファイルリネーム
	result = [self renameFileWithPath:pSrcPath toPath:pDestPath];
	if (result != 0) goto EXIT;
	
	//ファイル一覧の該当データを更新
	[m_pFileArray replaceObjectAtIndex:m_SelectedFileIndex withObject:pNewFileName];
	
	//ファイル一覧ソート：大文字小文字を区別しない
	[m_pFileArray sortUsingComparator:^(id str1, id str2) {
		return [((NSString*)str1) compare:((NSString*)str2) options:NSCaseInsensitiveSearch]; }];
	
	//テーブル再読み込み
	[m_pTableView reloadData];
	
EXIT:;
	//if (result != 0) YN_SHOW_ERR();
	return;
}

//******************************************************************************
// ファイルリネーム
//******************************************************************************
- (int)renameFileWithPath:(NSString*)pSrcPath toPath:(NSString*)pDestPath
{
	int result = 0;
	NSError* pError = nil;
	BOOL isMoved = NO;
	UIAlertController* pAlertController = nil;
	UIAlertAction* pAlertAction = nil;
	
	//リネーム実施
	isMoved = [[NSFileManager defaultManager] moveItemAtPath:pSrcPath toPath:pDestPath error:&pError];
	if (!isMoved) {
		//リネーム失敗
		result = YN_SET_ERR(@"File rename failed.", pError.code, 0);
		
		//アラート表示
		pAlertController = [UIAlertController alertControllerWithTitle:@"File rename failed."
															   message:[pError localizedDescription]
														preferredStyle:UIAlertControllerStyleAlert];
		pAlertAction = [UIAlertAction actionWithTitle:@"OK"
												style:UIAlertActionStyleDefault
											  handler:nil];
		[pAlertController addAction:pAlertAction];
		[self presentViewController:pAlertController animated:YES completion:nil];
		goto EXIT;
	}
	
EXIT:;
	return result;
}


@end

