//******************************************************************************
//
// MIDITrail / MTFileSearchResultsViewCtrl
//
// ファイルビュー制御クラス
//
// Copyright (C) 2019-2020 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "YNBaseLib.h"
#import "MTFileSearchResultsViewCtrl.h"
#import "MTNavCtrl.h"
#import "MTParam.h"


@implementation MTFileSearchResultsViewCtrl

//******************************************************************************
// ビュー登録完了
//******************************************************************************
- (void)viewDidLoad
{
    [super viewDidLoad];
	
	//検索結果配列初期化
	m_pFilteredFileArray = [[NSMutableArray alloc] init];
	
	return;
}

//******************************************************************************
// ビュー表示
//******************************************************************************
- (void)viewWillAppear:(BOOL)animated
{
	[super viewWillAppear:animated];
	
	if (@available(iOS 13.0, *)) {
	}
	else {
		//検索結果を選択してメインビューを表示した後にファイル一覧に戻ってくると
		//テーブルの先頭がナビゲーションバーと検索バーの下に潜り込んでしまうためコンテンツの位置をずらす
		//iOS12で問題が発生しているがiOS13では発生しない
		self.tableView.contentInset = UIEdgeInsetsMake(44,0,0,0);
	}
	
	return;
}

//******************************************************************************
// インターフェース自動回転確認（iOS6以降）
//******************************************************************************
//- (BOOL)shouldAutorotate
//{
//	//回転を許可しない
//	return NO;
//}

//******************************************************************************
// インターフェース自動回転確認：回転方向（iOS6以降）
//******************************************************************************
//- (NSUInteger)supportedInterfaceOrientations
//{
//	//全方向に対応する
//	return UIInterfaceOrientationMaskAll;
//}

//******************************************************************************
// セクション数
//******************************************************************************
-(NSInteger)numberOfSectionsInTableView:(UITableView*)tableView
{
	//セクション数はファイル一覧のみ
    return 1;
}

//******************************************************************************
// セクションごとの項目数
//******************************************************************************
- (NSInteger)tableView:(UITableView *)tableView
 numberOfRowsInSection:(NSInteger)section
{
	NSInteger numOfRows = 0;
	
	switch (section) {
		case 0:
			numOfRows = [m_pFilteredFileArray count];
			break;
		default:
			break;
	}
	
	return numOfRows;
}

//******************************************************************************
// 項目表示内容
//******************************************************************************
- (UITableViewCell*)tableView:(UITableView*)tableView
		cellForRowAtIndexPath:(NSIndexPath*)indexPath
{
	UITableViewCell* pCell = nil;
	
	switch (indexPath.section) {
		case 0:
			//ファイル一覧
			pCell = [self makeFileCellForIndexPath:indexPath];
			break;
		default:
			break;
	}
	
    return pCell;
}

//******************************************************************************
// テーブルセル作成：ファイル
//******************************************************************************
- (UITableViewCell*)makeFileCellForIndexPath:(NSIndexPath*)indexPath
{
    static NSString* pCellIdentifier = @"MTFileSearchResultsViewCtrl";
	UITableViewCell* pCell = nil;
	NSString* pFile = nil;
	
	//再利用可能セル生成
	pCell = [self.tableView dequeueReusableCellWithIdentifier:pCellIdentifier];
	if (pCell == nil) {
		pCell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleSubtitle
									   reuseIdentifier:pCellIdentifier];
	}
	
	//ファイル名を表示
	pFile = [m_pFilteredFileArray objectAtIndex:indexPath.row];
	if ([[m_pFileAttributesDictionary objectForKey:pFile] intValue] == MT_FILE_ATTRIBUTE_DIRECTORY) {
		//ディレクトリの場合
		pCell.textLabel.text = pFile;
		pCell.detailTextLabel.text = @"";
		pCell.accessoryType = UITableViewCellAccessoryDisclosureIndicator; //階層マーク
		pCell.editingAccessoryType = UITableViewCellAccessoryDisclosureIndicator;
	}
	else {
		//標準ファイルの場合
		pCell.textLabel.text = pFile;
		pCell.detailTextLabel.text = @"";
		pCell.accessoryType = UITableViewCellAccessoryNone;
		pCell.editingAccessoryType = UITableViewCellAccessoryDisclosureIndicator;
	}
	
	return pCell;
}

//******************************************************************************
// 検索結果登録
//******************************************************************************
- (int)setResults:(NSMutableArray*)pResultsArray fileAttributesDictionary:(NSMutableDictionary*)pFileAttributesDictionary
{
	int result = 0;
	
	//検索結果を保持
	[m_pFilteredFileArray release];
	m_pFilteredFileArray = [pResultsArray mutableCopy];
	[m_pFilteredFileArray retain];
	
	//ファイル属性辞書を保持
	[m_pFileAttributesDictionary release];
	m_pFileAttributesDictionary = [pFileAttributesDictionary mutableCopy];
	[m_pFileAttributesDictionary retain];
	
	//テーブル再読み込み
	[self.tableView reloadData];
	
	return result;
}

//******************************************************************************
// テーブルセル選択イベント
//******************************************************************************
- (void)tableView:(UITableView*)tableView didSelectRowAtIndexPath:(NSIndexPath*)indexPath
{
	//選択状態解除
    [tableView deselectRowAtIndexPath:indexPath animated:YES];
	
	switch (indexPath.section) {
		case 0:
			//ファイル一覧
			[self onSelectFileCellForIndexPath:indexPath];
			break;
		default:
			break;
	}
	
	return;
}

//******************************************************************************
// テーブルセル選択イベント：ファイル一覧
//******************************************************************************
- (void)onSelectFileCellForIndexPath:(NSIndexPath*)indexPath
{
	//ファイルなしの場合は何もしない
	if ([m_pFilteredFileArray count] == 0) goto EXIT;
	
	//選択行番号を保持
	m_SelectedFileIndex = indexPath.row;
	
	//ファイル選択通知
	[self postNotificationWithName:m_pNotificationNameForSelectFile];
	
EXIT:;
	return;
}

//******************************************************************************
// 選択ファイル名取得
//******************************************************************************
- (NSString*)selectedFileName
{
	NSString* pFileName = nil;
	
	pFileName = [m_pFilteredFileArray objectAtIndex:m_SelectedFileIndex];
	
	return pFileName;
}

//******************************************************************************
// 通知送信処理
//******************************************************************************
- (int)postNotificationWithName:(NSString*)pName
{
	int result = 0;
	NSNotification* pNotification = nil;
	NSNotificationCenter* pCenter = nil;
	
	//通知オブジェクトを作成
    pNotification = [NSNotification notificationWithName:pName
												  object:self
												userInfo:nil];
	//通知する
	pCenter = [NSNotificationCenter defaultCenter];
	
	//通知に対応する処理を演奏スレッドで処理させる場合
	//[pCenter postNotification:pNotification];
	
	//通知に対応する処理をメインスレッドに処理させる場合
	[pCenter performSelectorOnMainThread:@selector(postNotification:)
							  withObject:pNotification
						   waitUntilDone:NO];
	
	return result;
}

//******************************************************************************
// 破棄
//******************************************************************************
- (void)dealloc
{
	[m_pFilteredFileArray release];
	[m_pFileAttributesDictionary release];
	[m_pNotificationNameForSelectFile release];
	
	m_pFilteredFileArray = nil;
	m_pFileAttributesDictionary = nil;
	m_pNotificationNameForSelectFile = nil;
	
	[super dealloc];
}

//******************************************************************************
// ファイル選択通知名称設定
//******************************************************************************
- (void)setNotificationNameForSelectFile:(NSString*)pNotificationName
{
	[m_pNotificationNameForSelectFile release];
	m_pNotificationNameForSelectFile = [pNotificationName retain];
}

@end

