//******************************************************************************
//
// MIDITrail / MTSettingMIDIOutDevViewCtrl
//
// MIDI出力デバイス選択ビュークラス
//
// Copyright (C) 2017-2019 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "MTParam.h"
#import "MTSettingMIDIOutDevViewCtrl.h"


//******************************************************************************
// プライベートメソッド定義
//******************************************************************************
@interface MTSettingMIDIOutDevViewCtrl ()

//テーブルセル作成：デバイス
- (UITableViewCell*)makeDevCellForIndexPath:(NSIndexPath*)indexPath;

//テーブルセル選択イベント：デバイス一覧
- (void)onSelectDevCellForIndexPath:(NSIndexPath*)indexPath tableView:(UITableView*)tableView;

@end


@implementation MTSettingMIDIOutDevViewCtrl

//******************************************************************************
// 対象ポート設定
//******************************************************************************
- (void)setPortKey:(NSString*)pPortKey portLabel:(NSString*)pPortLabel
{
	[pPortKey retain];
	[m_pPortKey release];
	m_pPortKey = pPortKey;
    
    [pPortLabel retain];
    [m_pPortLabel release];
    m_pPortLabel = pPortLabel;
	
	//デバイス表示名称用のキー文字列：ex."PrtA_DevDisplayName"
	[m_PortDevDisplayNameKey release];
	m_PortDevDisplayNameKey = [NSString stringWithFormat:@"%@_DevDisplayName", m_pPortKey];
	[m_PortDevDisplayNameKey retain];
}

//******************************************************************************
// 初期化
//******************************************************************************
- (id)initWithNibName:(NSString *)nibNameOrNil bundle:(NSBundle *)nibBundleOrNil
{
	int result = 0;
	
	self = [super initWithNibName:nibNameOrNil bundle:nibBundleOrNil];
	if (self) {
		//ビュー設定
		self.title = @"";
		
		//ユーザ設定初期化
		m_pUserConf = [[YNUserConf alloc] init];
		if (m_pUserConf == nil) {
			result = YN_SET_ERR(@"Program error.", 0, 0);
			goto EXIT;
		}
	}
	
EXIT:;
	if (result != 0) YN_SHOW_ERR();
	return self;
}

//******************************************************************************
// ビュー登録完了
//******************************************************************************
- (void)viewDidLoad
{
    [super viewDidLoad];
	// Do any additional setup after loading the view.
	
	return;
}

//******************************************************************************
// インターフェース自動回転確認
//******************************************************************************
- (BOOL)shouldAutorotateToInterfaceOrientation:(UIInterfaceOrientation)interfaceOrientation
{
    return (interfaceOrientation == UIInterfaceOrientationPortrait);
}

//******************************************************************************
// インターフェース自動回転確認（iOS6以降）
//******************************************************************************
- (BOOL)shouldAutorotate
{
	//回転を許可する
	return YES;
}

//******************************************************************************
// インターフェース自動回転確認：回転方向（iOS6以降）
//******************************************************************************
- (NSUInteger)supportedInterfaceOrientations
{
	//全方向に対応する
	return UIInterfaceOrientationMaskAll;
}

//******************************************************************************
// ビュー表示
//******************************************************************************
- (void)viewWillAppear:(BOOL)animated
{
	int result = 0;
	
	//ビュー設定
	self.title = m_pPortLabel;
	
	//MIDI出力デバイス制御初期化
	result = m_OutDevCtrl.Initialize();
	if (result != 0) goto EXIT;	
	
	//テーブルビュー再表示
	[m_pTableView reloadData];
	
EXIT:;
	if (result != 0) YN_SHOW_ERR();
	return;
}

//******************************************************************************
// ビュー非表示
//******************************************************************************
- (void)viewWillDisappear:(BOOL)animated
{
	return;
}

//******************************************************************************
// セクション数
//******************************************************************************
-(NSInteger)numberOfSectionsInTableView:(UITableView*)tableView
{
    return 1;
} 

//******************************************************************************
// セクションヘッダ
//******************************************************************************
- (NSString*)tableView:(UITableView*)tableView titleForHeaderInSection:(NSInteger)section
{
    return nil;
} 

//******************************************************************************
// セクションごとの項目数
//******************************************************************************
- (NSInteger)tableView:(UITableView *)tableView
 numberOfRowsInSection:(NSInteger)section
{
	NSInteger numOfRows = 0;
	unsigned int devNum = 0;
	
	switch (section) {
		case 0:
			//デバイス数
			devNum = m_OutDevCtrl.GetDevNum();
            //noneを項目に追加
            numOfRows = devNum + 1;
			break;
		default:
			break;
	}
	
	return numOfRows;
}

//******************************************************************************
// 項目表示内容
//******************************************************************************
- (UITableViewCell*)tableView:(UITableView*)tableView
		cellForRowAtIndexPath:(NSIndexPath*)indexPath
{
	UITableViewCell* pCell = nil;
	
	switch (indexPath.section) {
		case 0:
			//デバイス一覧
			pCell = [self makeDevCellForIndexPath:indexPath];
			break;
		default:
			break;
	}
	
	return pCell;
}

//******************************************************************************
// テーブルセル作成：デバイス
//******************************************************************************
- (UITableViewCell*)makeDevCellForIndexPath:(NSIndexPath*)indexPath
{
	static NSString* pCellIdentifier = @"MTSettingMIDIOutDevViewCtrl";
	UITableViewCell* pCell = nil;
	NSString* pDevIdName = nil;
	NSString* pDevDisplayName = nil;
	NSString* pSelectedDevIdName = nil;
	NSString* pSelectedDevDisplayName = nil;
    unsigned int index = 0;
	
	//再利用可能セル生成
	pCell = [m_pTableView dequeueReusableCellWithIdentifier:pCellIdentifier];
	if (pCell == nil) {
		pCell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleSubtitle
									   reuseIdentifier:pCellIdentifier];
	}
	
	//ラベル設定
	if (indexPath.row == 0) {
		pCell.textLabel.text = @"(none)";
		pDevIdName = @"";
	}
	else {
        index = (unsigned int)indexPath.row - 1;
		pDevDisplayName = m_OutDevCtrl.GetDevDisplayName(index);
		pCell.textLabel.text = pDevDisplayName;
		pDevIdName = m_OutDevCtrl.GetDevIdName(index);
		pCell.detailTextLabel.text = m_OutDevCtrl.GetManufacturerName(index);
	}
	
	//MIDI OUTの設定を取得
	[m_pUserConf setCategory:MT_CONF_CATEGORY_MIDI];
	[m_pUserConf setSection:MT_CONF_SECTION_MIDIOUT];
	pSelectedDevIdName = [m_pUserConf strValueForKey:m_pPortKey defaultValue:@""];
	pSelectedDevDisplayName = [m_pUserConf strValueForKey:m_PortDevDisplayNameKey defaultValue:@""];
	
	//チェックマーク設定
	pCell.accessoryType = UITableViewCellAccessoryNone;
    if ([pSelectedDevIdName length] == 0) {
		//未選択の場合は先頭行の(none)にチェックマークを付ける
        if (indexPath.row == 0) {
            pCell.accessoryType = UITableViewCellAccessoryCheckmark;
        }
    }
    else {
		//選択デバイスのデバイス識別名称と一致する場合
		if ([pSelectedDevIdName isEqualToString:pDevIdName]) {
			//デバイス表示名称が保存されていない場合はデバイス識別名のみで判定
			if ([pSelectedDevDisplayName length] == 0) {
				pCell.accessoryType = UITableViewCellAccessoryCheckmark;
			}
			//デバイス表示名称が保存されている場合はデバイス表示名称の一致も判定
			else if ([pSelectedDevDisplayName isEqualToString:pDevDisplayName]) {
				pCell.accessoryType = UITableViewCellAccessoryCheckmark;
			}
		}
	}
	
	return pCell;
}

//******************************************************************************
// テーブルセル選択イベント
//******************************************************************************
- (void)tableView:(UITableView*)tableView didSelectRowAtIndexPath:(NSIndexPath*)indexPath
{
	//選択状態解除
	[tableView deselectRowAtIndexPath:indexPath animated:YES];
	
	switch (indexPath.section) {
		case 0:
			//ファイル一覧
			[self onSelectDevCellForIndexPath:indexPath tableView:tableView];
			break;
		default:
			break;
	}
	
	return;
}

//******************************************************************************
// テーブルセル選択イベント：デバイス一覧
//******************************************************************************
- (void)onSelectDevCellForIndexPath:(NSIndexPath*)indexPath tableView:(UITableView *)tableView
{
	NSString* pSelectedDevIdName = nil;
	NSString* pSelectedDevDisplayName = nil;
	unsigned int index = 0;
	
	if (indexPath.row == 0) {
		pSelectedDevIdName = @"";
		pSelectedDevDisplayName = @"";
	}
	else {
		index = (unsigned int)indexPath.row - 1;
		pSelectedDevIdName = m_OutDevCtrl.GetDevIdName(index);
		pSelectedDevDisplayName = m_OutDevCtrl.GetDevDisplayName(index);
	}
	
    //MIDI OUT設定を保存
    [m_pUserConf setCategory:MT_CONF_CATEGORY_MIDI];
    [m_pUserConf setSection:MT_CONF_SECTION_MIDIOUT];
    [m_pUserConf setStr:pSelectedDevIdName forKey:m_pPortKey];
	[m_pUserConf setStr:pSelectedDevDisplayName forKey:m_PortDevDisplayNameKey];
	
	//再表示
	[tableView performSelector:@selector(reloadData) withObject:nil afterDelay:0.1f];
	
	//1階層戻る
	//[self.navigationController popViewControllerAnimated:YES];
	
EXIT:;
	return;
}


@end

