//******************************************************************************
//
// OpenGL Utility / OGLH
//
// ヘルパ関数クラス
//
// Copyright (C) 2010-2019 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "OGLH.h"
#import <math.h>


//******************************************************************************
// ベクトル正規化
//******************************************************************************
OGLVECTOR3 OGLH::Vec3Normalize(
		const OGLVECTOR3 vector
	)
{
	double x, y, z, r, ri;
	OGLVECTOR3 normalizedVector;
	
	x = (double)(vector.x);
	y = (double)(vector.y);
	z = (double)(vector.z);
	r = sqrt((x*x) + (y*y) + (z*z));
	
	if (r < (1e-6)) {
		//正規化できないので同じベクトルを返す
		ri = 1.0;
	}
	else {
		ri = 1.0 / r;
	}
	
	normalizedVector.x = (float)(x * ri);
	normalizedVector.y = (float)(y * ri);
	normalizedVector.z = (float)(z * ri);
	
	return normalizedVector;
}

//******************************************************************************
// ベクトル内積
//******************************************************************************
GLfloat OGLH::Vec3Dot(
		const OGLVECTOR3 vector1,
		const OGLVECTOR3 vector2
	)
{
	GLfloat dot;
	
	dot = (vector1.x * vector2.x) + (vector1.y * vector2.y) + (vector1.z * vector2.z);
	
	return dot;
}

//******************************************************************************
// ベクトル外積
//******************************************************************************
OGLVECTOR3 OGLH::Vec3Cross(
		const OGLVECTOR3 vector1,
		const OGLVECTOR3 vector2
	)
{
	OGLVECTOR3 crossVector;
	
	crossVector.x = (vector1.y * vector2.z) - (vector1.z * vector2.y);
	crossVector.y = (vector1.z * vector2.x) - (vector1.x * vector2.z);
	crossVector.z = (vector1.x * vector2.y) - (vector1.y * vector2.x);
	
	return crossVector;
}

//******************************************************************************
// ラジアン算出
//******************************************************************************
float OGLH::ToRadian(float degree)
{
	return (degree * OGL_MATH_PI / 180.0);
}

//******************************************************************************
// 角度算出
//******************************************************************************
float OGLH::ToDegree(float radian)
{
	return (radian * 180.0 / OGL_MATH_PI);
}

//******************************************************************************
// 符号確認
//******************************************************************************
float OGLH::Sign(float value)
{
	float sign = 1.0f;
	
	if (value < 0) {
		sign = -1.0f;
	}
	
	return sign;
}

//******************************************************************************
// 座標回転：YZ平面
//******************************************************************************
OGLVECTOR3 OGLH::RotateYZ(
		float centerY,
		float centerZ,
		OGLVECTOR3 p1,
		float angle
	)
{
	OGLVECTOR3 p2;
	float rad = 0.0f;
	
	rad = OGLH::ToRadian(angle);
	p2.x = p1.x;
	p2.y = (float)(centerY + (sin(rad) * (p1.z - centerZ)) + (cos(rad) * (p1.y - centerY)));
	p2.z = (float)(centerZ + (cos(rad) * (p1.z - centerZ)) - (sin(rad) * (p1.y - centerY)));
	
	return p2;
}

