//******************************************************************************
//
// MIDITrail / MTMainViewCtrl
//
// メインビュー制御クラス
//
// Copyright (C) 2012-2021 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import <UIKit/UIKit.h>
#import <QuartzCore/QuartzCore.h>
#import <OpenGLES/ES1/gl.h>
#import <OpenGLES/ES1/glext.h>
#import <CoreMotion/CoreMotion.h>
#import <GameController/GameController.h>
#import <MediaPlayer/MediaPlayer.h>
#import "SMIDILib.h"
#import "OGLUtil.h"
#import "MTParam.h"
#import "MTMachTime.h"
#import "MTScene.h"
#import "MTSceneMsgQueue.h"
#import "MTHowToViewCtrl.h"
#import "MTGamePadCtrl.h"


//******************************************************************************
// メインビュー制御クラス
//******************************************************************************
@interface MTMainViewCtrl : UIViewController <UIActionSheetDelegate> {
	
	//ベースビュー
	IBOutlet UIView* m_pBaseView;
	
	//ボタン
	IBOutlet UIButton* m_pCloseButton;
	IBOutlet UIButton* m_pPlayButton;
	IBOutlet UIButton* m_pStopButton;
	IBOutlet UIButton* m_pSkipBackwardButton;
	IBOutlet UIButton* m_pSkipForwardButton;
	IBOutlet UIButton* m_pPlaySpeedDownButton;
	IBOutlet UIButton* m_pPlaySpeedUpButton;
	IBOutlet UIButton* m_pFolderPlaybackButton;
	IBOutlet UIButton* m_pRepeatButton;
	IBOutlet UIButton* m_pViewButton;
	IBOutlet UIButton* m_pHelpButton;
	IBOutlet UIActivityIndicatorView* m_pActivityIndicator;
	IBOutlet UILabel* m_pLabel;
	
	//ガイドイメージ
	IBOutlet UIImageView* m_pGuideImageUD;
	IBOutlet UIImageView* m_pGuideImageFB;
	IBOutlet UIImageView* m_pGuideImageLR;
	IBOutlet UIImageView* m_pGuideImageRT;
	
	//スクリーンサイズ
	GLint m_ScreenWidth;
	GLint m_ScreenHeight;
	CGRect m_ViewBounds;
	
	//OpenGL制御系
	EAGLContext* m_pEAGLContext;
	GLuint m_DefaultFrameBuffer;
	GLuint m_DefaultColorRenderBuffer;
	GLuint m_DefaultDepthRenderBuffer;
	GLuint m_MSAAFrameBuffer;
	GLuint m_MSAAColorRenderBuffer;
	GLuint m_MSAADepthRenderBuffer;
	id m_Timer;
	
	//描画制御系
	OGLRenderer m_Renderer;
	OGLRedererParam m_RendererParam;
	MTScene* m_pScene;
	MTMachTime m_MachTime;
	bool m_isBackground;
	
	//メッセージ制御系
	SMMsgQueue* m_pMsgQueue;
	MTSceneMsgQueue m_SceneMsgQueue;
	BOOL m_isStopScene;
	
	//画面制御系
	uint64_t m_LastTouchTime;
	uint64_t m_PrevDrawTime;
	uint64_t m_PrevFPSDrawTime;
	float m_FlameCount;
	
	//タッチ制御
	CGPoint m_PrevTouchPoint;
	float m_TounchesTotalDistance;
	
	//ボタン表示状態
	BOOL m_isDisplayButtons;
	
	//モニタフラグ
	BOOL m_isMonitor;
	
	//スクリーン種別
	MTScene::ScreenType m_ScreenType;
	
	//ヘルプビュー
	MTHowToViewCtrl* m_pHowToViewCtrl;
	
	//ボタン画像
	UIImage* m_pImgButtonPlayN;
	UIImage* m_pImgButtonPlayH;
	UIImage* m_pImgButtonPauseN;
	UIImage* m_pImgButtonPauseH;
	
    //スクリーンロックフラグ
    bool m_isScreenLocked;
	
	//デバイスモーション管理
	CMMotionManager* m_pMotionManager;
	
	//ゲームコントローラー制御
	MTGamePadCtrl m_GamePadCtrl;
	
	//ゲームパッド用視点番号
	int m_GamePadViewPointNo;
	
	//瞳孔間距離(Interpupillary distance)ポイント数
	int m_IPDPoints;
	
	//リワインド／スキップ制御
	int m_SkipBackTimeSpanInMsec;
	int m_SkipForwardTimeSpanInMsec;
	
	//前回演奏経過時間（秒）
	int m_PrevPlayTimeSec;
}

//--------------------------------------
// 生成と破棄
//--------------------------------------
//生成
- (id)initWithNibName:(NSString *)nibNameOrNil
			   bundle:(NSBundle *)nibBundleOrNil
		rendererParam:(OGLRedererParam)rendererParam;

//破棄
- (void)dealloc;

//初期化処理
- (int)initialize:(SMMsgQueue*)pMsgQueue;

//終了処理
- (void)terminate;

//ビュー境界取得
- (CGRect)getViewBounds;

//描画デバイス取得
- (OGLDevice*)getDevice;

//リワインド／スキップ設定
- (void)setSkipTimeSpanWithBack:(int)backTimeSpanInMsec
						 foward:(int)forwardTimeSpanInMSec;

//--------------------------------------
// ビューイベント
//--------------------------------------
//ビュー取得
- (UIView*)getView;

//ビュー登録完了
- (void)viewDidLoad;

//インターフェース自動回転確認
- (BOOL)shouldAutorotateToInterfaceOrientation:(UIInterfaceOrientation)interfaceOrientation;

//インターフェース自動回転確認（iOS6以降）
- (BOOL)shouldAutorotate;
- (NSUInteger)supportedInterfaceOrientations;

// ビュー表示開始
- (void)viewWillAppear:(BOOL)animated;

// ビュー表示終了
- (void)viewDidDisappear:(BOOL)animated;

//--------------------------------------
// シーン操作
//--------------------------------------
//シーン開始
- (int)startScene:(MTScene*)pScene isMonitor:(BOOL)isMonitor;

//シーン停止
- (int)stopScene;

//シーン操作：演奏開始
- (int)scene_PlayStart;

//シーン操作：演奏終了
- (int)scene_PlayEnd;

//シーン操作：巻き戻し
- (int)scene_Rewind;

//シーン操作：視点リセット
- (int)scene_ResetViewpoint;

//シーン操作：静的視点移動
- (int)scene_MoveToStaticViewpoint:(unsigned int)viewpointNo;

//シーン操作：視点登録
- (int)scene_SetViewpoint:(MTScene::MTViewParamMap*)pParamMap;

//シーン操作：視点取得
- (int)scene_GetViewpoint:(MTScene::MTViewParamMap*)pParamMap;

//シーン操作：エフェクト設定
- (int)scene_SetEffect:(MTScene::EffectType)type isEnable:(bool)isEnable;

//シーン操作：スクリーン種別設定
- (int)scene_SetScreenType:(MTScene::ScreenType)type;

//シーン操作：スクリーンロック
- (int)scene_LockScreen:(bool)isLocked;

//--------------------------------------
// ボタンイベント
//--------------------------------------
// クローズボタン押下
- (IBAction)onCloseButton;
// 再生ボタン押下
- (IBAction)onPlayButton;
// 停止ボタン押下
- (IBAction)onStopButton;
// 後方スキップボタン押下
- (IBAction)onSkipBackwardButton;
// 前方スキップボタン押下
- (IBAction)onSkipForwardButton;
// 再生スピードダウンボタン押下
- (IBAction)onPlaySpeedDownButton;
// 再生スピードアップボタン押下
- (IBAction)onPlaySpeedUpButton;
// フォルダ演奏ボタン押下
- (IBAction)onFolderPlaybackButton;
// リピートボタン押下
- (IBAction)onRepeatButton;
// ビューボタン押下
- (IBAction)onViewButton;
// ヘルプボタン押下
- (IBAction)onHelpButton;

//アクションシートボタン選択イベント
- (void)actionSheet:(UIActionSheet*)actionSheet clickedButtonAtIndex:(NSInteger)buttonIndex;

//--------------------------------------
// タッチイベント
//--------------------------------------
//タッチイベント：開始
- (void)touchesBegan:(NSSet*)touches withEvent:(UIEvent *)event;

//タッチイベント：移動
- (void)touchesMoved:(NSSet*)touches withEvent:(UIEvent *)event;

//タッチイベント：終了
- (void)touchesEnded:(NSSet*)touches withEvent:(UIEvent *)event;

//タッチイベント：キャンセル
- (void)touchesCancelled:(NSSet*)touches withEvent:(UIEvent *)event;

//--------------------------------------
// 状態設定
//--------------------------------------
//フォルダ演奏状態設定
- (void)setFolderPlaybackStatus:(BOOL)isFolderPlayback;

//リピート状態設定
- (void)setRepeatStatus:(BOOL)isRepeat;

//インジケータアニメーション開始
- (void)startActivityIndicator;

//インジケータアニメーション停止
- (void)stopActivityIndicator;

//カレントコンテキスト設定
- (void)setCurrentContext;

//演奏状態設定
- (void)setPlayStatus:(PlayStatus)status;

//ステータスバー非表示
- (BOOL)prefersStatusBarHidden;

//バックグラウンド状態設定
- (void)setBackground:(bool)isBackground;

//--------------------------------------
// ゲームコントローラー対応
//--------------------------------------
- (void)onGameControllerDidConnect;
- (void)onGameControllerDidDisconnect;
- (int)initializeGameController;
- (void)changeViewPoint:(int)step;

//--------------------------------------
// リモートコマンドハンドラ
//--------------------------------------
// リモートコマンドハンドラ：再生
- (MPRemoteCommandHandlerStatus)onPlayCommand:(MPRemoteCommandEvent*)pCommandEvent;

// リモートコマンドハンドラ：一時停止
- (MPRemoteCommandHandlerStatus)onPauseCommand:(MPRemoteCommandEvent*)pCommandEvent;

// リモートコマンドハンドラ：再生/一時停止トグル
- (MPRemoteCommandHandlerStatus)onTogglePlayPauseCommand:(MPRemoteCommandEvent*)pCommandEvent;

// リモートコマンドハンドラ：停止
- (MPRemoteCommandHandlerStatus)onStopCommand:(MPRemoteCommandEvent*)pCommandEvent;

// リモートコマンドハンドラ：後方スキップ
- (MPRemoteCommandHandlerStatus)onSkipBackwardCommand:(MPRemoteCommandEvent*)pCommandEvent;

// リモートコマンドハンドラ：前方スキップ
- (MPRemoteCommandHandlerStatus)onSkipForwardCommand:(MPRemoteCommandEvent*)pCommandEvent;

@end

