<?php

require_once(dirname(__FILE__).'/../config.php');

require_once(dirname(__FILE__).'/MilResult.php');
require_once(dirname(__FILE__).'/MilUsageCommand.php');
require_once(dirname(__FILE__).'/MilUsageEnvironment.php');

require_once(EXCEPTION_DIR.'/NoSuchPackageException.php');
require_once(EXCEPTION_DIR.'/NoSuchPackageInitializerException.php');
require_once(EXCEPTION_DIR.'/InvalidArgumentCountException.php');

class MilParser{
	
	/** ホワイトスペースとして認識する文字 */
	private static $white = array(" ", "\t", "\n", "\r", "\0", "\x0B");
	
	/** 変数 */
	private $vars = array();
	
	/** ページのタイトル */
	private $title = 'untitled';
	
	/** 元のソースコード */
	private $source;
	/** コンパイル後の HTML */
	private $result;
	
	/** 何回コマンドが実行されたか */
	private $commandCalled;
	/** 何回環境が実行されたか */
	private $environmentCalled;
	
	/** 現在遅延型コマンド or 環境をパースしているかどうか */
	private $isDelay;
	
	public function __construct($source){
		$this->source = $source;
		$this->result = $source;
		$this->initPackages();
		$this->prepare();
		$this->commandCalled = 0;
		$this->environmentCalled = 0;
		for(;;){
			for(;;){
				$num_replace = $this->parse(false);
				if($this->commandCalled + $this->environmentCalled > MAX_CALL){
					$this->result .= $this->callCommand('error', array('コマンド、または環境の実行回数制限を越えました'));
					break 2;
				}
				if($num_replace == 0)
					break;
			}
			$num_replace = $this->parse(true);
			if($num_replace == 0){
				break;
			}
			if($this->commandCalled + $this->environmentCalled > MAX_CALL){
				$this->result .= $this->callCommand('error', array('コマンド、または環境の実行回数制限を越えました'));
				break;
			}
		}
		$this->format();
	}
	
	/**
	 * パッケージをロードする
	 */
	private function initPackages(){
		global $packages;
		foreach($packages as $package_name){
			$this->initPackage($package_name);
		}
	}
	
	/**
	 * パッケージを一つだけロード
	 */
	private function initPackage($package_name){
		/*
		$load_path = PACKAGE_DIR.'/'.$package_name.'/load.php';
		if(!file_exists($load_path))
			throw new NoSuchPackageException("パッケージ、 $package_name の load.php が存在しません");
		require_once($load_path);
		*/
		$init_func = 'package_'.$package_name.'_initialize';
		if(!function_exists($init_func))
			throw new NoSuchPackageInitializerException("パッケージ、 $package_name の初期化函数が存在しません");
		$init_func($this);
	}
	
	private function prepare(){
		$this->removeComment();
		$this->encodeHTML();
		$this->enableEquation();
	}
	
	/**
	 * コメントを取り除く
	 */
	private function removeComment(){
		// 一行
		$this->result = preg_replace('/(?<!\\\\)%(.*)\r?\n/m', '', $this->result);
		// 複数行
		$this->result = preg_replace('/(?<!\\\\)\\/\\*(.*?)(?<!\\\\)\\*\\//s', '', $this->result);
	}
	
	/**
	 * HTML タグを変換
	 */
	private function encodeHTML(){
		$this->result = htmlentities($this->result, QUOTE_STYLE, ENCODING);
	}
	
	private function enableEquation(){
		$this->result = preg_replace('/(?<!\\\\)\\$(.*?)(?<!\\\\)\\$/s', ' \\\\equation{$1} ', $this->result);
	}
	
	/**
	 * コマンドを探し、実行する。
	 * ただし、すべてのコマンドが実行されるわけではない。
	 * @return int コマンド、または環境が実行された回数
	 */
	private function parse($is_delay){
		$this->isDelay = $is_delay;
		if($is_delay){
			$prefix = '@';
		}else{
			$prefix = '\\\\';
		}
		$pattern = '/(?<!\\\\)'.$prefix.'([_a-zA-Z][_a-zA-Z0-9]*)((?:\s*(?<!\\\\){.*?(?<!\\\\)})*)/s';
		
		$num_replace = 0;
		$offset = 0;
		for(;;){
			if($num_replace + $this->commandCalled + $this->environmentCalled > MAX_CALL){
				$this->result = substr_replace(
					$this->result,
					$this->callCommand('error', array('コマンド、または環境の実行回数制限を越えました')),
					$offset,
					0
				);
				return $num_replace;
			}
			$result = $this->findNextCommand($prefix, $offset);
			if($result === false)
				break;
			$num_replace++;
			// とりあえず展開してみる
			$start_offset = $result['begin_offset'];
			$length = $result['end_offset'] - $result['begin_offset'];
			$name = $result['name'];
			$args = $result['args'];
			// コマンド or 環境呼び出し
			//環境なら
			if($name == 'begin'){
				if(count($args) < 1){
					$result = $this->callCommand('error', array('環境名が指定されていません'));
				}else{
					$environment_name = trim($args[0]);
					array_shift($args);
//					$pattern_env = '/(?<!\\\\)'.$prefix.'end'.'\s*(?<!\\\\){\s*([_\w]+)(?<!\\\\)\s*}((?:\s*(?<!\\\\){.*?(?<!\\\\)})*)/s';
					$pattern_env = '/(?<!\\\\)'.$prefix.'end'.'\s*(?<!\\\\){\s*'.$environment_name.'(?<!\\\\)\s*}((?:\s*(?<!\\\\){.*?(?<!\\\\)})*)/s';
					$result_env = preg_match($pattern_env, $this->result, $matches, PREG_OFFSET_CAPTURE, $start_offset+$length);
					if($result_env == 0){
						$result = $this->callCommand('error', array($environment_name.' 環境: 環境の終了コマンドが見つかりませんでした'));
					}else{
						// 入れ子状になっていたら、こちらは無視する
//						if($matches[1][0] != $environment_name){
//							$offset++;
//							continue;
//						}else{
							$env_end_offset = $matches[0][1];
							$env_end_length = strlen($matches[0][0]);
							$main_arg = substr($this->result, $start_offset+$length, $env_end_offset-($start_offset+$length));
							// 呼ぶ
							$result = $this->callEnvironment($environment_name, $main_arg, $args);
							// lengthの調整
							$length = ($env_end_offset+$env_end_length) - $start_offset;
//						}
					}
				}
			// 環境の終了が見つかったら
			}else if($name == 'end'){
				$result = $this->callCommand('error', array("この環境は開始されていません (環境名={$name})"));
//				$offset++;
//				continue;
			//コマンドなら
			}else{
				$result = $this->callCommand($name, $args);
			}
			// そして置換
			$this->result = substr_replace($this->result, $result, $start_offset, $length);
			// オフセットの処理
			$offset = $start_offset;
			// 遅延型ならすぐに抜ける
			if($is_delay)
				return 1;
		}
		return $num_replace;
	}
	
	private function findNextCommand($prefix, $offset){
		// コマンド名のみ先に検出
		$pattern = '/(?<!\\\\)'.$prefix.'([_a-zA-Z][_a-zA-Z0-9]*)/s';
		$count = preg_match($pattern, $this->result, $matches, PREG_OFFSET_CAPTURE, $offset);
		if($count == 0)
			return false;
		
		$offset = $matches[0][1] + strlen($matches[0][0]);
		
		$result = array();
		$result['name'] = $matches[1][0];
		$result['args'] = array();
		$result['begin_offset'] = $matches[0][1];
		$result['end_offset'] = $offset;
		
		for(; $offset<strlen($this->result); $offset++){
			$c = $this->result[$offset];
			if($c == '{'){
				$begin = $offset + 1;
				$x = 0;
				for(; $offset<strlen($this->result); $offset++){
					if(($this->result[$offset] == '}') && ($this->result[$offset-1] != '\\'))
						$x--;
					if(($this->result[$offset] == '{') && ($this->result[$offset-1] != '\\'))
						$x++;
					if($x == 0)
						break;
				}
				if($offset == strlen($this->result)){
					$_result = array(
						'name' => 'error',
						'args' => array((($result['name']=='begin')?'環境':'コマンド'.$result['name']).'の"{"に対応する"}"がみつかりませんでした'),
						'begin_offset' => $result['begin_offset'],
						'end_offset' => strlen($this->result)
					);
					return $_result;
				}
				$length = $offset - $begin;
				$result['args'][] = substr($this->result, $begin, $length);
				$result['end_offset'] = $offset + 1;
			}else if(!$this->isWhite($c)){
				break;
			}
		}
		//$result['end_offset'] = $offset;
		return $result;
	}
	
	/** コマンド呼び出し */
	private function callCommand($name, $args){
		$command_name = 'command_'.$this->toRealFunctionName($name).'_call';
		if(function_exists($command_name)){
			$this->calling_name = $name;
			$this->calling_type = 'command';
			try{
				$result = $command_name($args, $this);
				return $this->makeResultCommand($result);
			}catch(InvalidArgumentCountException $e){
				$result = $this->callCommand('error', array($name.' コマンド: 引数の数が間違っています'));
				return $result;
			}
		// コマンドが存在しない！
		}else{
			return $this->callCommand('error', array("コマンドが見つかりませんでした (コマンド名={$name})"));
		}
	}
	
	/** 環境呼び出し */
	private function callEnvironment($name, $main_arg, $args){
		$environment_name = 'environment_'.$this->toRealFunctionName($name).'_call';
		if(function_exists($environment_name)){
			$this->calling_name = $name;
			$this->calling_type = 'environment';
			try{
				$result = $environment_name($main_arg, $args, $this);
				return $this->makeResultEnvironment($result);
			}catch(InvalidArgumentCountException $e){
				$result = $this->callCommand('error', array($name.' 環境: 引数の数が間違っています'));
				return $result;
			}
		// 環境が存在しない！
		}else{
			return $this->callCommand('error', array("環境が見つかりませんでした (環境名={$name})"));
		}
	}
	
	private function makeResultCommand($raw){
		return $this->makeResult($raw);
	}
	
	private function makeResultEnvironment($raw){
		$result = $this->makeResult($raw);
		// 環境ようの class 割り当て
		$result = "<div class=\"environment\">".trim($result)."</div>";
		return $result;
	}
	
	/** コマンドや環境の結果を整える */
	private function makeResult($raw){
		$result = preg_replace('/^(\s*\r?\n)+/m', '\\dummy'."\n", $raw);
		return $result;
	}
	
	private function format(){
		// 空行の処理
		$this->result = "<div class=\"paragraph\">\n".trim($this->result)."\n</div>";
		$this->result = preg_replace('/^(\s*\r?\n)+/m', '</div><div class="paragraph">'."\n", $this->result);
		// \によるエスケープを解除
		$this->result = preg_replace('/\\\\(.)/', '$1', $this->result);
		// ::verb:: の処理
		$this->result = str_replace('::verb::', '', $this->result);
	}
	
	public function getResult(){
		$result = new MilResult($this->source, $this->result, 0);
		$result->setTitle($this->title);
		return $result;
	}
	
	private static function isWhite($string){
		return in_array($string, MilParser::$white);
	}
	
	public static function toRealFunctionName($fname){
		$fname = str_replace('_', '__', $fname);
		$fname = preg_replace('/([A-Z])/', '_$1', $fname);
		return $fname;
	}
	
	public static function fromRealFunctionName($fname){
		$fname = preg_replace_callback(
			'/_([a-z])/',
			create_function(
				'$matches',
				'return strtoupper($matches[1]);'
			),
			$fname
		);
		$fname = str_replace('__', '_', $fname);
		return $fname;
	}
	
	public function isDelay(){
		return $this->isDelay;
	}
	
	public function addHead($cont){
		$this->result = $cont.$this->result;
	}
	
	public function addTail($cont){
		$this->result = $this->result.$cont;
	}
	
	public function getTitle(){
		return $this->title;
	}
	
	public function setTitle($title){
		$this->title = $title;
	}
	
	public function getVar($name){
		if(!isset($this->vars[$name]))
			return false;
		return $this->vars[$name];
	}
	
	public function setVar($name, $value){
		$this->vars[$name] = $value;
	}
	
	/** エラーメッセージを出力する */
	public function errorOut($msg){
		return '\\error{'.$msg.'}';
	}
	
	public static function commandUsage($command){
		$func = 'command_'.self::toRealFunctionName($command).'_usage';
		if(!function_exists($func)){
			return false;
		}
		$u = $func();
		if($u == null)
			return false;
		$u->setName($command);
		return $u;
	}
	
	public static function environmentUsage($command){
		$func = 'environment_'.self::toRealFunctionName($command).'_usage';
		if(!function_exists($func)){
			return false;
		}
		$u = $func();
		$u->setName($command);
		return $u;
	}
	
}


?>
