<?php

/*
 * 雑多なコマンド or 環境
 */

function command_vspace_call($args, $parser){
	if(count($args) == 0){
		throw new InvalidArgumentCountException();
	}
	return '<p style="margin-top:'.$args[0].';"></p>';
}

function command_vspace_usage(){
	$u = new MilUsageCommand('vspace');
	$u->setDescription('垂直方向にスペースを挿入します。');
	$u->setArgument('{ size }', array('size' => 'サイズ。em, px, \\% などで指定します。'));
	$u->addExample('この下に、\vspace{5ex}5ex 分のスペースを開ける。');
	return $u;
}

function command_hspace_call($args, $parser){
	if(count($args) == 0){
		throw new InvalidArgumentCountException();
	}
	return '<span style="margin-left:'.$args[0].';"></span>';
}

function command_hspace_usage(){
	$u = new MilUsageCommand('hspace');
	$u->setDescription('水平方向にスペースを挿入します。');
	$u->setArgument('{ size }', array('size' => 'サイズ。em, px, \\% などで指定します。'));
	$u->addExample('\hspace{4em}4em 分のスペースを開ける。タブの効果がある。');
	return $u;
}

function command_br_call(){
	return '<br />';
}

function command_br_usage(){
	$u = new MilUsageCommand('br');
	$u->setDescription('改行をします。');
	$u->addExample('一行開ける。\\br\\br一行開ける。');
	return $u;
}

function command_hr_call($args){
	$style = '';
	if(count($args) > 0){
		$style = $args[0];
	}
	return '<hr style="'.$style.'" />';
}

function command_hr_usage(){
	$u = new MilUsageCommand('hr');
	$u->setDescription('水平バーを引きます。');
	$u->setArgument('{ style }', array('style' => 'スタイル。'));
	$u->addExample('\\hr');
	$u->addExample('ちょっとした区切りに。\\hr{margin-left:0px;text-align:left;width:80px;}次の内容。');
	return $u;
}

function environment_itemize_call($arg_main, $args, $parser){
	return itemize_parse($arg_main, 'ul');
}

function environment_itemize_usage(){
	$u = new MilUsageEnvironment('itemize');
	$u->setDescription('番号なしのリストを生成します。');
	$u->setArgument(
		<<<EOM
		以下のように指定します。 \br
		\\\\item リスト項目1 \br
		\hspace{2em}[ \\\\subitem 子リスト項目1 \br
		\hspace{4em}[ \\\\subsubitem 孫リスト項目1 ] \br
		\hspace{4em}[ \\\\subsubitem 孫リスト項目2 ] \br
		\hspace{2em}] \br
		\\\\item リスト項目2 \br
		\\hspace{3em}\\equation{\\vdots}
EOM
	);
	$u->addExample(
		<<<EOM
		\\begin{itemize}
			\\item リスト項目1
				\\subitem 子リスト項目1
					\\subsubitem 孫リスト項目1
					\\subsubitem 孫リスト項目2
				\\subitem 子リスト項目2
					\\subsubitem 孫リスト項目1
					\\subsubitem 孫リスト項目2
			\\item リスト項目2
		\\end{itemize}
EOM
	);
	return $u;
}

function environment_enumerate_call($arg_main, $args, $parser){
	return itemize_parse($arg_main, 'ol');
}

function environment_enumerate_usage(){
	$u = new MilUsageEnvironment('enumerate');
	$u->setDescription('番号ありのリストを生成します。');
	$u->setArgument(
		<<<EOM
		以下のように指定します。 \br
		\\\\item リスト項目1 \br
		\hspace{2em}[ \\\\subitem 子リスト項目1 \br
		\hspace{4em}[ \\\\subsubitem 孫リスト項目1 ] \br
		\hspace{4em}[ \\\\subsubitem 孫リスト項目2 ] \br
		\hspace{2em}] \br
		\\\\item リスト項目2 \br
		\\hspace{3em}\\equation{\\vdots}
EOM
	);
	$u->addExample(
		<<<EOM
		\\begin{enumerate}
			\\item リスト項目1
				\\subitem 子リスト項目1
					\\subsubitem 孫リスト項目1
					\\subsubitem 孫リスト項目2
				\\subitem 子リスト項目2
					\\subsubitem 孫リスト項目1
					\\subsubitem 孫リスト項目2
			\\item リスト項目2
		\\end{enumerate}
EOM
	);
	return $u;
}

// \item / \subitem / \subsubitem で切り切り♪
function itemize_parse($string, $type){
	$tmp = preg_split(
		'/(?<!\\\\)\\\\(|sub|subsub)item/s',
		$string, -1, PREG_SPLIT_DELIM_CAPTURE
	);
	// $tmp[0] = \\(|sub|subsub)item前の文字列。
	// $tmp[1] = キャプチャ
	// $tmp[2] = 本体
	// and so on
	$item_pre = -1;
	$result = '';
	for($i=1; $i<count($tmp); $i+=2){
		$item = ($tmp[$i]=='')?0:(($tmp[$i]=='sub')?1:2);
		$content = trim($tmp[$i+1]);
		// 違う階層だった！
		if($item_pre != $item){
			if($item_pre != -1)
				$result .= "</{$type}>";
			$result .= "<{$type} class=\"list{$item}\">";
			$item_pre = $item;
		}
		$result .= "<li>{$content}</li>";
	}
	// 空じゃなければif文はいらんけど
	if($item_pre != -1)
		$result .= "</{$type}>";
	return $result;
}

// \begin{tabular}{(l|c|r||)+}[{(left|center|right)}]
function environment_tabular_call($main_arg, $args, $parser){
	if(count($args) < 1)
		throw new InvalidArgumentCountException();
	// まずは列定義を解釈
	$col_align = array();
	$col_border = array();
	$width = 0;
	$col_border[0] = false;
	$col_def_str = trim($args[0]);
	for($i=0; $i<strlen($col_def_str); $i++){
		switch($col_def_str[$i]){
			case 'l':
				$col_align[$width] = 'left';
				$width++;
				$col_border[$width] = false;
				break;
			case 'c':
				$col_align[$width] = 'center';
				$width++;
				$col_border[$width] = false;
				break;
			case 'r':
				$col_align[$width] = 'right';
				$width++;
				$col_border[$width] = false;
				break;
			case '|':
				$col_border[$width] = true;
				break;
			default:
				break;
		}
	}
	$result = '';
	$result .= '<table class="tabular" style="';
	if(count($args) == 2){
		$table_align = trim($args[1]);
	}else{
		$table_align = 'center';
	}
	switch($table_align){
			case 'left':
				$result .= 'margin-right:auto;';
				break;
			case 'right':
				$result .= 'margin-left:auto;';
				break;
			default:
				$result .= 'margin-left:auto;margin-right:auto;';
				break;
	}
	$result .= '">';
	// とりあえず\\でちょきちょき
	$raws = explode('\\\\', htmlspecialchars_decode($main_arg));
	for($r=0; $r<count($raws); $r++){
		$raw = trim($raws[$r]);
		if(($r+1 == count($raws)) && ($raw == ''))
			break;
		/*
		// \hlineをみつけたら
		if(strncmp($raw, '\\hline', 6) == 0){
			$above_border = true;
		}else{
			$above_border = false;
		}
		*/
		$elems = preg_split('/(?<!\\\\)&/', $raw);
		$result .= '<tr class="tabular">';
		for($c=0; $c<$width; $c++){
			$result .= '<td class="tabular" style="text-align:'.$col_align[$c].';">';
			if($c < count($elems)){
				$result .= trim($elems[$c]);
			}else{
				$result .= ' ';
			}
			$result .= '</td>';
		}
		$result .= '</tr>';
	}
	$result .= '</table>';
	return $result;
}

function environment_tabular_usage(){
	$u = new MilUsageEnvironment('tabular');
	$u->setDescription('表を作成する環境です。');
	$u->setArgument(
		'',
		'{ column_align } [ tabular_align ]',
		array(
			'column_align' => '各列の文字揃えを指定します。l は左寄せ(\\underline{l}eft)、c は中央寄せ(\\underline{c}enter)、r は右寄せ(\\underline{r}ight)です。\\br複数の列がある場合には、{ccc}といったように列数だけ文字を続けて書きます。',
			'tabular_align' => 'テーブル全体の位置揃えです。left で左に、center で中央に、right で右側に出力されます。未指定の場合には中央に出力されます。'
		)
	);
	$u->addExample('
		\\begin{tabular}{lccc}
			地名 & 北海道 & 東京 & 沖縄 \\\\
			特徴など & すごく寒い & 人が多い & 海開きが早い
		\\end{tabular}
	');
	$u->addExample('
		\\begin{tabular}{cc}{right}
			大文字 & 小文字 \\\\
			A & a \\\\
			B & b \\\\
			C & c \\\\
			\\equation{\\vdots} & \\equation{\\vdots}
		\\end{tabular}
	');
	return $u;
}

// \includegraphics{src}{alt}[{width}{height}[{link}]]
function command_includegraphics_call($args, $parser){
	$width = '100%';
	$height = '100%';
	$href = false;
	switch(count($args)){
		case 5:
			$href = trim($args[4]);
		case 4:
			$width = trim($args[2]);
			$height = trim($args[3]);
		case 2:
			$src = trim($args[0]);
			$alt = trim($args[1]);
			break;
		default:
			throw new InvalidArgumentCountException();
	}
	if($href === false){
		$href = $src;
	}
	$result = '<a href="'.$href.'" title="'.$alt.'" class="image"><img src="'.$src.'" alt="'.$alt.'" class="includegraphics" style="width:'.$width.';height:'.$height.';" /></a>';
	return $result;
}

function command_includegraphics_usage(){
	$u = new MilUsageCommand('includegraphics');
	$u->setDescription('画像を出力します。');
	$u->setArgument(
		'{ src } { alt } [ { width } { height } [ { link } ] ]',
		array(
			'src' => '画像へのパスを指定します。',
			'width' => '画像の幅を指定します。',
			'height' => '画像の高さを指定します。',
			'link' => '画像にリンクを設定します。未指定の場合には、画像の URL が設定されます。'
		)
	);
	$u->addExample('
		リンクを設定しない(クリックすると画像の URL へジャンプします)。
		\\begin{center}
		\\includegraphics{http://getfirefox.jp/b/88x31_1_orange}{Mozilla Firefox}{88px}{31px}
		\\end{center}
	');
	$u->addExample('
		リンクを設定する(クリックすると Firefox のダウンロードページへ飛びます)。
		\\begin{center}
		\\includegraphics{http://getfirefox.jp/b/88x31_1_orange}{Mozilla Firefox}{88px}{31px}{http://getfirefox.jp/}
		\\end{center}
	');
	return $u;
}

?>
