import unittest
from MindMap import *

class MindMapRootTestCase(unittest.TestCase):
  def setUp(self):
    self.mmr = MindMapRoot('root')
    self.root_code = '0'
    self.child0_code  = self.mmr.add(self.root_code,   'child0')
    self.child00_code = self.mmr.add(self.child0_code, 'child00')
    self.child1_code  = self.mmr.add(self.root_code,   'child1')
    self.child10_code = self.mmr.add(self.child1_code, 'child10')
    self.child11_code = self.mmr.add(self.child1_code, 'child11')

  def testIdNumber(self):
    self.assertEqual('0', self.root_code)
    self.assertEqual('1', self.child0_code)
    self.assertEqual('2', self.child00_code)
    self.assertEqual('3', self.child1_code)
    self.assertEqual('4', self.child10_code)
    self.assertEqual('5', self.child11_code)

    self.assertEqual('0', self.mmr.decode_map[self.root_code].code)
    self.assertEqual('1', self.mmr.decode_map[self.child0_code].code)
    self.assertEqual('2', self.mmr.decode_map[self.child00_code].code)
    self.assertEqual('3', self.mmr.decode_map[self.child1_code].code)
    self.assertEqual('4', self.mmr.decode_map[self.child10_code].code)
    self.assertEqual('5', self.mmr.decode_map[self.child11_code].code)

  def testColor(self):
    self.assertEqual('black', self.mmr.child.color)
 
  def testChangeRoot(self):
    child = self.mmr.child
    self.assertEqual('root', child.message)
    self.assertEqual('black', child.color)
    self.assertEqual('', child.link)

    self.mmr.change(self.root_code, 'changed message', 'red', 'link', 'linkType')
    self.assertEqual('changed message', child.message)
    self.assertEqual('red', child.color)
    self.assertEqual('link', child.link)
    self.assertEqual('linkType', child.linkType)

  def testChangeChild10(self):
    child10 = self.mmr.decode_map[self.child10_code]
    self.assertEqual('child10', child10.message)
    self.assertEqual('black', child10.color)

    self.mmr.change(self.child10_code, 'changed message', 'red', 'link', 'linkType')
    self.assertEqual('changed message', child10.message)
    self.assertEqual('red', child10.color)
    self.assertEqual('link', child10.link)
    self.assertEqual('linkType', child10.linkType)

  def testAdd(self):
    child10 = self.mmr.decode_map[self.child10_code]
    child10.color = 'green'

    code = self.mmr.add(self.child10_code, 'add message')
    self.assertEqual('6', code)

    child100 = self.mmr.decode_map[code]
    self.assertEqual('add message', child100.message)
    self.assertEqual('green', child100.getColor())

  def testCreateFromXML1(self):
    xml_string = '<map>\n'
    xml_string += ' <node TEXT="parent"/>\n'
    xml_string += '</map>\n'
    root = self.mmr.createFromXML(xml_string)
    parent = root.child
    self.assertEqual('parent', parent.message)

  def testCreateFromXML1n(self):
    xml_string = '<map>\n'
    xml_string += ' <node TEXT="parent\nsecond"/>\n'
    xml_string += '</map>\n'
    root = self.mmr.createFromXML(xml_string)
    parent = root.child
    self.assertEqual('parent second', parent.message)

  def testCreateFromXML2(self):
    xml_string = '<map>\n'
    xml_string += ' <node TEXT="parent">\n'
    xml_string += '  <node TEXT="child"/>\n'
    xml_string += ' </node>\n'
    xml_string += '</map>\n'
    root = self.mmr.createFromXML(xml_string)
    parent = root.child
    self.assertEqual('parent', parent.message)

    child = parent.children[0]
    self.assertEqual('child', child.message)
    

class MindMapTestCase(unittest.TestCase):
  def setUp(self):
    self.mindmap = MindMap('parent')
    self.child1 = MindMap('child1')
    self.child11 = MindMap('child11')
    self.child2 = MindMap('child2')
    self.child21 = MindMap('child21')
    self.child22 = MindMap('child22')

    self.mindmap.addChild(self.child1)
    self.mindmap.addChild(self.child2)
    self.child1.addChild(self.child11)
    self.child2.addChild(self.child21)
    self.child2.addChild(self.child22)

    self.mindmap.name = 'parent'
    self.child1.name  = 'child1'
    self.child11.name = 'child11'
    self.child2.name  = 'child2'
    self.child21.name = 'child21'
    self.child22.name = 'child2'
 
  def testMessage(self):
    self.assertEqual('parent', self.mindmap.message)
    self.assertEqual('child1', self.child1.message)
    self.assertEqual('child2', self.child2.message)
    self.assertEqual('child11', self.child11.message)
    self.assertEqual('child21', self.child21.message)
    self.assertEqual('child22', self.child22.message)

  def testColor(self):
    self.assertEqual('black', self.mindmap.color)

  def testGetColor(self):
    self.mindmap.color = 'blue'
    self.assertEqual('blue', self.mindmap.color)
    self.assertEqual('blue', self.mindmap.getColor())

    self.mindmap.color = ''
    self.assertEqual('', self.mindmap.color)
    self.assertEqual('black', self.mindmap.getColor())


  def testLink(self):
    self.assertEqual('', self.mindmap.link)
    self.assertEqual('', self.mindmap.linkType)

  def testChange(self):
    import time
    original_time = time.time
    time.time = mockTime

    self.mindmap.change('changed message', 'changed color')
    self.assertEquals('changed message', self.mindmap.message)
    self.assertEquals('changed color', self.mindmap.color)
    self.assertEquals(time.mktime((2004, 1, 11, 14, 38, 44, 6, 11, 0)),
                      self.mindmap.time)

    time.time = original_time

  def testRefresh(self):
    self.mindmap.refresh()
    self.assertEquals([self.child1, self.child2], self.mindmap.children)

  def testRefreshChild1(self):
    self.assertEquals([self.child11], self.child1.children)

    self.child11.message = ""
    self.mindmap.refresh()
    self.assertEquals([], self.child1.children)

  def testRefreshMindmap2(self):
    self.assertEquals([self.child1, self.child2], self.mindmap.children)

    self.child1.message = ""
    self.mindmap.refresh()
    self.assertEquals([self.child1, self.child2], self.mindmap.children)

    self.child11.message = ""
    self.mindmap.refresh()
    self.assertEquals([self.child2], self.mindmap.children)


def mockTime():
  return time.mktime((2004, 1, 11, 14, 38, 44, 6, 11, 0))

class BirthdayTestCase(unittest.TestCase):
  import time
  def setUp(self):
    self.original_time = time.time
    time.time = mockTime
    self.birthday = Birthday()

  def testTime(self):
    self.assertEqual(time.mktime((2004, 1, 11, 14, 38, 44, 6, 11, 0)),
                     self.birthday.time)

  def testAmountOfSecond(self):
    self.birthday.time = time.mktime((2004, 1, 11, 14, 38, 42, 6, 11, 0))
    self.assertEqual(2, self.birthday.amountOfSecond())

  def testRebirth(self):
    self.birthday.time = time.mktime((2004, 1, 11, 14, 38, 42, 6, 11, 0))
    self.birthday.rebirth()
    self.assertEqual(time.mktime((2004, 1, 11, 14, 38, 44, 6, 11, 0)),
                     self.birthday.time)

  def tearDown(self):
    time.time = self.original_time


class CompositeTestCase(unittest.TestCase):
  def setUp(self):
    self.parent  = Composite()
    self.child1  = Composite()
    self.child2  = Composite()
    self.child11 = Composite()
    self.child21 = Composite()
    self.child22 = Composite()

    self.parent.addChild(self.child1)
    self.parent.addChild(self.child2)
    self.child1.addChild(self.child11)
    self.child2.addChild(self.child21)
    self.child2.addChild(self.child22)

  def testChildren(self):
    self.assertEqual([self.child1, self.child2],   self.parent.children)
    self.assertEqual([self.child11],               self.child1.children)
    self.assertEqual([self.child21, self.child22], self.child2.children)
    self.assertEqual([], self.child11.children)
    self.assertEqual([], self.child21.children)
    self.assertEqual([], self.child22.children)

  def testAddChild(self):
    self.assertEqual([self.child11], self.child1.children)
    
    child12 = Composite()
    self.child1.addChild(child12)
    self.assertEqual([self.child11, child12], self.child1.children)

  def testRemoveChild(self):
    self.assertEqual([self.child21, self.child22], self.child2.children)
    
    self.child2.removeChild(self.child21)
    self.assertEqual([self.child22], self.child2.children)

if __name__ == "__main__":
  unittest.main()
