﻿//-----------------------------------------------------------------------------
// MMDModel.fx
//
// MikuMikuDance for XNA
// Copyright (C) Wilfrem and Microsoft.
//
// このソースコードはひにけにXNAの「頂点テクスチャでスキンアニメーション」の
// SkinnedModel.fxのソースと
// http://blogs.msdn.com/ito/archive/2009/05/07/more-bones-07.aspx
// XNAクリエイターズオンラインにあったBasicEffect.fxを混ぜた上で
// エフェクトMMDに合わせて変更したもの
// 他、表情、トゥーンやらいろいろいじってる
// 
//-----------------------------------------------------------------------------

//既知の問題
//XBox用にfxファイルをコンパイルする際に、
//ピクセルシェーダの構造体内に未使用のデータがあると、コンパイルエラーを起こす
//http://forums.xna.com/forums/t/42630.aspx

//-----------------------------------------------------------------------------
// 定数レジスタ宣言
//=============================================================================

//-----------------------------------------------------------------------------
// フォグ設定用レジスタ
//-----------------------------------------------------------------------------
//処理量軽減のためにカット
//uniform const float		FogEnabled		: register(c0);
//uniform const float		FogStart		: register(c1);
//uniform const float		FogEnd			: register(c2);
//uniform const float3	FogColor		: register(c3);

uniform const float3	EyePosition		: register(c4);		// in world space


//-----------------------------------------------------------------------------
// マテリアル設定用レジスタ
//-----------------------------------------------------------------------------
uniform const float3	DiffuseColor	: register(c5) = 1;
uniform const float		Alpha			: register(c6) = 1;
uniform const float3	EmissiveColor	: register(c7) = 0;
uniform const float3	SpecularColor	: register(c8) = 1;
uniform const float		SpecularPower	: register(c9) = 16;


//-----------------------------------------------------------------------------
// ライト設定用レジスタ。
// All directions and positions are in world space and must be unit vectors
//-----------------------------------------------------------------------------

uniform shared const float3	AmbientLightColor		: register(c10);

uniform shared const float3	DirLight0Direction		: register(c11);
uniform shared const float3	DirLight0DiffuseColor	: register(c12);
//uniform shared const float3	DirLight0SpecularColor	: register(c13);

uniform shared const float3	DirLight1Direction		: register(c14);
uniform shared const float3	DirLight1DiffuseColor	: register(c15);
//uniform shared const float3	DirLight1SpecularColor	: register(c16);

uniform shared const float3	DirLight2Direction		: register(c17);
uniform shared const float3	DirLight2DiffuseColor	: register(c18);
//uniform shared const float3	DirLight2SpecularColor	: register(c19);

//-----------------------------------------------------------------------------
// マトリックス
//-----------------------------------------------------------------------------
// オブジェクトのワールド座標
uniform const float4x4	World		: register(vs, c20);	// 20 - 23
// ビューのトランスフォーム
uniform shared const float4x4	View		: register(vs, c24);	// 24 - 27
// プロジェクションのトランスフォーム
uniform shared const float4x4	Projection	: register(vs, c28);	// 28 - 31

//-----------------------------------------------------------------------------
// トゥーン処理
//-----------------------------------------------------------------------------
// トゥーンライティング
uniform const float2 ToonThresholds			: register(c32);
uniform const float3 ToonBrightnessLevels	: register(c33);
uniform const bool  UseToonLighting		: register(c34);

//-----------------------------------------------------------------------------
// スフィア処理
//-----------------------------------------------------------------------------
// スフィアビットマップ
float  UseSphere;

//-----------------------------------------------------------------------------
// 構造体宣言
//-----------------------------------------------------------------------------

struct ColorPair
{
	float3 Diffuse;
	float3 Specular;
	float  LightAmount;
};

//-----------------------------------------------------------------------------
// テスクチャ
//-----------------------------------------------------------------------------

texture Texture;		// テクスチャ

//テクスチャのサンプラー
sampler Sampler = sampler_state
{
    Texture = (Texture);

    MinFilter = Linear;
    MagFilter = Linear;
    MipFilter = Linear;
};

//-----------------------------------------------------------------------------
// 頂点テクスチャ用の定数レジスタ宣言
//=============================================================================
float2 BoneTextureSize;	// ボーン用頂点テクスチャのサイズ

// ボーン用頂点テクスチャサンプラー宣言
texture BoneRotationTexture;

sampler BoneRotationSampler : register(vs,s0) = sampler_state
{
	Texture = (BoneRotationTexture);
	// 殆どのGPUでは以下のようなステート設定にしないと
	// 頂点テクスチャのフェッチがうまくいかない
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
	AddressU = Clamp;
	AddressV = Clamp;
};

texture BoneTranslationTexture;

sampler BoneTranslationSampler : register(vs,s1) = sampler_state
{
	Texture = (BoneTranslationTexture);
	// 殆どのGPUでは以下のようなステート設定にしないと
	// 頂点テクスチャのフェッチがうまくいかない
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
	AddressU = Clamp;
	AddressV = Clamp;
};

//表情の適応割合をセットする頂点テクスチャ
//フォーマット：表情番号-割合がfloat配列で格納されている
//毎フレーム転送される
float2 FaceRateTextureSize;//フェイス割合用の頂点テクスチャのサイズ
texture FaceRateTexture;

sampler FaceRateSampler : register(vs,s2) = sampler_state
{
	Texture = (FaceRateTexture);
	// 殆どのGPUでは以下のようなステート設定にしないと
	// 頂点テクスチャのフェッチがうまくいかない
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
	AddressU = Clamp;
	AddressV = Clamp;
};

bool UseGPUAnime;//GPU表情アニメーションフラグ

//表情のbase頂点番号から移動量が格納されているテクスチャ(CPUアニメーション時)
//表情のbase頂点番号から移動量を計算するまでに必要なデータが格納されているテクスチャ(GPUアニメーション時)
//フォーマット：
//頂点表情セクタのヘッダ(Vector4に割り当て)
//struct VertToFace{
//	float start;//base頂点→表情番号が書かれたフィールドの開始点(Vector4個数単位)
//	float count;//base頂点→表情番号が書かれたフィールドの個数(表情数ではなくVector4の個数)
//  float vstart;//base頂点→表情ごとの移動量が書かれたフィールドの開始点
//  float vcount;//base頂点→表情ごとの移動量が書かれたフィールドの個数(count*4-αと同じになるはずだが)
//};
//頂点→表情番号フィールド(float:Vector4に4個づつ割り当て)
//base頂点→表情ごとの移動量(Vector4。wは0)。indexは頂点→表情番号フィールドと一致

float2 FaceVertTextureSize;//フェイス頂点データ用の頂点テクスチャのサイズ
float2 FaceVertTextureColLines;//フェイス頂点データ用の頂点テクスチャの行数
texture FaceVertTexture;

sampler FaceVertSampler : register(vs,s3) = sampler_state
{
	Texture = (FaceVertTexture);
	// 殆どのGPUでは以下のようなステート設定にしないと
	// 頂点テクスチャのフェッチがうまくいかない
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
	AddressU = Clamp;
	AddressV = Clamp;
};
//-----------------------------------------------------------------------------
// 構造体宣言
//=============================================================================
// 頂点シェーダー入力構造体
// MMDのモデルは各種色、法線は必ず付いており、テクスチャがオプションである
// そのため、テクスチャのオン、オフの2つの入力構造体を用意すればいける

struct VS_INPUT_Tx
{
    float4 Position : POSITION0;
    float3 Normal	: NORMAL0;
    float2 TexCoord : TEXCOORD0;
    float4 Color	: COLOR;
    float4 BoneIndices : BLENDINDICES0;//ボーン
    float4 BoneWeights : BLENDWEIGHT0;//ボーン
    float2 VectorIndex : TEXCOORD1;//頂点番号
};

// 頂点シェーダー出力構造体
struct VertexLightingVSOutputTx
{
	float4	Position	: POSITION;		// Position in projection space
	float4	Diffuse		: COLOR0;
	float4	Specular	: COLOR1;
	float2	TexCoord	: TEXCOORD0;
	float	LightAmount	: TEXCOORD1;	//トゥーンライティング用光量
};

//-----------------------------------------------------------------------------
// クォータニオンヘルパーメソッド
//=============================================================================
// 4つクォータニオンと平行移動から行列に変換する
// SM2.0の一時レジスタ(12個)数制限を回避するために、一時レジスタの使用量を抑えるように
// 書き換えたもの
float4x4 CreateTransformFromQuaternionTransforms(
		float4 q1, float3 t1,
		float4 q2, float3 t2,
		float4 q3, float3 t3,
		float4 q4, float3 t4,
		float4 weights )
{
	float ww = q1.w * q1.w - 0.5f;
	float3 row10 = float3( ww         , q1.x * q1.y, q1.x * q1.z ) +
				   float3( q1.x * q1.x, q1.w * q1.z,-q1.w * q1.y );
	float3 row11 = float3( q1.x * q1.y, ww,          q1.y * q1.z ) +
	               float3(-q1.w * q1.z, q1.y * q1.y, q1.w * q1.x );
	float3 row12 = float3( q1.x * q1.z, q1.y * q1.z, ww          ) +
	               float3( q1.w * q1.y,-q1.w * q1.x, q1.z * q1.z );
	
	ww = q2.w * q2.w - 0.5f;
	float3 row20 = float3( ww,          q2.x * q2.y, q2.x * q2.z ) +
	               float3( q2.x * q2.x, q2.w * q2.z,-q2.w * q2.y );
	float3 row21 = float3( q2.x * q2.y, ww,          q2.y * q2.z ) +
	               float3(-q2.w * q2.z, q2.y * q2.y, q2.w * q2.x );
	float3 row22 = float3( q2.x * q2.z, q2.y * q2.z, ww          ) +
	               float3( q2.w * q2.y,-q2.w * q2.x, q2.z * q2.z );
	
	ww = q3.w * q3.w - 0.5f;
	float3 row30 = float3( ww,          q3.x * q3.y, q3.x * q3.z ) +
	               float3( q3.x * q3.x, q3.w * q3.z,-q3.w * q3.y );
	float3 row31 = float3( q3.x * q3.y, ww,          q3.y * q3.z ) +
	               float3(-q3.w * q3.z, q3.y * q3.y, q3.w * q3.x );
	float3 row32 = float3( q3.x * q3.z, q3.y * q3.z, ww          ) +
	               float3( q3.w * q3.y,-q3.w * q3.x, q3.z * q3.z );
	
	ww = q4.w * q4.w - 0.5f;
	float3 row40 = float3( ww,          q4.x * q4.y, q4.x * q4.z ) +
	               float3( q4.x * q4.x, q4.w * q4.z,-q4.w * q4.y );
	float3 row41 = float3( q4.x * q4.y, ww,          q4.y * q4.z ) +
	               float3(-q4.w * q4.z, q4.y * q4.y, q4.w * q4.x );
	float3 row42 = float3( q4.x * q4.z, q4.y * q4.z, ww          ) +
	               float3( q4.w * q4.y,-q4.w * q4.x, q4.z * q4.z );
	               
	float4 w2 = 2.0f * weights;
	
	return float4x4(
		row10 * w2.x + row20 * w2.y + row30 * w2.z + row40 * w2.w, 0,
		row11 * w2.x + row21 * w2.y + row31 * w2.z + row41 * w2.w, 0,
		row12 * w2.x + row22 * w2.y + row32 * w2.z + row42 * w2.w, 0, 
		t1 * weights.x + t2 * weights.y + t3 * weights.z + t4 * weights.w, 1
	);
	
}

//-----------------------------------------------------------------------------
// 頂点テクスチャからボーン情報のフェッチ
//=============================================================================
float4x4 CreateTransformFromBoneTexture( float4 boneIndices, float4 boneWeights )
{
	float2 uv = 1.0f / BoneTextureSize;
	uv.y *= 0.5f;
	float4 texCoord0 = float4( ( 0.5f + boneIndices.x ) * uv.x, uv.y, 0, 1 );
	float4 texCoord1 = float4( ( 0.5f + boneIndices.y ) * uv.x, uv.y, 0, 1 );
	float4 texCoord2 = float4( ( 0.5f + boneIndices.z ) * uv.x, uv.y, 0, 1 );
	float4 texCoord3 = float4( ( 0.5f + boneIndices.w ) * uv.x, uv.y, 0, 1 );

	/*float4 texCoord0 = float4( ( 0.5f + fmod(boneIndices.x,BoneTextureSize.x) ) * uv.x, ( 0.5f + floor(boneIndices.x/BoneTextureSize.x) ) *uv.y, 0, 1 );
	float4 texCoord1 = float4( ( 0.5f + fmod(boneIndices.y,BoneTextureSize.x) ) * uv.x, ( 0.5f + floor(boneIndices.y/BoneTextureSize.x) ) *uv.y, 0, 1 );
	float4 texCoord2 = float4( ( 0.5f + fmod(boneIndices.z,BoneTextureSize.x) ) * uv.x, ( 0.5f + floor(boneIndices.z/BoneTextureSize.x) ) *uv.y, 0, 1 );
	float4 texCoord3 = float4( ( 0.5f + fmod(boneIndices.w,BoneTextureSize.x) ) * uv.x, ( 0.5f + floor(boneIndices.w/BoneTextureSize.x) ) *uv.y, 0, 1 );
	*/
	// 回転部分のフェッチ
	float4 q1 = tex2Dlod( BoneRotationSampler, texCoord0 );
	float4 q2 = tex2Dlod( BoneRotationSampler, texCoord1 );
	float4 q3 = tex2Dlod( BoneRotationSampler, texCoord2 );
	float4 q4 = tex2Dlod( BoneRotationSampler, texCoord3 );

	// 平行移動部分のフェッチ
	float4 t1 = tex2Dlod( BoneTranslationSampler, texCoord0 );
	float4 t2 = tex2Dlod( BoneTranslationSampler, texCoord1 );
	float4 t3 = tex2Dlod( BoneTranslationSampler, texCoord2 );
	float4 t4 = tex2Dlod( BoneTranslationSampler, texCoord3 );
	
	return CreateTransformFromQuaternionTransforms(
					q1, t1,
					q2, t2,
					q3, t3,
					q4, t4,
					boneWeights );
}

//-----------------------------------------------------------------------------
// 頂点テクスチャから表情情報のフェッチ及び計算
//=============================================================================
float4 GetFVTexCoord(float pos,float2 uv)
{
	return float4(fmod(pos+0.5f,FaceVertTextureColLines.x)* uv.x,(0.5f+floor(pos*FaceVertTextureColLines.y))* uv.y, 0, 1 );
}
float4 InnerCalcFace(int faceNum,float vertPos,float2 uv,float2 uv2)
{
	float4 result=0;
	if(faceNum>=0)
	{//余り部分には-1が入っている
		//faceNumに表情番号が入っている
		//ここから表情量を取得
		float4 texCoord2=float4((0.5f+faceNum) * uv2.x, uv2.y, 0, 1 );
		float4 faceRate=tex2Dlod(FaceRateSampler,texCoord2);
		if(faceRate.x>0)
		{
			//次に対応する頂点の移動量をフェッチ
			//vertPosの位置に来るようになっている
			float4 texCoord3=GetFVTexCoord(vertPos,uv);
			float4 faceMove=tex2Dlod( FaceVertSampler, texCoord3 );
			//表情の適応量を掛け合わせて加える
			result+=faceMove*faceRate.x;
		}
	}
	
	return result;
}
float4 CalcFace( float2 VectorIndex)
{
	//FaceVertTextureのVector4 1データの目盛取得(複数行形式)
	float2 uv = 1.0f / FaceVertTextureSize;
	//頂点番号から(表情が書かれたエリア/表情頂点位置)の取得
	//Vector4ごとにに2つ含まれている(GPU)
	float4 texCoord0=GetFVTexCoord(VectorIndex.x,uv);
	// 関連表情データエリア部分/頂点位置のフェッチ
	float4 fa = tex2Dlod( FaceVertSampler, texCoord0 );
	float4 result=float4(0,0,0,0);
	if(UseGPUAnime==true)
	{//GPU処理時はシェーダ内で計算する
		//FaceRateTextureのfloat 1データの目盛取得
		float2 uv2=1.0f/FaceRateTextureSize;
		uv2.y*=0.5f;
		
		//表情ごとに処理
		for(int i=0;i<fa.y;i++)
		{
			float4 texCoord1=GetFVTexCoord(i+fa.x,uv);
			//表情番号データエリア部分のフェッチ
			float4 faces=tex2Dlod( FaceVertSampler, texCoord1 );
			result+=InnerCalcFace(faces.x,fa.z+i*4.0f,uv,uv2);
			result+=InnerCalcFace(faces.y,fa.z+i*4.0f+1,uv,uv2);
			result+=InnerCalcFace(faces.z,fa.z+i*4.0f+2,uv,uv2);
			result+=InnerCalcFace(faces.w,fa.z+i*4.0f+3,uv,uv2);
			
		}
	}
	else
	{//CPU処理時は処理済みデータが格納されている
		result=fa;
	}
	return result;
}



//-----------------------------------------------------------------------------
// Compute lighting
// E: Eye-Vector
// N: Unit vector normal in world space
//-----------------------------------------------------------------------------
ColorPair ComputeLights(float3 E, float3 N)
{
	ColorPair result;
	
	result.Diffuse = AmbientLightColor;
	result.Specular = 0;
	result.LightAmount=0;
	
	// Directional Light 0
	float3 L = -DirLight0Direction;
	float3 H = normalize(E + L);
	float2 ret = lit(dot(N, L), dot(N, H), SpecularPower).yz;
	result.Diffuse += DirLight0DiffuseColor * ret.x;
	//result.Specular += DirLight0SpecularColor * ret.y;
	//光の量(トゥーン処理用)
	result.LightAmount+=dot(normalize(N),normalize(DirLight0Direction));
	
	// Directional Light 1
	L = -DirLight1Direction;
	H = normalize(E + L);
	ret = lit(dot(N, L), dot(N, H), SpecularPower).yz;
	result.Diffuse += DirLight1DiffuseColor * ret.x;
	//result.Specular += DirLight1SpecularColor * ret.y;
	//光の量(トゥーン処理用)
	result.LightAmount+=dot(normalize(N),normalize(DirLight1Direction));
	
	// Directional Light 2
	L = -DirLight2Direction;
	H = normalize(E + L);
	ret = lit(dot(N, L), dot(N, H), SpecularPower).yz;
	result.Diffuse += DirLight2DiffuseColor * ret.x;
	//result.Specular += DirLight2SpecularColor * ret.y;
	//光の量(トゥーン処理用)
	result.LightAmount+=dot(normalize(N),normalize(DirLight0Direction));
	
	result.Diffuse *= DiffuseColor;
	result.Diffuse	+= EmissiveColor;
	result.Specular	*= SpecularColor;
		
	return result;
}
//-----------------------------------------------------------------------------
// Compute fog factor
//-----------------------------------------------------------------------------
//float ComputeFogFactor(float d)
//{
//    return clamp((d - FogStart) / (FogEnd - FogStart), 0, 1) * FogEnabled;
//}

//-----------------------------------------------------------------------------
// 頂点シェーダー
//=============================================================================
VertexLightingVSOutputTx VSBasicNmTxVc(VS_INPUT_Tx input)
{
    VertexLightingVSOutputTx output;
    
    // スキン変換行列の取得
    float4x4 skinTransform =
				CreateTransformFromBoneTexture( input.BoneIndices, input.BoneWeights );
	//スキン変換
	skinTransform = mul( skinTransform, World );
  
	//表情移動分の取得
	float4 faceTransform=float4(0,0,0,0);
	if(input.VectorIndex.x>=0)
		faceTransform=CalcFace(input.VectorIndex);
	
	//表情による頂点移動を合算
	float4 facePos=input.Position+faceTransform;
	//if(any(faceTransform))
	//	facePos=facePos+faceTransform;
	
    // 頂点変換
    float4 position = mul(facePos, skinTransform);
    output.Position = mul(mul(position, View), Projection);

    // 法線変換
    float3 normal = normalize( mul( input.Normal, skinTransform));
    //視線ベクトル取得
    float3 posToEye = EyePosition - position;
    float3 E = normalize(posToEye);
    //視線ベクトルと法線を元にライトを計算
    ColorPair lightResult = ComputeLights(E, normal);
    
    //ディフューズ色計算
    output.Diffuse = float4(lightResult.Diffuse.rgb, Alpha) * input.Color;
    //output.Specular = float4(lightResult.Specular, ComputeFogFactor(length(posToEye)));
    //output.Diffuse = float4(lightResult.Diffuse.rgb, 1) * input.Color;
    output.Specular = float4(lightResult.Specular,0);
    
	if(UseSphere>0)
		output.TexCoord=float2(normal.x/2+0.5,normal.y/2+0.5);
	else
		output.TexCoord=input.TexCoord;
	
	//光量をコピー
	output.LightAmount=lightResult.LightAmount;
    return output;
}
//-----------------------------------------------------------------------------
// ピクセルシェーダー
//=============================================================================
// ピクセルシェーダー入力構造体
struct VertexLightingPSInputTx
{
	float4	Diffuse		: COLOR0;
	float4	Specular	: COLOR1;
    float2	TexCoord : TEXCOORD0;
    float	LightAmount	: TEXCOORD1;
};

float4 PSBasicTx(VertexLightingPSInputTx pin) : COLOR
{
	
	float4 color;
	float  toonLight=1.0;
	if(all(pin.TexCoord))
	{
		color=tex2D(Sampler, pin.TexCoord) * pin.Diffuse + float4(pin.Specular.rgb, 0);
	}
	else
		color = pin.Diffuse + float4(pin.Specular.rgb, 0);
	//color.rgb = lerp(color.rgb, FogColor, pin.Specular.w);
	color.rgb=color.rgb;
	if(UseToonLighting>0)
	{
		if (pin.LightAmount > ToonThresholds.y)
			toonLight = ToonBrightnessLevels.z;
		else if (pin.LightAmount > ToonThresholds.x)
			toonLight = ToonBrightnessLevels.y;
		else
			toonLight = ToonBrightnessLevels.x;
	}
	return color*toonLight;
}

//-----------------------------------------------------------------------------
// 描画手法の宣言
//=============================================================================
//修正前
//int ShaderIndex = 0;

//VertexShader VSArray[1] =
//{
//	compile vs_3_0 VSBasicNmTxVc(),
//};

//PixelShader PSArray[1] =
//{
//	compile ps_3_0 PSBasicTx(),
//};


//Technique MMDBasicEffect
//{
//	Pass
//	{
//		VertexShader = (VSArray[ShaderIndex]);
//		PixelShader	 = (PSArray[ShaderIndex]);
//	}
//}

//修正後
//上のように書くと、エラーや警告が出たときにXBoxだとエフェクトコンパイラが落ちて謎エラーとなる。
//http://forums.xna.com/forums/t/42630.aspx

Technique MMDBasicEffect
{
	Pass
	{
		VertexShader = (compile vs_3_0 VSBasicNmTxVc());
		PixelShader	 = (compile ps_3_0 PSBasicTx());
	}
}