﻿//-----------------------------------------------------------------------------
// MMDModelReach.fx
//
// MikuMikuDance for XNA
// Copyright (C) Wilfrem and Microsoft.
//
// このソースコードはひにけにXNAの「頂点テクスチャでスキンアニメーション」と
// XNAクリエイターズオンラインにあったBasicEffect.fxを混ぜたもの。
// MMDModel.fxと違い、shader2.0モデルで調整
// 
//-----------------------------------------------------------------------------

//既知の問題(XNAチームに報告済み)
//XBox用にfxファイルをコンパイルする際に、
//ピクセルシェーダの構造体内に未使用のデータ、または未初期化変数があるとコンパイルエラーを起こす
//http://forums.xna.com/forums/t/42630.aspx

// 最大ボーン数
#define MaxBones 100

//-----------------------------------------------------------------------------
// 定数レジスタ宣言
//=============================================================================
uniform shared const float3	EyePosition				: register(c0);		// in world space


//-----------------------------------------------------------------------------
// 材質レジスタ宣言
//=============================================================================
uniform const float3	DiffuseColor	: register(c1) = 1;
uniform const float		Alpha			: register(c2) = 1;
uniform const float3	EmissiveColor	: register(c3) = 0;
uniform const float3	SpecularColor	: register(c4) = 1;
uniform const float		SpecularPower	: register(c5) = 16;

//-----------------------------------------------------------------------------
// ライト設定用レジスタ。
// All directions and positions are in world space and must be unit vectors
//-----------------------------------------------------------------------------
uniform shared const float3	AmbientLightColor		: register(c6);
uniform shared const float3	DirLight0Direction		: register(c7);


//-----------------------------------------------------------------------------
// マトリックス
//-----------------------------------------------------------------------------
// オブジェクトのワールド座標
uniform const float4x4	World						: register(vs, c8);		// 8 - 11
// ビューのトランスフォーム
uniform shared const float4x4	View				: register(vs, c12);	// 12 - 15
// プロジェクションのトランスフォーム
uniform shared const float4x4	Projection			: register(vs, c16);	// 16 - 19
// シャドウマップ用のトランスフォーム
uniform shared const float4x4 ShadowViewProj		: register(vs, c20);	// 20 - 23

//-----------------------------------------------------------------------------
// ボーン
//-----------------------------------------------------------------------------
// ボーンの回転部分
float4 BoneRotations[MaxBones];
// ボーンの平行移動部分
float4 BoneTranslations[MaxBones];

//-----------------------------------------------------------------------------
// テスクチャ
//-----------------------------------------------------------------------------
texture Texture;		// テクスチャ

//-----------------------------------------------------------------------------
// テスクチャサンプラ
//-----------------------------------------------------------------------------
//テクスチャのサンプラー
sampler Sampler = sampler_state
{
    Texture = <Texture>;

    MinFilter = Linear;
    MagFilter = Linear;
    MipFilter = Linear;
};

//-----------------------------------------------------------------------------
// 構造体宣言
//=============================================================================
struct ColorPair
{
	float3 Diffuse;
	float3 Specular;
};
struct CommonVSOutput
{
	float4 Position;
	float4 Diffuse;
	float4 Specular;
};
// 頂点シェーダー入力構造体

struct VSInput
{
	float4 Position : POSITION0;
    float4 BoneIndices : BLENDINDICES0;//ボーン
    float4 BoneWeights : BLENDWEIGHT0;//ボーン
};

struct VSInputVc
{
	float4 Position : POSITION0;
	float4 Color : COLOR;
    float4 BoneIndices : BLENDINDICES0;//ボーン
    float4 BoneWeights : BLENDWEIGHT0;//ボーン
};

struct VSInputNm
{
	float4 Position : POSITION0;
    float3 Normal	: NORMAL0;
    float4 BoneIndices : BLENDINDICES0;//ボーン
    float4 BoneWeights : BLENDWEIGHT0;//ボーン
};

struct VSInputNmVc
{
	float4 Position : POSITION0;
    float3 Normal	: NORMAL0;
    float4 Color : COLOR;
    float4 BoneIndices : BLENDINDICES0;//ボーン
    float4 BoneWeights : BLENDWEIGHT0;//ボーン
};

struct VSInputTx
{
    float4 Position : POSITION0;
    float2 TexCoord : TEXCOORD0;
    float4 BoneIndices : BLENDINDICES0;//ボーン
    float4 BoneWeights : BLENDWEIGHT0;//ボーン
};

struct VSInputTxVc
{
    float4 Position : POSITION0;
    float2 TexCoord : TEXCOORD0;
    float4 Color : COLOR;
    float4 BoneIndices : BLENDINDICES0;//ボーン
    float4 BoneWeights : BLENDWEIGHT0;//ボーン
};

struct VSInputNmTx
{
    float4 Position : POSITION0;
    float2 TexCoord : TEXCOORD0;
    float3 Normal	: NORMAL0;
    float4 BoneIndices : BLENDINDICES0;//ボーン
    float4 BoneWeights : BLENDWEIGHT0;//ボーン
};

struct VSInputNmTxVc
{
    float4 Position : POSITION0;
    float2 TexCoord : TEXCOORD0;
    float3 Normal	: NORMAL0;
    float4 Color : COLOR;
    float4 BoneIndices : BLENDINDICES0;//ボーン
    float4 BoneWeights : BLENDWEIGHT0;//ボーン
};

// 頂点シェーダー出力構造体
struct VSOutput
{
	float4	Position	: POSITION;
	float4	Diffuse		: COLOR0;
	float4	Specular	: COLOR1;
};
struct VSOutputTx
{
	float4	Position	: POSITION;
	float4	Diffuse		: COLOR0;
	float4	Specular	: COLOR1;
	float2	TexCoord	: TEXCOORD0;
};
// ピクセルシェーダー入力構造体
struct PSInput
{
	float4	Diffuse		: COLOR0;
	float4	Specular	: COLOR1;
};
struct PSInputTx
{
	float4	Diffuse		: COLOR0;
	float4	Specular	: COLOR1;
	float2	TexCoord	: TEXCOORD0;
};

//-----------------------------------------------------------------------------
// クォータニオンヘルパーメソッド
//=============================================================================
// 2つクォータニオンと平行移動から行列に変換する
// ひにけにさんのサンプルをMMD用に調整したもの
float4x4 CreateTransformFromQuaternionTransforms(
		float4 q1, float3 t1,
		float4 q2, float3 t2,
		float2 weights )
{
	float ww = q1.w * q1.w - 0.5f;
	float3 row10 = float3( ww         , q1.x * q1.y, q1.x * q1.z ) +
				   float3( q1.x * q1.x, q1.w * q1.z,-q1.w * q1.y );
	float3 row11 = float3( q1.x * q1.y, ww,          q1.y * q1.z ) +
	               float3(-q1.w * q1.z, q1.y * q1.y, q1.w * q1.x );
	float3 row12 = float3( q1.x * q1.z, q1.y * q1.z, ww          ) +
	               float3( q1.w * q1.y,-q1.w * q1.x, q1.z * q1.z );
	
	ww = q2.w * q2.w - 0.5f;
	float3 row20 = float3( ww,          q2.x * q2.y, q2.x * q2.z ) +
	               float3( q2.x * q2.x, q2.w * q2.z,-q2.w * q2.y );
	float3 row21 = float3( q2.x * q2.y, ww,          q2.y * q2.z ) +
	               float3(-q2.w * q2.z, q2.y * q2.y, q2.w * q2.x );
	float3 row22 = float3( q2.x * q2.z, q2.y * q2.z, ww          ) +
	               float3( q2.w * q2.y,-q2.w * q2.x, q2.z * q2.z );
	              
	float2 w2 = 2.0f * weights;
	
	return float4x4(
		row10 * w2.x + row20 * w2.y, 0,
		row11 * w2.x + row21 * w2.y, 0,
		row12 * w2.x + row22 * w2.y, 0, 
		t1 * weights.x + t2 * weights.y, 1
	);
	
}

//MMDの色計算式
//DiffuseRGB出力=マテリアルディフューズ*環境光アンビエント
//				+マテリアルエミッシブ
//DiffuseA出力=マテリアルα
//SpecularRGB出力=マテリアルスペキュラ*環境光アンビエント*pow(dot(法線,ハーフベクトル),マテリアルShininess) (但し dot(法線,ライト)>0。そうでない場合は0)
//出力色=(Diffuse*テクスチャ色+スペキュラ)
//		*or+スフィアマップ色
//		*トゥーン色
//		テクスチャ、スフィア、トゥーンは場合によって存在しない

//-----------------------------------------------------------------------------
// Compute lighting
// E: Eye-Vector
// N: Unit vector normal in world space
//-----------------------------------------------------------------------------
ColorPair ComputeLights(float3 E, float3 N)
{
	ColorPair result;
	
	result.Diffuse = AmbientLightColor;
	result.Specular = 0;
	// Directional Light 0
	float3 L = normalize(-DirLight0Direction);
	float3 H = normalize(E+L);
	float2 ret = lit(dot(N, L), dot(N, H), SpecularPower).yz;//VectorIndex.y=パレット番号
	result.Specular+=AmbientLightColor*ret.y;
	
	//MMDではEmissiveを足してからsaturateするのが正解らしい。
	result.Diffuse *= DiffuseColor;
	result.Diffuse	+= EmissiveColor;
	result.Diffuse	= saturate(result.Diffuse);
	result.Specular	*= SpecularColor;
	
	return result;
}
//-----------------------------------------------------------------------------
// 共用関数
//=============================================================================
CommonVSOutput ComputeCommonVSOutput(float4 Position, float2 BoneIndices, float2 BoneWeights)
{
	CommonVSOutput output;
	// スキン変換行列の取得
    float4x4 skinTransform =
				CreateTransformFromQuaternionTransforms( 
				BoneRotations[BoneIndices.x], BoneTranslations[BoneIndices.x],
				BoneRotations[BoneIndices.y], BoneTranslations[BoneIndices.y], BoneWeights );
	//スキン変換
	skinTransform = mul( skinTransform, World );
	
	
    // 頂点変換
    float4 position = mul(Position, skinTransform);
    output.Position = mul(mul(position, View), Projection);
    //色計算
    output.Diffuse = float4(DiffuseColor.rgb+EmissiveColor, Alpha);
    output.Specular = 0;
    
	return output;
}
CommonVSOutput ComputeCommonVSOutputWithLighting(float4 Position, float3 Normal, float2 BoneIndices, float2 BoneWeights)
{
	CommonVSOutput output;
	// スキン変換行列の取得
    float4x4 skinTransform;
    if(BoneIndices.x<0)
	{
		skinTransform=World;
	}
	else
	{
		skinTransform =
				CreateTransformFromQuaternionTransforms( 
				BoneRotations[BoneIndices.x], BoneTranslations[BoneIndices.x],
				BoneRotations[BoneIndices.y], BoneTranslations[BoneIndices.y], BoneWeights );
	//スキン変換
	skinTransform = mul( skinTransform, World );
	}
	
    // 頂点変換
    float4 position = mul(Position, skinTransform);
    output.Position = mul(mul(position, View), Projection);
    // 法線変換
    float3 normal = normalize( mul( Normal, skinTransform));
    //視線ベクトル取得
    float3 posToEye = EyePosition - position;
    float3 E = normalize(posToEye);
    //視線ベクトルと法線を元にライトを計算
    ColorPair lightResult = ComputeLights(E, normal);
    //色計算
    output.Diffuse = float4(lightResult.Diffuse.rgb, Alpha);
    output.Specular = float4(lightResult.Specular,1);
    
	return output;
}
//-----------------------------------------------------------------------------
// 頂点シェーダー
//=============================================================================
VSOutput VSBasic(VSInput input)
{
	VSOutput output;
	CommonVSOutput vout = ComputeCommonVSOutput
					(input.Position,input.BoneIndices.xy, input.BoneWeights.xy);
	
	output.Position = vout.Position;
    output.Diffuse = vout.Diffuse;
    output.Specular = vout.Specular;
    return output;
}
VSOutput VSBasicVc(VSInputVc input)
{
	VSOutput output;
	CommonVSOutput vout = ComputeCommonVSOutput
					(input.Position,input.BoneIndices.xy, input.BoneWeights.xy);
	
	output.Position = vout.Position;
    output.Diffuse = vout.Diffuse * input.Color;
    output.Specular = vout.Specular;
    return output;
}
VSOutput VSBasicNm(VSInputNm input)
{
	VSOutput output;
	CommonVSOutput vout = ComputeCommonVSOutputWithLighting
					(input.Position,input.Normal,input.BoneIndices.xy, input.BoneWeights.xy);
	
	output.Position = vout.Position;
    output.Diffuse = vout.Diffuse;
    output.Specular = vout.Specular;
    return output;
}
VSOutput VSBasicNmVc(VSInputNmVc input)
{
	VSOutput output;
	CommonVSOutput vout = ComputeCommonVSOutputWithLighting
					(input.Position,input.Normal,input.BoneIndices.xy, input.BoneWeights.xy);
	
	output.Position = vout.Position;
    output.Diffuse = vout.Diffuse * input.Color;
    output.Specular = vout.Specular;
    return output;
}
VSOutputTx VSBasicTx(VSInputTx input)
{
	VSOutputTx output;
	CommonVSOutput vout = ComputeCommonVSOutput
					(input.Position,input.BoneIndices.xy, input.BoneWeights.xy);
	
	output.Position = vout.Position;
    output.Diffuse = vout.Diffuse;
    output.Specular = vout.Specular;
	output.TexCoord = input.TexCoord;
	return output;
}

VSOutputTx VSBasicTxVc(VSInputTxVc input)
{
	VSOutputTx output;
	CommonVSOutput vout = ComputeCommonVSOutput
					(input.Position,input.BoneIndices.xy, input.BoneWeights.xy);
	
	output.Position = vout.Position;
    output.Diffuse = vout.Diffuse * input.Color;
    output.Specular = vout.Specular;
	output.TexCoord = input.TexCoord;
	return output;
}

VSOutputTx VSBasicNmTx(VSInputNmTx input)
{
	VSOutputTx output;
	CommonVSOutput vout = ComputeCommonVSOutputWithLighting
					(input.Position,input.Normal,input.BoneIndices.xy, input.BoneWeights.xy);
	
	output.Position = vout.Position;
    output.Diffuse = vout.Diffuse;
    output.Specular = vout.Specular;
	output.TexCoord = input.TexCoord;
	return output;
}

VSOutputTx VSBasicNmTxVc(VSInputNmTxVc input)
{
	VSOutputTx output;
	CommonVSOutput vout = ComputeCommonVSOutputWithLighting
					(input.Position,input.Normal,input.BoneIndices.xy, input.BoneWeights.xy);
	
	output.Position = vout.Position;
    output.Diffuse = vout.Diffuse * input.Color;
    output.Specular = vout.Specular;
	output.TexCoord = input.TexCoord;
	return output;
}

//-----------------------------------------------------------------------------
// ピクセルシェーダー
//=============================================================================
float4 PSBasic(PSInput pin) : COLOR
{
	float4 color;
	color = pin.Diffuse + float4(pin.Specular.rgb, 0);
	return color;
}
float4 PSBasicTx(PSInputTx pin) : COLOR
{
	float4 color;
	color = tex2D(Sampler, pin.TexCoord) * pin.Diffuse + float4(pin.Specular.rgb, 0);
	return color;
}
//-----------------------------------------------------------------------------
// 描画手法の宣言
//=============================================================================
int ShaderIndex = 0;

VertexShader VSArray[8] =
{
	compile vs_2_0 VSBasic(),
	compile vs_2_0 VSBasicVc(),
	compile vs_2_0 VSBasicTx(),
	compile vs_2_0 VSBasicTxVc(),
	
	compile vs_2_0 VSBasicNm(),
	compile vs_2_0 VSBasicNmVc(),
	compile vs_2_0 VSBasicNmTx(),
	compile vs_2_0 VSBasicNmTxVc(),
	
};

PixelShader PSArray[8] =
{
	compile ps_2_0 PSBasic(),
	compile ps_2_0 PSBasic(),
	compile ps_2_0 PSBasicTx(),
	compile ps_2_0 PSBasicTx(),

	compile ps_2_0 PSBasic(),
	compile ps_2_0 PSBasic(),
	compile ps_2_0 PSBasicTx(),
	compile ps_2_0 PSBasicTx(),
};

Technique MMDBasicEffect
{
	Pass
	{
		VertexShader = (VSArray[ShaderIndex]);
		PixelShader	 = (PSArray[ShaderIndex]);
	}
}

