﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using NiVE2.Plugin.Interface;
using SlimDX.Direct3D9;
using System.IO;
using System.Windows.Forms;
using MikuMikuDance.SlimDX;
using MikuMikuDance.Core.Model;
using MikuMikuDance.SlimDX.Accessory;
using System.Drawing;
using SlimDX;
using NiVE2.Utils.Cache;
using System.Reflection;
using MikuMikuDance.Core.Misc;
using MikuMikuDance.Core.Motion;
using NiVE2.Plugin.Property;
using MikuMikuDance.SlimDX.Misc;
using NiVE2.Plugin.Controls;

namespace MikuMikuDance.NiVE2
{
    /// <summary>
    /// MMDのpmd入力プラグイン
    /// </summary>
    public class PMDInput : InputChild
    {
        const string motionKey = "motion";
        string filename;
        MMDModel model;
        MotionDataSet motion = null;
        EdgeManager edgeManager = null;
        ScreenManager screen = null;//スクリーンマネージャを活用
        
        //スクリーンの大きさ
        int width;
        int height;
        //MMDは経過時間を計算するので、前回Updateの時間情報を使用する
        double lastUpdate = 0;

        /// <summary>
        /// プラグイン名
        /// </summary>
        public override string PluginName { get { return "MikuMikuDance for NiVE pmd入力プラグイン"; } }
        /// <summary>
        /// 作者名
        /// </summary>
        public override string Author { get { return "WilfremP"; } }
        /// <summary>
        /// リンク
        /// </summary>
        public override string InfoLink { get { return "http://sourceforge.jp/projects/mmdx/releases/"; } }
        /// <summary>
        /// 説明文
        /// </summary>
        public override string Description { get { return "MikuMikuDanceのモデルファイル(*.pmd)を読み込むことができるプラグインです。このプラグインはMMDXプロジェクトの一環として作りました。"; } }
        /// <summary>
        /// ファイル拡張子
        /// </summary>
        public override string CorrespondingType { get { return "MikuMikuDanceモデルファイル(*.pmd)|*.pmd"; } }
        /// <summary>
        /// ファイル名
        /// </summary>
        public override string FileName { get { return filename; } }
        /// <summary>
        /// 子入力の識別番号。一度読み込んだら変更しないこと。
        /// </summary>
        public override int ChildCode { get { return 0; } }
        /// <summary>
        /// アイテム名
        /// </summary>
        public override string MediaName
        {
            get
            {
                return Path.GetFileNameWithoutExtension(filename) + "+" + Path.GetFileNameWithoutExtension(motion.MotionFileName);
            }
        }
        /// <summary>
        /// オーディオデータを持っているか
        /// </summary>
        public override bool HaveAudio { get { return false; } }
        /// <summary>
        /// ビデオデータを持っているか
        /// </summary>
        public override bool HaveVideo { get { return true; } }
        /// <summary>
        /// 画像データかどうか
        /// </summary>
        public override bool IsImage { get { return false; } }
        /// <summary>
        /// 動画長
        /// </summary>
        public override double Length { 
            get {
                return (double)(model.AnimationPlayer[motionKey].MaxFrame / model.AnimationPlayer[motionKey].FramePerSecond);
            } 
        }
        /// <summary>
        /// 親
        /// </summary>
        public override InputBase Parent { get { return this; } }
        /// <summary>
        /// InnerLoad
        /// </summary>
        public override bool LoadSettingPrevLoad { get { return true; } }
        
        public PMDInput()
            : base()
        {
            
        }
        /// <summary>
        /// 子入力を取得する。
        /// </summary>
        /// <returns>取得した子入力。</returns>
        public override InputChild[] GetChildInput()
        {
            return new InputChild[] { this };
        }
        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            try
            {
                ObjectManager.Init();
                width = MikuMikuDanceNiVE.Default.Width;
                height = MikuMikuDanceNiVE.Default.Height;
                //スクリーンマネージャ作成しておく
                if (screen == null)
                {
                    screen = new ScreenManager(width, height);
                }
                if (edgeManager == null)
                {
                    edgeManager = new EdgeManager(width, height);
                }
            }
            catch (Exception e)
            {
                MessageBox.Show("不明なエラー: " + e.ToString());
            }
        }
        /// <summary>
        /// ファイルを読み込む。
        /// </summary>
        /// <param name="file">読み込むファイル。</param>
        /// <returns>読み込みに成功した場合はtrue、そうでない場合はfalse。</returns>
        public override bool Load(string file)
        {
            filename = file;
            model = SlimMMDXCore.Instance.LoadModelFromFile(filename);
            if (motion != null)
            {
                model.AnimationPlayer.AddMotion(motionKey, motion.Motion, MMDMotionTrackOptions.UpdateWhenStopped);
            }
            else
            {
                //ロード処理
                try
                {
                    using (FrmMotionApply form = new FrmMotionApply(filename))
                    {
                        while (form.ShowDialog() != DialogResult.OK) ;
                        motion = new MotionDataSet();
                        motion.MotionFileName = form.MotionFileName;
                        model.AnimationPlayer.AddMotion(motionKey, form.MotionData, MMDMotionTrackOptions.UpdateWhenStopped);
                        motion.Motion = form.MotionData;
                    }
                }
                catch (Exception e)
                {
                    MessageBox.Show("モーション読み込みでエラーが発生しました: " + e.ToString());
                }
            }
            return true;
        }
        /// <summary>
        /// メディア情報を取得する
        /// </summary>
        /// <returns>取得したMediaInfo。</returns>
        public override global::NiVE2.Utils.MediaInfo GetMedaiInfo()
        {
            return new global::NiVE2.Utils.MediaInfo((double)(model.AnimationPlayer[motionKey].MaxFrame / model.AnimationPlayer[motionKey].FramePerSecond),
                60.0, false, false, true, new System.Drawing.Size(width, height));
        }

        /// <summary>
        /// 入力プロパティを取得する。
        /// </summary>
        /// <returns>取得した入力プロパティ。</returns>
        public override PropertyBase[] GetDefaultProperty()
        {
            return new PropertyBase[] { new BooleanProperty("カリング", true), new NumberProperty("エッジ太さ", 1, 3, 0)};
        }

        /// <summary>
        /// 入力プロパティを編集するプロパティコントロールを取得する。
        /// </summary>
        /// <returns>取得したプロパティコントロール。</returns>
        public override PropertyEditControlBase[] GetControl()
        {
            return new PropertyEditControlBase[] { new BooleanPropertyEditControl("カリング"), new NumberPropertyEditControl("エッジ太さ",0.1,NumberPropertyEditControlType.Double) };
        }

        /// <summary>
        /// イメージに外接する四角形を取得する。
        /// </summary>
        /// <param name="time">取得する時間。</param>
        /// <param name="composition">取得元のコンポジション。</param>
        /// <param name="property">入力プロパティ。</param>
        /// <returns>取得した外接する四角形を表すRectangleF。</returns>
        public override RectangleF GetBounds(double time, IComposition composition, global::NiVE2.Utils.ReadOnlyDictionary<string, PropertyBase> property)
        {
            return new RectangleF(0.0F, 0.0F, width * (float)composition.ResolutionRate, height * (float)composition.ResolutionRate);
        }

        /// <summary>
        /// ビデオのフレーム、またはイメージを取得する。
        /// </summary>
        /// <param name="time">取得する時間。</param>
        /// <param name="composition">取得元のコンポジション。</param>
        /// <param name="property">入力プロパティ。</param>
        /// <returns>取得したイメージ。</returns>
        public override global::NiVE2.Drawing.NBitmap GetImage(double time, IComposition composition, global::NiVE2.Utils.ReadOnlyDictionary<string, PropertyBase> property)
        {
            global::NiVE2.Drawing.NBitmap result = null;
            //スクリーン設定
            SlimMMDXCore.Instance.ScreenManager = screen;
            //エッジ設定
            edgeManager.EdgeWidth = (float)((NumberProperty)property["エッジ太さ"]).DoubleValue;
            bool UseEdge = (((NumberProperty)property["エッジ太さ"]).DoubleValue != 0);
            //エッジマネージャ
            SlimMMDXCore.Instance.EdgeManager = edgeManager;
            //カメラ設定
            ObjectManager.Instance.Camera.Update(time, composition);
            //ライト設定
            ObjectManager.Instance.Light.Update(time, composition);
            //カリング設定
            model.Culling = (bool)((BooleanProperty)property["カリング"]);
            //MMDXの更新
            model.AnimationPlayer[motionKey].NowFrame = ((decimal)time) * model.AnimationPlayer[motionKey].FramePerSecond;
            SlimMMDXCore.Instance.Update(0.01666666f);
            //画質モードが変更されている場合は変形用のキャッシュも確保する
            LockTemporalCache cache = null;
            if (composition.ResolutionRate != 1.0)
            {
                cache = new LockTemporalCache(width * height * 12);
            }
            else
            {
                cache = new LockTemporalCache(width * height * 8);
            }
            //描画処理
            if (ObjectManager.Instance.Device == null)
                throw new MMDXException();
            try
            {
                if (!ObjectManager.Instance.CheckLostDevice())
                {
                    try
                    {
                        ObjectManager.Instance.Device.BeginScene();
                        if (UseEdge)
                        {
                            //エッジ検出の開始
                            edgeManager.StartEdgeDetection();
                            //モデルのエッジ検出
                            model.Draw();
                            //エッジ検出の終了
                            edgeManager.EndEdgeDetection();
                        }
                        //キャプチャの開始
                        screen.StartCapture(Color.FromArgb(0, 0, 0, 0));
                        //モデルの描画
                        model.Draw();
                        //エッジの描画
                        if (UseEdge)
                            edgeManager.DrawEdge();
                        //キャプチャの終了
                        screen.EndCapture();
                        //描画処理の終了
                        ObjectManager.Instance.Device.EndScene();
                        //バックバッファに描画(……dummyFormに)
                        ObjectManager.Instance.Device.Present(Present.None);
                        //スクリーンからビットマップ抽出して、NiVEに変換
                        using (Bitmap bitmap = new Bitmap(Texture.ToStream(screen.Screen, ImageFileFormat.Png)))
                        {
                            result = new global::NiVE2.Drawing.NBitmap(bitmap);
                        }
                    }
                    catch (Direct3D9Exception e)
                    {
                        if (e.ResultCode == ResultCode.DeviceLost)
                        {//デバイスロスト……
                            ObjectManager.Instance.OnLostDevice();
                            result = new global::NiVE2.Drawing.NBitmap(Resource.orz);
                        }
                        else
                            throw;
                    }
                }
                else
                {//デバイスロスト中…
                    result = new global::NiVE2.Drawing.NBitmap(Resource.orz);
                }
            }
            catch (Exception e)
            {
                MessageBox.Show(e.ToString());
                result = new global::NiVE2.Drawing.NBitmap(Resource.orz);
            }
            finally
            {
                cache.Unlock();
                lastUpdate = time;
            }
            return result;
        }
        /// <summary>
        /// オーディオを取得する。
        /// </summary>
        /// <param name="time">取得する時間。</param>
        /// <param name="length">取得する長さ。</param>
        /// <param name="composition">取得元のコンポジション。</param>
        /// <returns>取得したオーディオ。</returns>
        public override byte[] GetAudio(double time, int length, IComposition composition)
        {
            //InputChild.HaveAudioがfalseの場合は呼ばれない
            throw new NotImplementedException();
        }
        /// <summary>
        /// 内部データを保存する。
        /// </summary>
        /// <returns>保存するシリアル化可能なobject。</returns>
        public override object SaveInnerData()
        {
            return  motion;
        }

        /// <summary>
        /// 内部データを読み込む。
        /// </summary>
        /// <param name="data">読み込むobject。</param>
        public override void LoadInnerData(object data) 
        {
            motion = data as MotionDataSet;
            
        }

        /// <summary>
        /// 破棄処理
        /// </summary>
        public override void Dispose()
        {
                    
                
        }
    }
}
