/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: BBSConfig.cpp 1871 2006-10-22 08:36:57Z svn $
 *****************************************************************************/

#include "Environment.h"

#include <cstring>

#include "apr_file_io.h"
#include "apr_strings.h"

#include "BBSConfig.h"
#include "BBSThreadList.h"
#include "BBSThreadManager.h"
#include "BBSThreadListReader.h"
#include "PostFlowController.h"
#include "BBSTemplate.h"
#include "Macro.h"
#include "Auxiliary.h"
#include "Message.h"
#include "SourceInfo.h"

SOURCE_INFO_ADD("$Id: BBSConfig.cpp 1871 2006-10-22 08:36:57Z svn $");

#define PARAM_PREFIX "FastBBS"

const BBSConfig::item_t BBSConfig::BASE_URL                     = {
    PARAM_PREFIX "BaseUrl",
    "Base URL",
};
const BBSConfig::item_t BBSConfig::DATA_DIRECTORY               = {
    PARAM_PREFIX "DataDirectory",
    "Data Directory Path",
};
const BBSConfig::item_t BBSConfig::FILE_DIRECTORY               = {
    PARAM_PREFIX "FileDirectory",
    "File Directory Path",
};
const BBSConfig::item_t BBSConfig::TEMP_DIRECTORY               = {
    PARAM_PREFIX "TempDirectory",
    "Temporary Directory Path",
};
const BBSConfig::item_t BBSConfig::INDEX_VIEW_TEMPLATE          = {
    PARAM_PREFIX "IndexVIewTemplate",
    "Template of index view",
};
const BBSConfig::item_t BBSConfig::COMMENT_VIEW_TEMPLATE        = {
    PARAM_PREFIX "CommentVIewTemplate",
    "Template of comment view",
};
const BBSConfig::item_t BBSConfig::THREAD_VIEW_TEMPLATE         = {
    PARAM_PREFIX "ThreadVIewTemplate",
    "Template of thread view",
};
const BBSConfig::item_t BBSConfig::POST_THREAD_VIEW_TEMPLATE    = {
    PARAM_PREFIX "PostThreadVIewTemplate",
    "Template of post thread view",
};

/******************************************************************************
 * public メソッド
 *****************************************************************************/
BBSConfig::BBSConfig(apr_pool_t *pool)
  : base_url(NULL),
    data_dir_path(NULL),
    file_dir_path(NULL),
    temp_dir_path(NULL),
    is_debug_mode(false),
    pool_(pool),
    list_shm_(NULL),
    manager_shm_(NULL),
    flow_controller_shm_(NULL),
    thread_list_(NULL),
    thread_manager_(NULL)

{
    apr_pool_create(&pool_, NULL);

    apr_temp_dir_get(&temp_dir_path, pool_);
    memset(tmpls_, 0, sizeof(tmpls_));
}

void BBSConfig::init()
{
    check();
    load_template();

    list_shm_ = create_shm(pool_, BBSThreadList::get_memory_size());
    manager_shm_ = create_shm(pool_, BBSThreadManager::get_memory_size());
    flow_controller_shm_ = create_shm(pool_,
                                      PostFlowController::get_memory_size());

    thread_list_ = BBSThreadListReader::read(pool_, data_dir_path,
                                             list_shm_);
    thread_manager_ = BBSThreadManager::get_instance(manager_shm_, pool_,
                                                     data_dir_path);
    thread_manager_->init(thread_list_);

    flow_controller_ = PostFlowController::get_instance(flow_controller_shm_,
                                                        PST_INTERVAL_SEC);
}

void BBSConfig::child_init()
{
    attach_shm(pool_, list_shm_);
    attach_shm(pool_, manager_shm_);
    attach_shm(pool_, flow_controller_shm_);

    thread_list_ = BBSThreadList::child_init(list_shm_);
    thread_manager_ = BBSThreadManager::child_init(manager_shm_, thread_list_);
}

void BBSConfig::finalize()
{
    thread_manager_->sync_all_thread();

    apr_shm_destroy(flow_controller_shm_);
    apr_shm_destroy(manager_shm_);
    apr_shm_destroy(list_shm_);
}

void BBSConfig::set_template(template_type_t type, const char *tmpl_file_path)
{
    if (tmpls_[type] != NULL) {
        delete tmpls_[type];
    }
    tmpls_[type] = new BBSTemplate(pool_, tmpl_file_path);
}

const char *BBSConfig::to_string(apr_pool_t *pool)
{
    return apr_pstrcat(pool,
                       apr_psprintf(pool, "%-18s : %s\n",
                                    "base_url", base_url),
                       apr_psprintf(pool, "%-18s : %s\n",
                                    "data_dir_path", data_dir_path),
                       apr_psprintf(pool, "%-18s : %s\n",
                                    "file_dir_path", file_dir_path),
                       apr_psprintf(pool, "%-18s : %s\n",
                                    "temp_dir_path", temp_dir_path),
                       apr_psprintf(pool, "%-18s : %s\n",
                                    "temp_dir_path", temp_dir_path),

                       NULL);
}


/******************************************************************************
 * private メソッド
 *****************************************************************************/
void BBSConfig::check()
{
    if ((base_url == NULL) ||
        (data_dir_path == NULL) ||
        (file_dir_path == NULL) ||
        (temp_dir_path == NULL)) {
        THROW(MESSAGE_CONF_PARAM_COMPULSORY_NOT_SPECIFIED);
    }

    check_template();
}

void BBSConfig::check_template()
{
    for (apr_size_t i = 0; i < ARRAY_SIZE_OF(tmpls_); i++) {
        if (tmpls_[i] == NULL) {
            THROW(MESSAGE_CONF_TEMPLATE_INITIALIZE_FAILED);
        }
    }
}

void BBSConfig::load_template()
{
    for (apr_size_t i = 0; i < ARRAY_SIZE_OF(tmpls_); i++) {
        tmpls_[i]->load();
    }
}

void BBSConfig::update_template(template_type_t type)
{
    tmpls_[type]->update();
}

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
