/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: BBSCommentIterator.cpp 1887 2006-10-22 18:49:12Z svn $
 *****************************************************************************/

#include "Environment.h"

#include <algorithm>

#include "apr.h"

#include "BBSCommentIterator.h"
#include "BBSThread.h"
#include "MmapCommentIterator.h"
#include "ShmCommentIterator.h"
#include "BBSThreadReader.h"
#include "SourceInfo.h"

SOURCE_INFO_ADD("$Id: BBSCommentIterator.cpp 1887 2006-10-22 18:49:12Z svn $");

const apr_size_t BBSCommentIterator::MEMORY_SIZE        =
    max(sizeof(MmapCommentIterator), sizeof(ShmCommentIterator));
const apr_uint16_t BBSCommentIterator::RANGE_FROM_LAST  = 0xf000;

/******************************************************************************
 * public メソッド
 *****************************************************************************/
BBSCommentIterator::BBSCommentIterator(BBSThread *bbs_thread,
   	                                   range_t *ranges,
                                       apr_size_t range_count)
  : bbs_thread_(bbs_thread),
    ranges_(ranges),
    range_count_(range_count),
    curr_index_(0)
{
    validate_range();

    calc_size();

    init_curr_no_ = curr_no_;
    init_stop_no_ = stop_no_;

    bcomment_ = bbs_thread_->get_bcomment(curr_no_);
}

BBSCommentIterator::~BBSCommentIterator()
{

}

void BBSCommentIterator::get_comment(BBSThread::bbs_comment_p_t *bcomment_p) const throw()
{
    BBSThread::fill_bcomment_p(curr_no_, bcomment_, bcomment_p);
}

bool BBSCommentIterator::go_next() throw()
{
    if (LIKELY(curr_no_ != stop_no_)) {
        curr_no_++;
        bcomment_ = bbs_thread_->get_next_bcomment(bcomment_);

        return true;
    }

    if (++curr_index_ >= range_count_) {
        return false;
    }

    curr_no_ = ranges_[curr_index_].start_no;
    stop_no_ = ranges_[curr_index_].stop_no;
    bcomment_ = bbs_thread_->get_bcomment(curr_no_);

    return true;
}

void BBSCommentIterator::reset() throw()
{
    curr_index_ = 0;
    curr_no_ = init_curr_no_;
    stop_no_ = init_stop_no_;

    bcomment_ = bbs_thread_->get_bcomment(curr_no_);
}

BBSCommentIterator *BBSCommentIterator::read(apr_pool_t *pool,
                                             const char *dir_path,
                                             apr_size_t thread_id,
                                             apr_uint16_t start_no,
                                             apr_uint16_t stop_no)
{
    BBSThread *bbs_thread;
    range_t *range;

    bbs_thread = BBSThreadReader::read(pool, dir_path, thread_id);

    APR_PALLOC(range, range_t *, pool, sizeof(range_t));
    range->start_no = start_no;
    range->stop_no = stop_no;

    return new BBSCommentIterator(bbs_thread, range);
}


/******************************************************************************
 * private メソッド
 *****************************************************************************/
void BBSCommentIterator::validate_range()
{
    if (range_count_ == 0) { // 指定が無い場合は全て
        range_count_ = 1;
        curr_no_ = 1;
        stop_no_ = bbs_thread_->get_comment_count();

        return;
    } else if ((range_count_ == 1) &&
               (ranges_[0].start_no & RANGE_FROM_LAST)) { // 末尾からの個数指定
        if (ranges_[0].stop_no == 0) {
            curr_no_ = bbs_thread_->get_comment_count();
        } else if (bbs_thread_->get_comment_count() > ranges_[0].stop_no) {
            curr_no_ = bbs_thread_->get_comment_count() - ranges_[0].stop_no + 1;
        } else {
            curr_no_ = 1;
        }
        stop_no_ = bbs_thread_->get_comment_count();

        return;
    }

#ifdef DEBUG
    if (ranges_ == NULL) {
        THROW(MESSAGE_BUG_FOUND);
    }
#endif

    // 開始番号と終了番号を補正
    if (ranges_[0].start_no == 0) {
        ranges_[0].start_no = 1;
    } else if (ranges_[0].start_no > bbs_thread_->get_comment_count()) {
        ranges_[0].start_no = bbs_thread_->get_comment_count();
    }
    if ((ranges_[range_count_-1].stop_no == 0) ||
        (ranges_[range_count_-1].stop_no > bbs_thread_->get_comment_count())) {
        ranges_[range_count_-1].stop_no = bbs_thread_->get_comment_count();
    }

    // 昇順に並んでいることをチェック
    if (ranges_[range_count_-1].stop_no < ranges_[range_count_-1].start_no) {
        THROW(MESSAGE_COMMENT_RANGE_INVALID);
    }
    for (apr_size_t i = 0; i < (range_count_-1); i++) {
       	if (ranges_[i].stop_no < ranges_[i].start_no) {
            THROW(MESSAGE_COMMENT_RANGE_INVALID);
        } else if (ranges_[i+1].start_no <= ranges_[i].stop_no) {
            THROW(MESSAGE_COMMENT_RANGE_INVALID);
        }
    }

    curr_no_ = ranges_[0].start_no;
    stop_no_ = ranges_[0].stop_no;
}

void BBSCommentIterator::calc_size()
{
    size_ = stop_no_ - curr_no_ + 1;

    for (apr_size_t i = 1; i < range_count_; i++) {
        size_ += ranges_[i].stop_no - ranges_[i].start_no + 1;
    }
}

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
