/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or without_lockied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: BBSThreadManager.cpp 1959 2006-11-11 03:05:17Z svn $
 *****************************************************************************/

#include "Environment.h"

#include <cstdlib>
#include <cstring>
#include <algorithm>
#include <memory>

#include "apr_time.h"
#include "apr_file_info.h"

#include "BBSThreadManager.h"
#include "BBSThread.h"
#include "BBSThreadList.h"
#include "MmapCommentIterator.h"
#include "ShmCommentIterator.h"
#include "TemporaryPool.h"
#include "Locker.h"
#include "ReadLocker.h"
#include "WriteLocker.h"
#include "Message.h"
#include "Macro.h"
#include "SourceInfo.h"

SOURCE_INFO_ADD("$Id: BBSThreadManager.cpp 1959 2006-11-11 03:05:17Z svn $");

#ifdef DEBUG
#include <iostream>
#endif

#define AS_BOOL(pointer) reinterpret_cast<bool *>(pointer)
#define AS_BTHREAD_MANAGER(pointer) reinterpret_cast<BBSThreadManager *>(pointer)

#ifdef DEBUG
#define REQUIRE_THREAD_WRITE_LOCKED(index)                      \
    do {                                                        \
        if (!WriteLocker::is_locked(threads_ref_count_ + i)) {  \
            throw MESSAGE_BBS_THREAD_SHOULD_LOCKED              \
                 " (" __FILE__ ":" APR_STRINGIFY(__LINE__) ")"; \
        }                                                       \
    } while (0)
#else
#define REQUIRE_THREAD_WRITE_LOCKED(index)
#endif

#ifdef DEBUG
#define REQUIRE_MANAGER_LOCKED()                                \
    do {                                                        \
        if (!ReadLocker::is_locked(manager_ref_count_)) {      \
            throw MESSAGE_BBS_SHOULD_LOCKED                     \
                " (" __FILE__ ":" APR_STRINGIFY(__LINE__) ")";  \
        }                                                       \
    } while (0)
#else
#define REQUIRE_MANAGER_LOCKED()
#endif

const apr_uint16_t BBSThreadManager::NULL_INDEX = static_cast<apr_uint16_t>(-1);

/******************************************************************************
 * public メソッド
 *****************************************************************************/
void BBSThreadManager::init(BBSThreadList *thread_list)
{
    clear_shm();

    thread_list_ = thread_list;

    init_shm();
    init_threads();
}

void BBSThreadManager::get_thread_info_sort_by_view(apr_pool_t *pool,
                                                    apr_uint16_t *thread_count,
                                                    apr_size_t **ids_sort_by_view,
                                                    apr_uint16_t **indexes_sort_by_view) const
{
    apr_size_t *thread_info;
    ReadLocker manager_locker(manager_ref_count_);

    REQUIRE_MANAGER_LOCKED();

    *thread_count = min(*thread_count, get_thread_count());

    APR_PCALLOC(thread_info, apr_size_t *, pool,
                2 * sizeof(apr_size_t) * (*thread_count));

    *ids_sort_by_view = thread_info;
    *indexes_sort_by_view = AS_UINT16((*ids_sort_by_view) + (*thread_count));

    memcpy(*ids_sort_by_view, ids_sort_by_view_,
           sizeof(apr_size_t) * (*thread_count));
    memcpy(*indexes_sort_by_view, indexes_sort_by_view_,
           sizeof(apr_size_t) * (*thread_count));
}

void BBSThreadManager::get_thread_by_id(apr_pool_t *pool,
                                        apr_size_t thread_id,
                                        BBSCommentIterator *bcomment_iter,
                                        BBSCommentIterator::range_t *ranges,
                                        apr_size_t range_count) const
{
    ReadLocker manager_locker(manager_ref_count_);

    REQUIRE_MANAGER_LOCKED();

    get_thread_by_id_without_lock(pool, thread_id, bcomment_iter,
                                  ranges, range_count);
}

void BBSThreadManager::get_thread_by_index(apr_pool_t *pool,
                                           apr_uint16_t i,
                                           apr_size_t thread_id,
                                           BBSCommentIterator *bcomment_iter,
                                           BBSCommentIterator::range_t *ranges,
                                           apr_size_t range_count) const
{
    BBSThread *bbs_thread;
    ReadLocker manager_locker(manager_ref_count_);

    REQUIRE_MANAGER_LOCKED();

    bbs_thread = get_thread_by_index(i);

    // i は効率化のための参考情報．thread_id が大切．
    if (LIKELY(bbs_thread->get_id() == thread_id)) {
        new(bcomment_iter) ShmCommentIterator(bbs_thread,
                                              threads_ref_count_ + i,
                                              ranges, range_count);
    } else {
        // メモリ上から削除された場合
        // get_thread_by_id を呼ぶのは若干無駄だけど，滅多に起こらない
        // はずなので OK
        get_thread_by_id_without_lock(pool, thread_id, bcomment_iter,
                                      ranges, range_count);
    }
}

apr_size_t BBSThreadManager::create_thread(const char *subject,
                                           const char *name, const char *trip,
                                           const char *email,
                                           const char *message,
                                           BBSThread::image_t *image,
                                           apr_time_t time, bool is_age)
{
    WriteLocker manager_locker(manager_ref_count_);
    apr_size_t thread_id;
    BBSThread *bbs_thread;
    apr_uint16_t i;

    REQUIRE_MANAGER_LOCKED();

    thread_id = create_thread_id();

    if (LIKELY(get_thread_count() == MGR_ON_MEMORY_THREAD_COUNT)) {
        // メモリが満杯なので，更新日時が古いものを書き出したあと，
        // スレッドをロード
        i = indexes_sort_by_mtime_[get_thread_count() - 1];

        WriteLocker thread_locker(threads_ref_count_ + i);
        REQUIRE_THREAD_WRITE_LOCKED(i);

        bbs_thread = create_and_add_comment(thread_id, i,
                                            subject, name, trip, email,
                                            message, image,
                                            time, is_age, true);
    } else {
        // メモリに空きがあるので，スレッドをそこへロード
        i = (*thread_count_)++;

        WriteLocker thread_locker(threads_ref_count_ + i);
        REQUIRE_THREAD_WRITE_LOCKED(i);

        bbs_thread = create_and_add_comment(thread_id, i,
                                            subject, name, trip, email,
                                            message, image,
                                            time, is_age);
    }

    // スレッドの存在を示すため，ファイルに書き出しておく．
    write_thread(bbs_thread);

    thread_list_->add(bbs_thread);

    return thread_id;
}

apr_uint16_t BBSThreadManager::add_comment(apr_size_t thread_id,
                                         const char *name, const char *trip,
                                         const char *email,
                                         const char *message,
                                         BBSThread::image_t *image,
                                         apr_time_t time, bool is_age)
{
    WriteLocker manager_locker(manager_ref_count_);
    BBSThread *bbs_thread;
    apr_uint16_t i;

    REQUIRE_MANAGER_LOCKED();

    bbs_thread = get_thread_by_id(thread_id);

    if (LIKELY(bbs_thread != NULL)) {
        // メモリ上にスレッドがある
        i = get_index_by_thread(bbs_thread);

        WriteLocker thread_locker(threads_ref_count_ + i);
        REQUIRE_THREAD_WRITE_LOCKED(i);

        exec_add_comment(bbs_thread, i, name, trip, email, message, image,
                         time, is_age);
        thread_list_->update(bbs_thread);

        return bbs_thread->get_comment_count();
    } else {
        // メモリ上にスレッドが無い
        if (LIKELY(*thread_count_ != MGR_ON_MEMORY_THREAD_COUNT)) {
            // メモリに空きがあるので，スレッドをそこへロード
            i = (*thread_count_)++;
            WriteLocker thread_locker(threads_ref_count_ + i);
            REQUIRE_THREAD_WRITE_LOCKED(i);

            return read_and_add_comment(thread_id, i,
                                        name, trip, email, message, image,
                                        time, is_age);
        } else {
            // メモリが満杯なので，更新日時が古いものを書き出したあと，
            // スレッドをロード
            i = indexes_sort_by_mtime_[*thread_count_ - 1];

            WriteLocker thread_locker(threads_ref_count_ + i);
            REQUIRE_THREAD_WRITE_LOCKED(i);

            return read_and_add_comment(thread_id, i,
                                        name, trip, email, message, image,
                                        time, is_age, true);
        }
    }
}

void BBSThreadManager::sync_all_thread()
{
    for (apr_uint16_t i = 0; i < *thread_count_; i++) {
        write_thread(get_thread_by_index(i, false));
    }
}

BBSThreadManager *BBSThreadManager::get_instance(apr_shm_t *shm,
                                                 apr_pool_t *pool,
                                                 const char *data_dir_path)
{
    BBSThreadManager *thread_manager;

#ifdef DEBUG
    if (apr_shm_size_get(shm) != get_memory_size()) {
        THROW(MESSAGE_SHM_SIZE_INVALID);
    }
#endif

    thread_manager = AS_BTHREAD_MANAGER(apr_shm_baseaddr_get(shm));
    new(thread_manager) BBSThreadManager(pool, data_dir_path);

    return thread_manager;
}

BBSThreadManager *BBSThreadManager::child_init(apr_shm_t *shm,
                                               BBSThreadList *thread_list)
{
    BBSThreadManager *thread_manager;

    thread_manager = AS_BTHREAD_MANAGER(apr_shm_baseaddr_get(shm));
    thread_manager->child_init_impl(thread_list);

    return thread_manager;
}

apr_size_t BBSThreadManager::get_memory_size()
{
    return sizeof(BBSThreadManager) +
        (BBSThread::MAX_THREAD_DATA_SIZE + sizeof(apr_size_t)
         + sizeof(apr_uint16_t) + sizeof(apr_uint16_t) + sizeof(apr_atomic_t)) *
        MGR_ON_MEMORY_THREAD_COUNT +
        sizeof(apr_uint16_t) + sizeof(apr_atomic_t);
}

void BBSThreadManager::check_finalize_state(BBSThreadManager *thread_manager)
{
    if (ReadLocker::is_locked(thread_manager->manager_ref_count_)) {
        THROW(MESSAGE_BBS_SHOULD_NOT_LOCKED);
    }

    for (apr_size_t i = 0; i < *(thread_manager->thread_count_); i++) {
        if (thread_manager->threads_ref_count_[i] != 0) {
            THROW(MESSAGE_BBS_THREAD_SHOULD_NOT_REFERRED);
        }
    }
}


/******************************************************************************
 * private メソッド
 *****************************************************************************/
BBSThreadManager::BBSThreadManager(apr_pool_t *pool, const char *data_dir_path)
  : pool_(pool),
    data_dir_path_(data_dir_path),
    thread_reader_(pool_, data_dir_path),
    thread_writer_(pool_, data_dir_path)
{

}

void BBSThreadManager::child_init_impl(BBSThreadList *thread_list)
{
    thread_list_ = thread_list;

    attach_shm();
}

void BBSThreadManager::clear_shm()
{
    memset(this + 1, 0, get_memory_size() - sizeof(BBSThreadManager));
}

void BBSThreadManager::init_shm()
{
    threads_ = AS_BTHREAD(this + 1);

    ids_sort_by_view_
        = AS_SIZE(AS_CHAR(threads_) +
                  BBSThread::MAX_THREAD_DATA_SIZE*MGR_ON_MEMORY_THREAD_COUNT);

    indexes_sort_by_view_ = AS_UINT16(ids_sort_by_view_ +
                                      MGR_ON_MEMORY_THREAD_COUNT);
    indexes_sort_by_mtime_ = indexes_sort_by_view_ + MGR_ON_MEMORY_THREAD_COUNT;

    manager_ref_count_ = AS_ATOMIC(indexes_sort_by_mtime_ + MGR_ON_MEMORY_THREAD_COUNT);
    threads_ref_count_ = manager_ref_count_ + 1;
    thread_count_ = AS_UINT16(threads_ref_count_ + MGR_ON_MEMORY_THREAD_COUNT);

#ifdef __INTEL_COMPILER
#pragma warning(disable:1684)
#endif
    ASSERT_EQUAL(AS_CHAR(thread_count_) + sizeof(*thread_count_),
                 AS_CHAR(threads_) +
                 (get_memory_size() - sizeof(BBSThreadManager)));
#ifdef __INTEL_COMPILER
#pragma warning(default:1684)
#endif
}

void BBSThreadManager::attach_shm()
{
    init_shm();
}

void BBSThreadManager::init_threads()
{
    TemporaryPool temp_pool(pool_);

    for (apr_uint16_t i = 0; i < MGR_ON_MEMORY_THREAD_COUNT; i++) {
        new(get_thread_by_index(i, false)) BBSThread;
    }

#ifdef DEBUG
    if (thread_list_ == NULL) {
        THROW(MESSAGE_INITIALIZE_IMCOMPLETE);
    }
#endif
    *thread_count_ = MGR_ON_MEMORY_THREAD_COUNT;
    thread_list_->get_ids(temp_pool.get(), ids_sort_by_view_, thread_count_);

    // indexes_sort_by_{view,mtime}_ を初期化
    for (apr_size_t i = 0; i < MGR_ON_MEMORY_THREAD_COUNT; i++) {
        indexes_sort_by_view_[i] = NULL_INDEX;
        indexes_sort_by_mtime_[i] = NULL_INDEX;
    }

    for (apr_uint16_t i = 0; i < *thread_count_; i++) {
        thread_reader_.read(ids_sort_by_view_[i],
                            get_thread_by_index(i, false));

        indexes_sort_by_view_[i] = i;
        indexes_sort_by_mtime_[i] = i;
    }

    sort_indexes_by_mtime();
}

BBSThread *BBSThreadManager::get_thread_by_id(apr_size_t thread_id) const
{
    BBSThread *bbs_thread;

    REQUIRE_MANAGER_LOCKED();

    for (apr_size_t i = 0; i < get_thread_count(); i++) {
        // 表示順に調べる（この探索方法はびみょー）
        bbs_thread = get_thread_by_index(indexes_sort_by_view_[i], false);
        if (bbs_thread->get_id() == thread_id) {
            return bbs_thread;
        }
    }

    return NULL;
}

inline void BBSThreadManager::get_thread_by_id_without_lock(apr_pool_t *pool,
                                                            apr_size_t thread_id,
                                                            BBSCommentIterator *bcomment_iter,
                                                            BBSCommentIterator::range_t *ranges,
                                                            apr_size_t range_count) const
{
    BBSThread *bbs_thread;
    apr_uint16_t i;

    bbs_thread = get_thread_by_id(thread_id);

    if (LIKELY(bbs_thread != NULL)) {
        // メモリ上にスレッドがある
        i = get_index_by_thread(bbs_thread);

        new(bcomment_iter) ShmCommentIterator(bbs_thread,
                                              threads_ref_count_ + i,
                                              ranges, range_count);
    } else {
        // メモリ上にスレッドが無い
        File bthread_file(pool);

        thread_reader_.read(thread_id, &bbs_thread, pool, bthread_file);

        new(bcomment_iter) MmapCommentIterator(bbs_thread, bthread_file,
                                               ranges, range_count);
    }
}

inline BBSThread *BBSThreadManager::get_thread_by_index(apr_uint16_t i,
                                                        bool is_check_index) const
{
    if (is_check_index) {
#ifdef _MSC_VER
#pragma warning(push)
#pragma warning(disable:4267)
#endif
        ASSERT_LESS(i, get_thread_count());
#ifdef _MSC_VER
#pragma warning(pop)
#endif
    }

    return AS_BTHREAD(AS_CHAR(threads_) +
                         BBSThread::MAX_THREAD_DATA_SIZE*i);
}

inline apr_uint16_t BBSThreadManager::get_index_by_thread(BBSThread *bbs_thread) const
{
    apr_uint16_t i;
#ifdef __INTEL_COMPILER
#pragma warning(push)
#pragma warning(disable:1684)
#endif
    i = static_cast<apr_uint16_t>
        ((reinterpret_cast<apr_size_t>(bbs_thread) -
          reinterpret_cast<apr_size_t>(threads_)) >>
         BBSThread::MAX_THREAD_DATA_SHIFT);
#ifdef __INTEL_COMPILER
#pragma warning(pop)
#endif

#ifdef _MSC_VER
#pragma warning(push)
#pragma warning(disable:4267)
#endif
    ASSERT_LESS(i, get_thread_count());
#ifdef _MSC_VER
#pragma warning(pop)
#endif

    return i;
}

void BBSThreadManager::exec_add_comment(BBSThread *bbs_thread,
                                        apr_uint16_t i,
                                        const char *name, const char *trip,
                                        const char *email, const char *message,
                                        BBSThread::image_t *image,
                                        apr_time_t time, bool is_age,
                                        bool is_insert)
{
    REQUIRE_MANAGER_LOCKED();
    REQUIRE_THREAD_WRITE_LOCKED(i);

    if (!(bbs_thread->can_add_comment())) {
        THROW(MESSAGE_BBS_THREAD_COMMENT_ADD_FORBIDDEN);
    }

    bbs_thread->add_comment(name, trip, email, message, image, time, is_age);

    update_indexes_sort_by_mtime(i, is_insert);

    if (is_age) {
        update_indexes_sort_by_view(i, bbs_thread->get_id(), is_insert);
    }

#ifdef WIN32
    // 何かとトラブル(と誤解)を招きそうなので，Windows の場合は毎回書き出す
    write_thread(bbs_thread);
#else
    // 未同期のコメントの数が一定値以上になったら書き出す
    if (UNLIKELY(bbs_thread->get_unsynced_comment_count() >
                 MGR_UNSYNCED_COMMENT_LIMIT)) {
        write_thread(bbs_thread);
    }
#endif
}

apr_uint16_t BBSThreadManager::read_and_add_comment(apr_size_t thread_id,
                                                    apr_uint16_t i,
                                                    const char *name,
                                                    const char *trip,
                                                    const char *email,
                                                    const char *message,
                                                    BBSThread::image_t *image,
                                                    apr_time_t time,
                                                    bool is_age,
                                                    bool is_need_write)
{
    BBSThread *bbs_thread;

    REQUIRE_MANAGER_LOCKED();
    REQUIRE_THREAD_WRITE_LOCKED(i);

    bbs_thread = get_thread_by_index(i);

    if (is_need_write) {
        write_thread(bbs_thread);
        memset(bbs_thread, 0, BBSThread::MAX_THREAD_DATA_SIZE);
    }

    thread_reader_.read(thread_id, bbs_thread);
    exec_add_comment(bbs_thread, i, name, trip, email, message, image, time,
                     is_age);
    thread_list_->update(bbs_thread);

    return bbs_thread->get_comment_count();
}

BBSThread * BBSThreadManager::create_and_add_comment(apr_size_t thread_id,
                                                     apr_uint16_t i,
                                                     const char *subject,
                                                     const char *name,
                                                     const char *trip,
                                                     const char *email,
                                                     const char *message,
                                                     BBSThread::image_t *image,
                                                     apr_time_t time,
                                                     bool is_age,
                                                     bool is_need_write)
{
    BBSThread *bbs_thread;

    REQUIRE_MANAGER_LOCKED();
    REQUIRE_THREAD_WRITE_LOCKED(i);

    bbs_thread = get_thread_by_index(i);

    if (is_need_write) {
        write_thread(bbs_thread);
    }

    bbs_thread = BBSThread::get_instance(bbs_thread, thread_id, subject, time);

    exec_add_comment(bbs_thread, i, name, trip, email, message, image, time,
                     is_age, true);

    return bbs_thread;
}

void BBSThreadManager::write_thread(BBSThread *bbs_thread)
{
    if (bbs_thread->is_removed()) {
        return;
    }
    thread_writer_.write(bbs_thread);
}

apr_size_t BBSThreadManager::create_thread_id() const
{
    apr_size_t thread_id;

    thread_id = static_cast<apr_size_t>(apr_time_sec(apr_time_now()));

    while (thread_reader_.is_exist(thread_id)) {
        thread_id++;
    }

    return thread_id;
}

void BBSThreadManager::update_indexes_sort_by_view(apr_uint16_t i,
                                                   apr_size_t thread_id,
                                                   bool is_insert)
{
    update_array<apr_uint16_t>(indexes_sort_by_view_, i, is_insert);
    update_array<apr_size_t>(ids_sort_by_view_, thread_id, is_insert);
}

void BBSThreadManager::update_indexes_sort_by_mtime(apr_uint16_t i,
                                                    bool is_insert)
{
    update_array<apr_uint16_t>(indexes_sort_by_mtime_, i, is_insert);
}

template<class T>
void BBSThreadManager::update_array(T *array, T i, bool is_insert)
{
    apr_size_t j;

    REQUIRE_MANAGER_LOCKED();

    if (is_insert) {
        memmove(array + 1, array, sizeof(T) * (get_thread_count() - 1));
        array[0] = i;

        return;
    }

    if (array[0] == i) {
        // 既に先頭に i があれば何もする必要がない
        return;
    }

    j = NULL_INDEX;
    for (apr_size_t k = 1; k < get_thread_count(); k++) {
        if (array[k] == i) {
            j = k;
            break;
        }
    }

#ifdef DEBUG
    if (j == NULL_INDEX) {
        THROW(MESSAGE_BUG_FOUND);
    }
#endif

    memmove(array + 1, array, sizeof(T) * j);
    array[0] = i;
}

int BBSThreadManager::cmp_indexes_by_mtime(apr_uint16_t left, apr_uint16_t right) const
{
    // NULL_INDEX の場合の処理
    if ((indexes_sort_by_mtime_[left] == NULL_INDEX) &&
        (indexes_sort_by_mtime_[right] == NULL_INDEX)) {
        return 0;
    } else if (indexes_sort_by_mtime_[left] == NULL_INDEX) {
        return -1;
    } else if (indexes_sort_by_mtime_[right] == NULL_INDEX) {
        return 1;
    }

    if (get_thread_by_index(indexes_sort_by_mtime_[left])->get_mtime() ==
        get_thread_by_index(indexes_sort_by_mtime_[right])->get_mtime()) {
        return 0;
    } else if (get_thread_by_index(indexes_sort_by_mtime_[left])->get_mtime() >
               get_thread_by_index(indexes_sort_by_mtime_[right])->get_mtime()) {
        return 1;
    } else {
        return -1;
    }
}

void BBSThreadManager::sort_indexes_by_mtime(apr_uint16_t left,
                                             apr_uint16_t right)
{
    apr_uint16_t pivot;
    apr_uint16_t left_right;
    apr_uint16_t right_left;
    apr_uint16_t temp;

    // 比較関数の中で threads_ を参照したかったので，クイックソートをコー
    // ディング
    pivot = (left + right) / 2;
    left_right = left;
    right_left = right;

    while (1) {
        while (left_right < right) {
            if (cmp_indexes_by_mtime(left_right, pivot) < 0) {
                break;
            }
            left_right++;
        }
        while (left < right_left) {
            if (cmp_indexes_by_mtime(right_left, pivot) >= 0) {
                break;
            }
            right_left--;
        }

        if (left_right >= right_left) {
            break;
        }

        temp = indexes_sort_by_mtime_[left_right];
        indexes_sort_by_mtime_[left_right] = indexes_sort_by_mtime_[right_left];
        indexes_sort_by_mtime_[right_left] = temp;

        left_right++;
        right_left--;
    }

    if (left_right != left) {
        sort_indexes_by_mtime(left, left_right - 1);
    }
    if (left_right != right) {
        sort_indexes_by_mtime(left_right + 1, right);
    }
}


/******************************************************************************
 * テスト
 *****************************************************************************/
#ifdef DEBUG_BBSThreadManager
#include <string>

#include "apr_shm.h"

#include "TestRunner.h"
#include "BBSThreadListReader.h"
#include "CleanPointer.h"
#include "Auxiliary.h"

static const apr_size_t TARGET_THREAD_COUNT = 10;
static const apr_size_t RUN_ITER_COUNT      = 5000;

void show_usage(const char *prog_name)
{
    cerr << "Usage: " << prog_name << " <DAT_DIR_PATH>" << endl;
}

static void run_read_by_index(apr_pool_t *pool,
                              BBSThreadManager *thread_manager,
                              apr_size_t thread_count,
                              apr_size_t *ids_sort_by_view,
                              apr_uint16_t *indexes_sort_by_view)
{
    volatile double start_time;
    volatile double end_time;
    BBSCommentIterator *bcomment_iter;

    APR_PALLOC(bcomment_iter, BBSCommentIterator *, pool,
               BBSCommentIterator::MEMORY_SIZE);

    start_time = get_usage_sec();
    for (apr_size_t i = 0; i < RUN_ITER_COUNT; i++) {
        for (apr_size_t j = 0; j < thread_count; j++) {
            thread_manager->get_thread_by_index(pool,
                                                indexes_sort_by_view[j],
                                                ids_sort_by_view[j],
                                                bcomment_iter);
            CleanPointer<BBSCommentIterator> clean_ptr(bcomment_iter);
        }
    }
    end_time = get_usage_sec();

    show_item("by index",
              (end_time - start_time) * 1000/RUN_ITER_COUNT/thread_count,
              " msec");
}

static void run_read_by_id(apr_pool_t *pool, BBSThreadManager *thread_manager,
                           apr_size_t thread_count,
                           apr_size_t *ids_sort_by_view)
{
    volatile double start_time;
    volatile double end_time;
    BBSCommentIterator *bcomment_iter;
    BBSThread::bbs_comment_p_t bcomment_p;
    apr_size_t total_message_size;

    APR_PALLOC(bcomment_iter, BBSCommentIterator *, pool,
               BBSCommentIterator::MEMORY_SIZE);

    total_message_size = 0;

    start_time = get_usage_sec();
    for (apr_size_t i = 0; i < RUN_ITER_COUNT; i++) {
        for (apr_size_t j = 0; j < thread_count; j++) {
            thread_manager->get_thread_by_id(pool, ids_sort_by_view[j],
                                             bcomment_iter);

            bcomment_iter->get_comment(&bcomment_p);
            total_message_size += strlen(bcomment_p.message);

            CleanPointer<BBSCommentIterator> clean_ptr(bcomment_iter);
        }
    }
    end_time = get_usage_sec();

    show_item("by id",
              (end_time - start_time) * 1000/RUN_ITER_COUNT/thread_count,
              " msec");
}

static void run_all_read(apr_pool_t *pool, BBSThreadManager *thread_manager)
{
    apr_uint16_t thread_count;
    apr_size_t *ids_sort_by_view;
    apr_uint16_t *indexes_sort_by_view;

    show_test_name("read");

    thread_count = TARGET_THREAD_COUNT;
    thread_manager->get_thread_info_sort_by_view(pool, &thread_count,
                                                 &ids_sort_by_view,
                                                 &indexes_sort_by_view);

    run_read_by_index(pool, thread_manager, thread_count,
                      ids_sort_by_view, indexes_sort_by_view);
    run_read_by_id(pool, thread_manager, thread_count, ids_sort_by_view);
}

void run_all(apr_pool_t *pool, int argc, const char * const *argv)
{
    const char *data_dir_path;
    apr_shm_t *manager_shm;
    apr_shm_t *list_shm;

    if (argc != 2) {
        THROW(MESSAGE_ARGUMENT_INVALID);
    }

    data_dir_path = argv[1];

    if (!is_exist(pool, data_dir_path)) {
        THROW(MESSAGE_DAT_DIR_NOT_FOUND);
    }

    manager_shm = create_shm(pool, BBSThreadManager::get_memory_size());
    list_shm = create_shm(pool, BBSThreadList::get_memory_size());


    BBSThreadManager *thread_manager
        = BBSThreadManager::get_instance(manager_shm, pool, data_dir_path);
    BBSThreadList *thread_list = BBSThreadListReader::read(pool, data_dir_path,
                                                           list_shm);
    thread_manager->init(thread_list);

    show_item("data_dir", data_dir_path);
    show_item("manager memory",
              BBSThreadManager::get_memory_size()/1024/static_cast<double>(1024),
              " MB");
    show_item("on memory thread", thread_manager->get_thread_count(), "");
    show_line();

    run_all_read(pool, thread_manager);

    // チェック
    BBSThreadManager::check_finalize_state(thread_manager);
}

#endif

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
