/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: BBSThreadReader.cpp 1898 2006-10-26 17:46:14Z svn $
 *****************************************************************************/

#include "Environment.h"

#include <algorithm>

#include "apr_file_info.h"

#include "BBSThreadReader.h"
#include "BBSThread.h"
#include "TemporaryPool.h"
#include "TemporaryFile.h"
#include "Message.h"
#include "Macro.h"
#include "SourceInfo.h"

SOURCE_INFO_ADD("$Id: BBSThreadReader.cpp 1898 2006-10-26 17:46:14Z svn $");

/******************************************************************************
 * public メソッド
 *****************************************************************************/
BBSThreadReader::BBSThreadReader(apr_pool_t *pool, const char *dir_path)
  : BBSThreadIO(pool, dir_path)
{

}

void BBSThreadReader::read(apr_size_t thread_id, BBSThread *bbs_thread,
                           bool with_comments)
{
    const char *bthread_file_path;
    apr_mmap_t *bthread_file_map;

    TemporaryPool temp_pool(pool_);
    bthread_file_path = get_file_path(temp_pool.get(), thread_id);
    File bthread_file(temp_pool.get(), bthread_file_path);

    read_and_check(&bthread_file, &bthread_file_map);

    if (LIKELY(with_comments)) {
        memcpy(bbs_thread, bthread_file_map->mm,
               min(bthread_file_map->size, BBSThread::MAX_THREAD_DATA_SIZE));
    } else {
        memcpy(bbs_thread, bthread_file_map->mm, sizeof(BBSThread));
    }
}

void BBSThreadReader::read(apr_size_t thread_id, BBSThread **bbs_thread,
                           apr_pool_t *pool, File& bthread_file) const
{
    const char *bthread_file_path;
    apr_mmap_t *bthread_file_map;

    bthread_file_path = get_file_path(pool, thread_id);
    bthread_file = File(pool, bthread_file_path);

    read_and_check(&bthread_file, &bthread_file_map);

    *bbs_thread = AS_BTHREAD(bthread_file_map->mm);
}

bool BBSThreadReader::is_exist(apr_size_t thread_id) const
{
    const char *bthread_file_path;
    apr_finfo_t info;
    TemporaryPool temp_pool;

    bthread_file_path = get_file_path(temp_pool.get(), thread_id);

    return !APR_STATUS_IS_ENOENT(apr_stat(&info, bthread_file_path,
                                          APR_FINFO_MIN, temp_pool.get()));
}

BBSThread *BBSThreadReader::read(apr_pool_t *pool, const char *dir_path,
                                 apr_size_t thread_id)
{
    BBSThread *bbs_thread;

    BBSThreadReader reader(pool, dir_path);

    APR_PALLOC(bbs_thread, BBSThread *, pool, BBSThread::MAX_THREAD_DATA_SIZE);
    reader.read(thread_id, bbs_thread);

    return bbs_thread;
}


/******************************************************************************
 * private メソッド
 *****************************************************************************/
inline void BBSThreadReader::read_and_check(File *bthread_file,
                                            apr_mmap_t **bthread_file_map) const
{
    bthread_file->open(APR_READ|APR_BINARY);
    *bthread_file_map = bthread_file->mmap(0, 0, APR_MMAP_READ, true);

    check_format(*bthread_file_map);
}

inline void BBSThreadReader::check_format(apr_mmap_t *bthread_file_map)
{
    BBSThread *bbs_thread;

    bbs_thread = AS_BTHREAD(bthread_file_map->mm);

    if (bthread_file_map->size < sizeof(BBSThread)) {
        THROW(MESSAGE_BBS_THREAD_FORMAT_INVALID);
    } else if (bthread_file_map->size != bbs_thread->get_valid_size()) {
        cout << bthread_file_map->size << endl;
        cout << bbs_thread->get_valid_size() << endl;
        THROW(MESSAGE_BBS_THREAD_FORMAT_INVALID);
    }

    if (UNLIKELY(strncmp(bbs_thread->get_identifier(), PACKAGE_NAME,
                         TRD_MAX_IDENTIFIER_SIZE) != 0)) {
        THROW(MESSAGE_BBS_THREAD_FORMAT_INVALID);
    }
    if (UNLIKELY(strncmp(bbs_thread->get_version(), PACKAGE_VERSION,
                                 TRD_MAX_VERSION_SIZE) > 0)) {
        THROW(MESSAGE_PROGRAM_TOO_OLD);
    }

#ifdef DEBUG
    check_comment_count(bthread_file_map);
    check_offset(bthread_file_map);
#endif
}

void BBSThreadReader::check_comment_count(apr_mmap_t *bthread_file_map)
{
    BBSThread *bbs_thread;
    BBSThread::bbs_comment_t *bcomment;
    apr_size_t total_offset;

    bbs_thread = AS_BTHREAD(bthread_file_map->mm);

    total_offset = sizeof(BBSThread);
    bcomment = bbs_thread->get_bcomment(1);
    for (apr_size_t i = 0; i < bbs_thread->get_comment_count(); i++) {
        total_offset += bcomment->size;
        bcomment = bbs_thread->get_next_bcomment(bcomment);
    }

    if (total_offset != bbs_thread->get_valid_size()) {
        THROW(MESSAGE_BBS_THREAD_FORMAT_INVALID);
    }
}

void BBSThreadReader::check_offset(apr_mmap_t *bthread_file_map)
{
    BBSThread *bbs_thread;
    BBSThread::bbs_comment_t *bcomment;

    bbs_thread = AS_BTHREAD(bthread_file_map->mm);

    for (apr_size_t no = 1; no <= bbs_thread->get_comment_count(); no++) {
        bcomment = bbs_thread->get_bcomment(1);
        for (apr_size_t i = 1; i < no; i++) {
            bcomment = bbs_thread->get_next_bcomment(bcomment);
        }

        if (bcomment != bbs_thread->get_bcomment(no)) {
            THROW(MESSAGE_BBS_THREAD_FORMAT_INVALID);
        }
    }
}

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
