/*
 * GetRawTransactionTestKuma.java
 *
 * Copyright 2014 the Monache.org.
 * http://www.monache.org/
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package org.monache.lib.api;

import java.math.BigDecimal;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.monache.lib.TestResources;
import org.monache.lib.api.impl.DefaultCoinAPIFactory;
import org.monache.lib.api.result.GetRawTransactionResult;
import org.monache.lib.api.result.Vin;
import org.monache.lib.api.result.Vout;
import org.monache.lib.jsonrpc.JsonRPCClient;
import org.seasar.framework.container.S2Container;
import org.seasar.framework.container.factory.S2ContainerFactory;

import static org.junit.Assert.*;


/**
 * GetRawTransactionのテスト(kumacoin)。
 * 
 * @version 
 * @author monache.org
 * @since 
 */
public class GetRawTransactionTestKuma {

	static S2Container container = null;
	static DefaultCoinAPIFactory factory = null;
	static JsonRPCClient jsonRPC = null;
	
	/**
	 * 
	 * @throws java.lang.Exception
	 * @since 
	 */
	@BeforeClass
	public static void setUpBeforeClass() throws Exception {
		container = S2ContainerFactory.create("test-kumacoin.dicon");
		jsonRPC = (JsonRPCClient)container.getComponent(JsonRPCClient.class);
		factory = DefaultCoinAPIFactory.newInstance();
	}

	/**
	 * @throws java.lang.Exception
	 * @since 
	 */
	@AfterClass
	public static void tearDownAfterClass() throws Exception {
		jsonRPC.close();
	}

	/**
	 * @throws java.lang.Exception
	 * @since 
	 */
	@Before
	public void setUp() throws Exception {
	}

	/**
	 * @throws java.lang.Exception
	 * @since 
	 */
	@After
	public void tearDown() throws Exception {
	}

	/**
	 * Test
	 * GetRawTransactionResult.call(JsonRPCClient jsonRPCClient, String transactionId)
	 * 
	 * 【テスト仕様】
	 * 受信の場合、コンソールでの実行結果とJSON-RPCでの実行結果を比較する。
	 * getrawtransaction ce2e2691ff718cf0553759195096958760f11e778f6b90194575b904a23f298d
	 * 
	 * 【確認事項】
	 * 結果が一致すること。
	 * 
	 * @throws Exception
	 * @since 1.0
	 */
	@Test
	public void test1() throws Exception {
		GetRawTransaction obj = (GetRawTransaction)container.getComponent(GetRawTransaction.class);
		TestResources resources = (TestResources)container.getComponent(TestResources.class);
		if (resources.isTestGetRawTransaction_test1() == false) {
			System.out.println("test skip GetTransactionTestKuma.isTestGetRawTransaction_test1");
			return;
		}

		System.out.println("***** GetRawTransactionTestKuma.test1 *********************************");
		GetRawTransactionResult result = obj.call(jsonRPC, "ce2e2691ff718cf0553759195096958760f11e778f6b90194575b904a23f298d");
		assertEquals("01000000d0487353034f077f84b65bd19d31117ac5188cd55f7df7632481a55eb97a17d1bac739a293010000006b483045022100ae2f0cba383d6fe092419cbcbc01c7e4e2c4ede9119d77657be1347cdbe5a018022010b3ec64d344f0a23326c05135f00a75208ae7326a849efde7c5c5a6343c7f710121036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514ffffffffebe0fac93e201b5b0b8f2afd5cef68aec520bd960a23c037cd0b30bd8b84eccf010000006c493046022100871098791b3fe08fc9f00c8eae99f6c2321cb08c1465ef4e1cf20fa5df045f67022100d32fe5f73fe926fa984135d1788d194b6668acc149d831472d972e533e4a41ce0121036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514fffffffff3ac3688f7f4447cc1cb87af9d2d7d4b903ebd6a2e07a0bf5db6f4c450dd3902010000006b483045022100ad5f296c4f5497336140c66243da445748e2299a4a0bf19f4a83ccc1d13ba14e02201565cc0a21211a57c85440fcd04b0aad12aab713acb770deb369ed3645c155880121036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514ffffffff02f0a29a3b000000001976a914bc56f4975e095c6d8c5b649d47c8aadc068bb55688ac0042dc06030000001976a914b516e35b64aee9805dcd3ff3a930ed94d3a633a788ac00000000", result.getHex());
		
	}
	
	/**
	 * Test
	 * GetRawTransactionResult.call(JsonRPCClient jsonRPCClient, String transactionId)
	 * 
	 * 【テスト仕様】
	 * 受信の場合、コンソールでの実行結果とJSON-RPCでの実行結果を比較する。
	 * getrawtransaction ce2e2691ff718cf0553759195096958760f11e778f6b90194575b904a23f298d 1
	 * 
	 * 【確認事項】
	 * 結果が一致すること。
	 * 
	 * @throws Exception
	 * @since 1.0
	 */
	@Test
	public void test2() throws Exception {
		GetRawTransaction obj = (GetRawTransaction)container.getComponent(GetRawTransaction.class);
		TestResources resources = (TestResources)container.getComponent(TestResources.class);
		if (resources.isTestGetRawTransaction_test2() == false) {
			System.out.println("test skip GetTransactionTestKuma.isTestGetRawTransaction_test2");
			return;
		}

		System.out.println("***** GetRawTransactionTestKuma.test2 *********************************");
		GetRawTransactionResult result = obj.call(jsonRPC, "ce2e2691ff718cf0553759195096958760f11e778f6b90194575b904a23f298d", 1L);
		out(result);

		assertEquals("01000000d0487353034f077f84b65bd19d31117ac5188cd55f7df7632481a55eb97a17d1bac739a293010000006b483045022100ae2f0cba383d6fe092419cbcbc01c7e4e2c4ede9119d77657be1347cdbe5a018022010b3ec64d344f0a23326c05135f00a75208ae7326a849efde7c5c5a6343c7f710121036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514ffffffffebe0fac93e201b5b0b8f2afd5cef68aec520bd960a23c037cd0b30bd8b84eccf010000006c493046022100871098791b3fe08fc9f00c8eae99f6c2321cb08c1465ef4e1cf20fa5df045f67022100d32fe5f73fe926fa984135d1788d194b6668acc149d831472d972e533e4a41ce0121036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514fffffffff3ac3688f7f4447cc1cb87af9d2d7d4b903ebd6a2e07a0bf5db6f4c450dd3902010000006b483045022100ad5f296c4f5497336140c66243da445748e2299a4a0bf19f4a83ccc1d13ba14e02201565cc0a21211a57c85440fcd04b0aad12aab713acb770deb369ed3645c155880121036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514ffffffff02f0a29a3b000000001976a914bc56f4975e095c6d8c5b649d47c8aadc068bb55688ac0042dc06030000001976a914b516e35b64aee9805dcd3ff3a930ed94d3a633a788ac00000000", result.getHex());
		assertEquals("ce2e2691ff718cf0553759195096958760f11e778f6b90194575b904a23f298d", result.getTxid());
		assertEquals(new Long(1), result.getVersion());
		assertEquals(new Long(1400064316), result.getTime());
		assertEquals(new Long(0), result.getLocktime());
		
		Vin[] vin = result.getVin();
		assertEquals("93a239c7bad1177ab95ea5812463f77d5fd58c18c57a11319dd15bb6847f074f", vin[0].getTxid());
		assertEquals(new Long(1), vin[0].getVout());
		assertEquals("3045022100ae2f0cba383d6fe092419cbcbc01c7e4e2c4ede9119d77657be1347cdbe5a018022010b3ec64d344f0a23326c05135f00a75208ae7326a849efde7c5c5a6343c7f7101 036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514", vin[0].getScriptSig().getAsm());
		assertEquals("483045022100ae2f0cba383d6fe092419cbcbc01c7e4e2c4ede9119d77657be1347cdbe5a018022010b3ec64d344f0a23326c05135f00a75208ae7326a849efde7c5c5a6343c7f710121036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514", vin[0].getScriptSig().getHex());
		assertEquals(new Long("4294967295"), vin[0].getSequence());

		assertEquals("cfec848bbd300bcd37c0230a96bd20c5ae68ef5cfd2a8f0b5b1b203ec9fae0eb", vin[1].getTxid());
		assertEquals(new Long(1), vin[1].getVout());
		assertEquals("3046022100871098791b3fe08fc9f00c8eae99f6c2321cb08c1465ef4e1cf20fa5df045f67022100d32fe5f73fe926fa984135d1788d194b6668acc149d831472d972e533e4a41ce01 036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514", vin[1].getScriptSig().getAsm());
		assertEquals("493046022100871098791b3fe08fc9f00c8eae99f6c2321cb08c1465ef4e1cf20fa5df045f67022100d32fe5f73fe926fa984135d1788d194b6668acc149d831472d972e533e4a41ce0121036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514", vin[1].getScriptSig().getHex());
		assertEquals(new Long("4294967295"), vin[1].getSequence());

		assertEquals("0239dd50c4f4b65dbfa0072e6abd3e904b7d2d9daf87cbc17c44f4f78836acf3", vin[2].getTxid());
		assertEquals(new Long(1), vin[2].getVout());
		assertEquals("3045022100ad5f296c4f5497336140c66243da445748e2299a4a0bf19f4a83ccc1d13ba14e02201565cc0a21211a57c85440fcd04b0aad12aab713acb770deb369ed3645c1558801 036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514", vin[2].getScriptSig().getAsm());
		assertEquals("483045022100ad5f296c4f5497336140c66243da445748e2299a4a0bf19f4a83ccc1d13ba14e02201565cc0a21211a57c85440fcd04b0aad12aab713acb770deb369ed3645c155880121036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514", vin[2].getScriptSig().getHex());
		assertEquals(new Long("4294967295"), vin[2].getSequence());

		Vout[] vout = result.getVout();
		assertEquals(new BigDecimal("999.99000000"), vout[0].getValue());
		assertEquals(new Long("0"), vout[0].getN());
		assertEquals("OP_DUP OP_HASH160 bc56f4975e095c6d8c5b649d47c8aadc068bb556 OP_EQUALVERIFY OP_CHECKSIG", vout[0].getScriptPubKey().getAsm());
		assertEquals("76a914bc56f4975e095c6d8c5b649d47c8aadc068bb55688ac", vout[0].getScriptPubKey().getHex());
		assertEquals(new Long("1"), vout[0].getScriptPubKey().getReqSigs());
		assertEquals("pubkeyhash", vout[0].getScriptPubKey().getType());
		assertEquals("KQNzWimL7CXoPGA1xRsjMpT8Tmaw637dZY", vout[0].getScriptPubKey().getAddresses()[0]);

		assertEquals(new BigDecimal("13000.00000000"), vout[1].getValue());
		assertEquals(new Long("1"), vout[1].getN());
		assertEquals("OP_DUP OP_HASH160 b516e35b64aee9805dcd3ff3a930ed94d3a633a7 OP_EQUALVERIFY OP_CHECKSIG", vout[1].getScriptPubKey().getAsm());
		assertEquals("76a914b516e35b64aee9805dcd3ff3a930ed94d3a633a788ac", vout[1].getScriptPubKey().getHex());
		assertEquals(new Long("1"), vout[1].getScriptPubKey().getReqSigs());
		assertEquals("pubkeyhash", vout[1].getScriptPubKey().getType());
		assertEquals("KPif2uJRjzUJVpNjHjUmDB2nrX474Njrj3", vout[1].getScriptPubKey().getAddresses()[0]);

		assertNotNull(result.getConfirmations());
		assertEquals("000000001789312697a57d98a6ee0023311a48ac612c63032d84647e40b949db", result.getBlockhash());
		assertEquals(new Long(1400064316), result.getBlocktime());
		
	}

	/**
	 * Test
	 * GetTransactionResult.call(JsonRPCClient jsonRPCClient, String transactionId)
	 * 
	 * 【テスト仕様】
	 * 送信の場合、コンソールでの実行結果とJSON-RPCでの実行結果を比較する。
	 * gettransaction b5ded1d59d1f0e5b294ab73ad2aa05cde7a09fec2d0001f3818a5598f0193acd
	 * 
	 * 【確認事項】
	 * 結果が一致すること。
	 * 
	 * @throws Exception
	 * @since 1.0
	 */
	@Test
	public void test3() throws Exception {
		GetRawTransaction obj = (GetRawTransaction)container.getComponent(GetRawTransaction.class);
		TestResources resources = (TestResources)container.getComponent(TestResources.class);
		if (resources.isTestGetRawTransaction_test3() == false) {
			System.out.println("test skip GetTransactionTestKuma.isTestGetRawTransaction_test3");
			return;
		}

		System.out.println("***** GetRawTransactionTestKuma.test3 *********************************");
		GetRawTransactionResult result = obj.call(jsonRPC, "7e38c9023af7b17a5442dc818034f1e8616912b2b4a8c5e01c207dced8511fbf", 1L);
		out(result);

		assertEquals("01000000bab2815302ed47475321bf54bfda086a5477d04212cf9de1794923fbd910636d9806b227ce000000006b483045022100aae0aedf6fefc5cd5f3498459d4e349276f502d4fe0f3564708cfaa15db157bf02200a5d3ee9984289f898295ab26bbcf73ae31b5bf60e2b6e42d1145d33f940ac4c012103f95a9c865e0cba328c1ae61d4de1d30ae50aea131386679ab683aa6e422df7e7ffffffffed47475321bf54bfda086a5477d04212cf9de1794923fbd910636d9806b227ce020000006b48304502201373bda11bde4fcd99d7aa416cbea922437fc6e304457492ad5d8eb4d6819b91022100d12d814bd1fccb951799a8dcbc8edf0f9dfd6adc81c907517a52205b41ec58aa01210362fd4e511b8a9e02b0370771a78235ee9de1aab8de1281b72954d603f819ce42ffffffff0220f40e00000000001976a914411f1cf43d4897f1506a0d382bb7ab72aed717c488ac40420f00000000001976a914a3df06c2dafc77471b9b0a6295ef68833a834b7888ac00000000", result.getHex());
		assertEquals("7e38c9023af7b17a5442dc818034f1e8616912b2b4a8c5e01c207dced8511fbf", result.getTxid());
		assertEquals(new Long(1), result.getVersion());
		assertEquals(new Long(1401008830), result.getTime());
		assertEquals(new Long(0), result.getLocktime());
		
		Vin[] vin = result.getVin();
		assertEquals("ce27b206986d6310d9fb234979e19dcf1242d077546a08dabf54bf21534747ed", vin[0].getTxid());
		assertEquals(new Long(0), vin[0].getVout());
		assertEquals("3045022100aae0aedf6fefc5cd5f3498459d4e349276f502d4fe0f3564708cfaa15db157bf02200a5d3ee9984289f898295ab26bbcf73ae31b5bf60e2b6e42d1145d33f940ac4c01 03f95a9c865e0cba328c1ae61d4de1d30ae50aea131386679ab683aa6e422df7e7", vin[0].getScriptSig().getAsm());
		assertEquals("483045022100aae0aedf6fefc5cd5f3498459d4e349276f502d4fe0f3564708cfaa15db157bf02200a5d3ee9984289f898295ab26bbcf73ae31b5bf60e2b6e42d1145d33f940ac4c012103f95a9c865e0cba328c1ae61d4de1d30ae50aea131386679ab683aa6e422df7e7", vin[0].getScriptSig().getHex());
		assertEquals(new Long("4294967295"), vin[0].getSequence());

		assertEquals("ce27b206986d6310d9fb234979e19dcf1242d077546a08dabf54bf21534747ed", vin[1].getTxid());
		assertEquals(new Long(2), vin[1].getVout());
		assertEquals("304502201373bda11bde4fcd99d7aa416cbea922437fc6e304457492ad5d8eb4d6819b91022100d12d814bd1fccb951799a8dcbc8edf0f9dfd6adc81c907517a52205b41ec58aa01 0362fd4e511b8a9e02b0370771a78235ee9de1aab8de1281b72954d603f819ce42", vin[1].getScriptSig().getAsm());
		assertEquals("48304502201373bda11bde4fcd99d7aa416cbea922437fc6e304457492ad5d8eb4d6819b91022100d12d814bd1fccb951799a8dcbc8edf0f9dfd6adc81c907517a52205b41ec58aa01210362fd4e511b8a9e02b0370771a78235ee9de1aab8de1281b72954d603f819ce42", vin[1].getScriptSig().getHex());
		assertEquals(new Long("4294967295"), vin[1].getSequence());

		Vout[] vout = result.getVout();
		assertEquals(new BigDecimal("0.98000000"), vout[0].getValue());
		assertEquals(new Long("0"), vout[0].getN());
		assertEquals("OP_DUP OP_HASH160 411f1cf43d4897f1506a0d382bb7ab72aed717c4 OP_EQUALVERIFY OP_CHECKSIG", vout[0].getScriptPubKey().getAsm());
		assertEquals("76a914411f1cf43d4897f1506a0d382bb7ab72aed717c488ac", vout[0].getScriptPubKey().getHex());
		assertEquals(new Long("1"), vout[0].getScriptPubKey().getReqSigs());
		assertEquals("pubkeyhash", vout[0].getScriptPubKey().getType());
		assertEquals("KD9UW8dnCLxKxKJeFPeEoQmiyf4n3TG68n", vout[0].getScriptPubKey().getAddresses()[0]);

		assertEquals(new BigDecimal("1.00000000"), vout[1].getValue());
		assertEquals(new Long("1"), vout[1].getN());
		assertEquals("OP_DUP OP_HASH160 a3df06c2dafc77471b9b0a6295ef68833a834b78 OP_EQUALVERIFY OP_CHECKSIG", vout[1].getScriptPubKey().getAsm());
		assertEquals("76a914a3df06c2dafc77471b9b0a6295ef68833a834b7888ac", vout[1].getScriptPubKey().getHex());
		assertEquals(new Long("1"), vout[1].getScriptPubKey().getReqSigs());
		assertEquals("pubkeyhash", vout[1].getScriptPubKey().getType());
		assertEquals("KN9cdJVAvivaE82XjLL3k3hDuQbiqriiCB", vout[1].getScriptPubKey().getAddresses()[0]);

		assertNotNull(result.getConfirmations());
		assertEquals("000000003447f7b82ae79f9c9d40980c341bac743618765be4d96bc175217a67", result.getBlockhash());
		assertEquals(new Long(1401008830), result.getBlocktime());
		
		
	}


	
	/**
	 * Test(Factory)
	 * GetTransactionResult.call(JsonRPCClient jsonRPCClient, String transactionId)
	 * 
	 * 【テスト仕様】
	 * 受信の場合、コンソールでの実行結果とJSON-RPCでの実行結果を比較する。
	 * gettransaction b5ded1d59d1f0e5b294ab73ad2aa05cde7a09fec2d0001f3818a5598f0193acd
	 * テストにはFactoryを利用する。
	 * 
	 * 【確認事項】
	 * 結果が一致すること。
	 * 
	 * @throws Exception
	 * @since 1.0
	 */
	@Test
	public void testFactory() throws Exception {
		// Factory
		GetRawTransaction obj = factory.getGetRawTransaction();
		TestResources resources = (TestResources)container.getComponent(TestResources.class);
		if (resources.isTestGetRawTransaction_testFactory() == false) {
			System.out.println("test skip GetTransactionTestKuma.isTestGetRawTransaction_testFactory");
			return;
		}

		System.out.println("***** GetRawTransactionTestKuma.testFactory *********************************");
		GetRawTransactionResult result = obj.call(jsonRPC, "ce2e2691ff718cf0553759195096958760f11e778f6b90194575b904a23f298d");
		assertEquals("01000000d0487353034f077f84b65bd19d31117ac5188cd55f7df7632481a55eb97a17d1bac739a293010000006b483045022100ae2f0cba383d6fe092419cbcbc01c7e4e2c4ede9119d77657be1347cdbe5a018022010b3ec64d344f0a23326c05135f00a75208ae7326a849efde7c5c5a6343c7f710121036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514ffffffffebe0fac93e201b5b0b8f2afd5cef68aec520bd960a23c037cd0b30bd8b84eccf010000006c493046022100871098791b3fe08fc9f00c8eae99f6c2321cb08c1465ef4e1cf20fa5df045f67022100d32fe5f73fe926fa984135d1788d194b6668acc149d831472d972e533e4a41ce0121036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514fffffffff3ac3688f7f4447cc1cb87af9d2d7d4b903ebd6a2e07a0bf5db6f4c450dd3902010000006b483045022100ad5f296c4f5497336140c66243da445748e2299a4a0bf19f4a83ccc1d13ba14e02201565cc0a21211a57c85440fcd04b0aad12aab713acb770deb369ed3645c155880121036c172e7c01419a063763653f1105284c350876fb846a3eea5d5137171883a514ffffffff02f0a29a3b000000001976a914bc56f4975e095c6d8c5b649d47c8aadc068bb55688ac0042dc06030000001976a914b516e35b64aee9805dcd3ff3a930ed94d3a633a788ac00000000", result.getHex());

	}

	private void out(GetRawTransactionResult result) throws Exception {

		System.out.println("hex = " + result.getHex());
		System.out.println("json = " + result.getJsonString());
		System.out.println("txid = " + result.getTxid());
		System.out.println("version = " + result.getVersion());
		System.out.println("time = " + result.getTime());
		System.out.println("locktime = " + result.getLocktime());
		
		int i = 0;
		for (Vin vin : result.getVin()) {
			System.out.println("** vin " + i + " **");			
			System.out.println("txid = " + vin.getTxid());
			System.out.println("vout = " + vin.getVout());
			System.out.println("asm = " + vin.getScriptSig().getAsm());
			System.out.println("hex = " + vin.getScriptSig().getHex());
			System.out.println("sequence = " + vin.getSequence());
		}

		i = 0;
		for (Vout vout : result.getVout()) {
			System.out.println("** vout " + i + " **");			
			System.out.println("value = " + vout.getValue());
			System.out.println("n = " + vout.getN());
			System.out.println("asm = " + vout.getScriptPubKey().getAsm());
			System.out.println("hex = " + vout.getScriptPubKey().getHex());
			System.out.println("reqSigs = " + vout.getScriptPubKey().getReqSigs());
			System.out.println("type = " + vout.getScriptPubKey().getType());
			for (String address : vout.getScriptPubKey().getAddresses()) {
				System.out.println("address = " + address);
			}
		}
		
		System.out.println("confirmations = " + result.getConfirmations());
		System.out.println("blockhash = " + result.getBlockhash());
		System.out.println("blocktime = " + result.getBlocktime());
		
	}
}
