/*
 * SendFromTestKuma.java
 *
 * Copyright 2014 the Monache.org.
 * http://www.monache.org/
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package org.monache.lib.api;

import static org.junit.Assert.*;

import java.math.BigDecimal;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.monache.lib.TestResources;
import org.monache.lib.api.WalletLock;
import org.monache.lib.api.WalletPassphrase;
import org.monache.lib.api.impl.DefaultCoinAPIFactory;
import org.monache.lib.api.result.Details;
import org.monache.lib.api.result.GetTransactionResult;
import org.monache.lib.api.result.SendFromResult;
import org.monache.lib.jsonrpc.JsonRPCClient;
import org.seasar.framework.container.S2Container;
import org.seasar.framework.container.factory.S2ContainerFactory;

/**
 * 
 * 
 * @version 
 * @author monache.org
 * @since 
 */
public class SendFromTestKuma {

	static S2Container container = null;
	static DefaultCoinAPIFactory factory = null;
	static JsonRPCClient jsonRPC = null;
	static TestResources resources = null;

	/**
	 * 
	 * @throws java.lang.Exception
	 * @since 
	 */
	@BeforeClass
	public static void setUpBeforeClass() throws Exception {
		container = S2ContainerFactory.create("test-kumacoin.dicon");
		jsonRPC = (JsonRPCClient)container.getComponent(JsonRPCClient.class);
		factory = DefaultCoinAPIFactory.newInstance();
		resources = (TestResources)container.getComponent(TestResources.class);
	}

	/**
	 * @throws java.lang.Exception
	 * @since 
	 */
	@AfterClass
	public static void tearDownAfterClass() throws Exception {
		jsonRPC.close();
	}

	/**
	 * @throws java.lang.Exception
	 * @since 
	 */
	@Before
	public void setUp() throws Exception {
		if (resources.isWalletEncry()) {
			WalletPassphrase unlock = (WalletPassphrase)container.getComponent(WalletPassphrase.class);
			unlock.call(jsonRPC, resources.getPassphrase(), 100L);
		}

	}

	/**
	 * @throws java.lang.Exception
	 * @since 
	 */
	@After
	public void tearDown() throws Exception {
		if (resources.isWalletEncry()) {
			WalletLock lock = (WalletLock)container.getComponent(WalletLock.class);
			lock.call(jsonRPC);
		}
	}

	
	/**
	 * Test
	 * SendFrom.call(JsonRPCClient jsonRPCClient, String acount, String toAddress, BigDecimal amount )
	 * 
	 * 【テスト仕様】
	 * 送金し、送金処理がされていることを確認する。
	 * 
	 * 【確認事項】
	 * ・送金数が一致していること。
	 * ・金額が一致していること。
	 * 
	 * @throws Exception
	 * @since 1.0
	 */
	@Test
	public void test1() throws Exception {

		if (resources.isTestSendFrom_test1() == false) {
			System.out.println("test skip SendManyTestKuma.isTestSendFrom_test1");
			return;
		}

		SendFrom obj = (SendFrom)container.getComponent(SendFrom.class);
		SendFromResult result = obj.call(jsonRPC, resources.getTestSendMany_fromAccount(), resources.getTestSendFrom_address1(), new BigDecimal("0.01"));

		assertNotNull(result.getTransactionId());

		GetTransaction obj2 = (GetTransaction)container.getComponent(GetTransaction.class);
		GetTransactionResult result2 = obj2.call(jsonRPC, result.getTransactionId());

		int sendCount = 0;
		for (int i = 0; i < result2.getDetails().length; i++) {
			Details details = result2.getDetails()[i];
			if ("send".equals(details.getCategory())) {
				if (resources.getTestSendMany_address1().equals(details.getAddress())) {
					assertEquals(new BigDecimal("-0.01000000"),details.getAmount());
				} 
				sendCount++;
			}
		}
		assertEquals(1, sendCount);
		assertEquals(null,result2.getComment());
	}

	
	/**
	 * Test
	 * SendFrom.call(JsonRPCClient jsonRPCClient, String acount, String toAddress, BigDecimal amount, Long minconf)
	 * 
	 * 【テスト仕様】
	 * 送金し、送金処理がされていることを確認する。
	 * 
	 * 【確認事項】
	 * ・送金数が一致していること。
	 * ・金額が一致していること。
	 * 
	 * @throws Exception
	 * @since 1.0
	 */
	@Test
	public void test2() throws Exception {
		if (resources.isTestSendFrom_test2() == false) {
			System.out.println("test skip SendManyTestKuma.isTestSendFrom_test2");
			return;
		}

		SendFrom obj = (SendFrom)container.getComponent(SendFrom.class);
		SendFromResult result = obj.call(jsonRPC, resources.getTestSendMany_fromAccount(), resources.getTestSendFrom_address1(), new BigDecimal("0.01"), 2L);

		assertNotNull(result.getTransactionId());

		GetTransaction obj2 = (GetTransaction)container.getComponent(GetTransaction.class);
		GetTransactionResult result2 = obj2.call(jsonRPC, result.getTransactionId());

		int sendCount = 0;
		for (int i = 0; i < result2.getDetails().length; i++) {
			Details details = result2.getDetails()[i];
			if ("send".equals(details.getCategory())) {
				if (resources.getTestSendMany_address1().equals(details.getAddress())) {
					assertEquals(new BigDecimal("-0.01000000"),details.getAmount());
				} 
				sendCount++;
			}
		}
		assertEquals(1, sendCount);
		assertEquals(null,result2.getComment());
	}

	/**
	 * Test
	 * SendFrom.call(JsonRPCClient jsonRPCClient, String acount, String toAddress, BigDecimal amount, String comment, String commentTo)
	 * 
	 * 【テスト仕様】
	 * 送金し、送金処理がされていることを確認する。
	 * 
	 * 【確認事項】
	 * ・送金数が一致していること。
	 * ・金額が一致していること。
	 * ・コメントが一致していること。
	 * 
	 * @throws Exception
	 * @since 1.0
	 */
	@Test
	public void test3() throws Exception {
		if (resources.isTestSendFrom_test3() == false) {
			System.out.println("test skip SendManyTestKuma.isTestSendFrom_test3");
			return;
		}

		SendFrom obj = (SendFrom)container.getComponent(SendFrom.class);
		SendFromResult result = obj.call(jsonRPC, resources.getTestSendMany_fromAccount(), resources.getTestSendFrom_address1(), new BigDecimal("0.01"), "com1", "com2");

		assertNotNull(result.getTransactionId());

		GetTransaction obj2 = (GetTransaction)container.getComponent(GetTransaction.class);
		GetTransactionResult result2 = obj2.call(jsonRPC, result.getTransactionId());

		int sendCount = 0;
		for (int i = 0; i < result2.getDetails().length; i++) {
			Details details = result2.getDetails()[i];
			if ("send".equals(details.getCategory())) {
				if (resources.getTestSendMany_address1().equals(details.getAddress())) {
					assertEquals(new BigDecimal("-0.01000000"),details.getAmount());
				} 
				sendCount++;
			}
		}
		assertEquals(1, sendCount);
		assertEquals("com1",result2.getComment());
		assertEquals("com2",result2.getTo());
	}

	/**
	 * Test
	 * SendFrom.call(JsonRPCClient jsonRPCClient, String acount, String toAddress, BigDecimal amount, Long minconf, String comment, String commentTo)
	 * 
	 * 【テスト仕様】
	 * 送金し、送金処理がされていることを確認する。
	 * 
	 * 【確認事項】
	 * ・送金数が一致していること。
	 * ・金額が一致していること。
	 * ・コメントが一致していること。
	 * 
	 * @throws Exception
	 * @since 1.0
	 */
	@Test
	public void test4() throws Exception {
		if (resources.isTestSendFrom_test4() == false) {
			System.out.println("test skip SendManyTestKuma.isTestSendFrom_test4");
			return;
		}

		SendFrom obj = (SendFrom)container.getComponent(SendFrom.class);
		SendFromResult result = obj.call(jsonRPC, resources.getTestSendMany_fromAccount(), resources.getTestSendFrom_address1(), new BigDecimal("0.01"), 2L, "com1", "com2");

		assertNotNull(result.getTransactionId());

		GetTransaction obj2 = (GetTransaction)container.getComponent(GetTransaction.class);
		GetTransactionResult result2 = obj2.call(jsonRPC, result.getTransactionId());

		int sendCount = 0;
		for (int i = 0; i < result2.getDetails().length; i++) {
			Details details = result2.getDetails()[i];
			if ("send".equals(details.getCategory())) {
				if (resources.getTestSendMany_address1().equals(details.getAddress())) {
					assertEquals(new BigDecimal("-0.01000000"),details.getAmount());
				} 
				sendCount++;
			}
		}
		assertEquals(1, sendCount);
		assertEquals("com1",result2.getComment());
		assertEquals("com2",result2.getTo());
	}

	/**
	 * Test
	 * SendFrom.call(JsonRPCClient jsonRPCClient, String acount, String toAddress, BigDecimal amount, Long minconf, String comment, String commentTo)
	 * 
	 * 【テスト仕様】
	 * 送金し、送金処理がされていることを確認する。
	 * 
	 * 【確認事項】
	 * ・送金数が一致していること。
	 * ・金額が一致していること。
	 * ・コメントが一致していること。
	 * 
	 * @throws Exception
	 * @since 1.0
	 */
	@Test
	public void test5() throws Exception {
		if (resources.isTestSendFrom_test5() == false) {
			System.out.println("test skip SendManyTestKuma.isTestSendFrom_test5");
			return;
		}

		SendFrom obj = (SendFrom)container.getComponent(SendFrom.class);
		SendFromResult result = obj.call(jsonRPC, resources.getTestSendMany_fromAccount(), resources.getTestSendFrom_address1(), new BigDecimal("0.01"), null, "com1", null);

		assertNotNull(result.getTransactionId());

		GetTransaction obj2 = (GetTransaction)container.getComponent(GetTransaction.class);
		GetTransactionResult result2 = obj2.call(jsonRPC, result.getTransactionId());

		int sendCount = 0;
		for (int i = 0; i < result2.getDetails().length; i++) {
			Details details = result2.getDetails()[i];
			if ("send".equals(details.getCategory())) {
				if (resources.getTestSendMany_address1().equals(details.getAddress())) {
					assertEquals(new BigDecimal("-0.01000000"),details.getAmount());
				} 
				sendCount++;
			}
		}
		assertEquals(1, sendCount);
		assertEquals("com1",result2.getComment());
		assertEquals(null,result2.getTo());
	}
	
	/**
	 * Test
	 * SendFrom.call(JsonRPCClient jsonRPCClient, String acount, String toAddress, BigDecimal amount, Long minconf, String comment, String commentTo)
	 * 
	 * 【テスト仕様】
	 * 送金し、送金処理がされていることを確認する。
	 * 
	 * 【確認事項】
	 * ・送金数が一致していること。
	 * ・金額が一致していること。
	 * ・コメントが一致していること。
	 * 
	 * @throws Exception
	 * @since 1.0
	 */
	@Test
	public void test6() throws Exception {
		if (resources.isTestSendFrom_test6() == false) {
			System.out.println("test skip SendManyTestKuma.isTestSendFrom_test6");
			return;
		}

		SendFrom obj = (SendFrom)container.getComponent(SendFrom.class);
		SendFromResult result = obj.call(jsonRPC, resources.getTestSendMany_fromAccount(), resources.getTestSendFrom_address1(), new BigDecimal("0.01"), null, null, "com2");

		assertNotNull(result.getTransactionId());

		GetTransaction obj2 = (GetTransaction)container.getComponent(GetTransaction.class);
		GetTransactionResult result2 = obj2.call(jsonRPC, result.getTransactionId());

		int sendCount = 0;
		for (int i = 0; i < result2.getDetails().length; i++) {
			Details details = result2.getDetails()[i];
			if ("send".equals(details.getCategory())) {
				if (resources.getTestSendMany_address1().equals(details.getAddress())) {
					assertEquals(new BigDecimal("-0.01000000"),details.getAmount());
				} 
				sendCount++;
			}
		}
		assertEquals(1, sendCount);
		assertEquals(null,result2.getComment());
		assertEquals("com2",result2.getTo());
	}
	
	
	/**
	 * Test Factory
	 * 
	 * @throws Exception
	 * @since 1.0
	 */
	@Test
	public void testFactory() throws Exception {

		if (resources.isTestSendFrom_testFactory() == false) {
			System.out.println("test skip SendManyTestKuma.isTestSendFrom_testFactory");
			return;
		}

		SendFrom obj = factory.getSendFrom();
		SendFromResult result = obj.call(jsonRPC, resources.getTestSendMany_fromAccount(), resources.getTestSendFrom_address1(), new BigDecimal("0.01"));

		assertNotNull(result.getTransactionId());

		GetTransaction obj2 = (GetTransaction)container.getComponent(GetTransaction.class);
		GetTransactionResult result2 = obj2.call(jsonRPC, result.getTransactionId());

		int sendCount = 0;
		for (int i = 0; i < result2.getDetails().length; i++) {
			Details details = result2.getDetails()[i];
			if ("send".equals(details.getCategory())) {
				if (resources.getTestSendMany_address1().equals(details.getAddress())) {
					assertEquals(new BigDecimal("-0.01000000"),details.getAmount());
				} 
				sendCount++;
			}
		}
		assertEquals(1, sendCount);
		assertEquals(null,result2.getComment());
	}
}
