package monalipse.views;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import monalipse.MonalipsePlugin;
import monalipse.actions.BookmarkAction;
import monalipse.bookmark.BookmarkManager;
import monalipse.bookmark.IBookmarkTreeNode;
import monalipse.editors.ThreadEditorInput;
import monalipse.editors.ThreadViewerEditor;
import monalipse.server.BBSServerManager;
import monalipse.server.IBBSBoard;
import monalipse.server.IBBSReference;
import monalipse.server.IThreadContentProvider;
import monalipse.utils.CancelableRunner;

import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerDropAdapter;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TreeEditor;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DragSourceEvent;
import org.eclipse.swt.dnd.DragSourceListener;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.dnd.TransferData;
import org.eclipse.swt.events.FocusAdapter;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Item;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.swt.widgets.Widget;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.XMLMemento;
import org.eclipse.ui.part.DrillDownAdapter;
import org.eclipse.ui.part.PluginTransfer;
import org.eclipse.ui.part.PluginTransferData;

public class BookmarksView extends AbstractTreeView implements SelectionListener, IResourceChangeListener
{
	public static final String ID_BOOKMARKS = BookmarksView.class.getName();

	private static final Logger logger = MonalipsePlugin.getLogger();

	private MappedTreeViewer viewer;
	private DrillDownAdapter drillDownAdapter;
	private IAction checkAction;
	private IAction abortAction;
	private IAction createFolderAction;
	private IAction removeFolderAction;
	private IAction doubleClickAction;
	private BookmarkAction bookmarkAction;
	private BookmarksContentProvider provider;
	private CancelableRunner cancelable;
	private TreeEditor treeEditor;
	private Text treeEditorText;
	private TreeItem treeEditItem;
	private IBookmarkTreeNode treeEditTarget;
	private Object lastSelected;

	public BookmarksView()
	{
	}

	protected ITreeContentProvider getContentProvider()
	{
		return provider;
	}

	protected ILabelProvider getLabelProvider()
	{
		return provider;
	}

	protected Logger getLogger()
	{
		return logger;
	}

	protected TreeViewer getViewer()
	{
		return viewer;
	}

	public void createPartControl(Composite parent)
	{
//logger.finest("enter");
		viewer = new MappedTreeViewer(parent, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL);
		drillDownAdapter = new DrillDownAdapter(viewer);
		provider = new BookmarksContentProvider();
		viewer.setContentProvider(provider);
		viewer.setLabelProvider(provider);
//		viewer.setSorter(new NameSorter());
		viewer.setInput(BookmarkManager.getBookmarks());
		makeActions();
		hookContextMenu();
		hookDoubleClickAction();
		contributeToActionBars();
		getSite().setSelectionProvider(viewer);
		viewer.getTree().addSelectionListener(this);
		ResourcesPlugin.getWorkspace().addResourceChangeListener(this);
		hookDragAndDrop();
		partCreated();

		viewer.getTree().addKeyListener(new KeyAdapter()
			{
				public void keyPressed(KeyEvent e)
				{
					if(e.keyCode == SWT.F5)
						checkAction.run();
				}
			});
	}
	
	public void dispose()
	{
//logger.finest("enter");
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);
		super.dispose();
	}
	
	public void resourceChanged(IResourceChangeEvent event)
	{
//logger.finest("enter");
		logger.finest("begin");
		
		boolean refresh = BookmarkManager.bookmarkChanged(event) || MonalipsePlugin.projectModified(event.getDelta());
		boolean update = !refresh && MonalipsePlugin.resourceModified(IResourceDelta.ADDED | IResourceDelta.REMOVED | IResourceDelta.CHANGED, event.getDelta(), MonalipsePlugin.getProject());
		
		if(refresh)
		{
			getSite().getShell().getDisplay().asyncExec(new Runnable()
				{
					public void run()
					{
						IMemento memento = XMLMemento.createWriteRoot(BookmarksView.class.getName());
						saveState(memento);
						logger.finest("refresh");
						viewer.setInput(BookmarkManager.getBookmarks());
						restoreState(memento);
					}
				});
		}

		if(update)
		{
			logger.finest("update");
			final IBookmarkTreeNode[] nodes = BookmarkManager.getChangedNodes(BookmarkManager.getBookmarks(), event);
			if(nodes != null)
			{
				getSite().getShell().getDisplay().asyncExec(new Runnable()
					{
						public void run()
						{
							viewer.update(nodes, null);
						}
					});
			}
		}

		logger.finest("done");
	}

	public void widgetSelected(SelectionEvent e)
	{
//logger.finest("enter");
		finishTreeEditor();

		logger.finest("begin");
		IBBSReference ref = bookmarkAction.updateState(viewer.getSelection());
		if(ref != null)
		{
			IThreadContentProvider thread = BBSServerManager.getThreadContentProviderOf(ref.getURL());
			if(thread != null)
				ThreadViewerEditor.activateEditor(getSite(), thread);
		}
		logger.finest("done");
		
		if(viewer.getSelection() instanceof IStructuredSelection && ((IStructuredSelection)viewer.getSelection()).size() == 1)
		{
			Object first = ((IStructuredSelection)viewer.getSelection()).getFirstElement();
			if(lastSelected == first && lastSelected instanceof IBookmarkTreeNode)
			{
				treeEditItem = (TreeItem) e.item;
				getSite().getShell().getDisplay().timerExec(300, new Runnable()
					{
						public void run()
						{
							if(treeEditItem == null)
								return;
							logger.finest("begin");

							treeEditTarget = (IBookmarkTreeNode)lastSelected;
			
							treeEditorText = new Text(viewer.getTree(), SWT.SINGLE | SWT.BORDER);
							treeEditor = new TreeEditor(viewer.getTree());
							treeEditor.horizontalAlignment = SWT.LEFT;
							treeEditor.grabHorizontal = true;
							treeEditor.minimumWidth = 50;
							treeEditor.setEditor(treeEditorText, treeEditItem);
							treeEditorText.setText(treeEditTarget.getName());
							treeEditorText.selectAll();
							treeEditorText.addFocusListener(new FocusAdapter()
								{
									public void focusLost(FocusEvent e)
									{
										finishTreeEditor();
									}
								});
							treeEditorText.addSelectionListener(new SelectionAdapter()
								{
									public void widgetDefaultSelected(SelectionEvent e)
									{
										finishTreeEditor();
									}
								});
							treeEditorText.setFocus();

							logger.finest("end");
						}
					});
			}
			else
			{
				lastSelected = first;
			}
		}
		else
		{
			lastSelected = null;
		}
	}

	private void finishTreeEditor()
	{
		logger.finest("begin");

		treeEditItem = null;

		if(treeEditorText != null)
		{
			logger.finest("closing");
			String name = treeEditorText.getText().trim();
			logger.finest("result: " + name);
			if(0 < name.length() && !treeEditTarget.getName().equals(name))
			{
				logger.finest("save");
				treeEditTarget.setName(name);
				BookmarkManager.setBookmarks(treeEditTarget.getRoot());
			}
			logger.finest("disposeText");
			treeEditorText.dispose();
			treeEditorText = null;
		}
		
		if(treeEditor != null)
		{
			logger.finest("disposeEditor");
			treeEditor.dispose();
		}
		treeEditor = null;

		logger.finest("done");
	}

	public void widgetDefaultSelected(SelectionEvent e)
	{
	}

	private void hookContextMenu()
	{
		MenuManager menuMgr = new MenuManager("#PopupMenu");
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener()
			{
				public void menuAboutToShow(IMenuManager manager)
				{
					BookmarksView.this.fillContextMenu(manager);
				}
			});
		Menu menu = menuMgr.createContextMenu(viewer.getControl());
		viewer.getControl().setMenu(menu);
		getSite().registerContextMenu(menuMgr, viewer);
	}

	private void contributeToActionBars()
	{
		IActionBars bars = getViewSite().getActionBars();
//		fillLocalPullDown(bars.getMenuManager());
		fillLocalToolBar(bars.getToolBarManager());
		fillLocalStatusBar(bars.getStatusLineManager());
	}

	private void fillLocalStatusBar(IStatusLineManager manager)
	{
		cancelable = new CancelableRunner(manager, getSite().getShell().getDisplay(), abortAction);
	}

	private void fillLocalPullDown(IMenuManager manager)
	{
		manager.add(checkAction);
		manager.add(abortAction);
	}

	private void fillContextMenu(IMenuManager manager)
	{
		bookmarkAction.updateState(viewer.getSelection());
		manager.add(checkAction);
		manager.add(abortAction);
		manager.add(new Separator());
		manager.add(createFolderAction);
		manager.add(new Separator());
		IStructuredSelection sel = (IStructuredSelection)viewer.getSelection();
		if(sel.getFirstElement() instanceof IBookmarkTreeNode && ((IBookmarkTreeNode)sel.getFirstElement()).hasChildren())
			manager.add(removeFolderAction);
		else
			manager.add(bookmarkAction);
		manager.add(new Separator());
		drillDownAdapter.addNavigationActions(manager);
		manager.add(new Separator("Additions"));
	}

	private void fillLocalToolBar(IToolBarManager manager)
	{
		manager.add(checkAction);
		manager.add(abortAction);
//		manager.add(new Separator());
//		drillDownAdapter.addNavigationActions(manager);
	}

	private void makeActions()
	{
		String iconPath = "icons/"; //$NON-NLS-1$		
		URL installURL = Platform.getPlugin(MonalipsePlugin.PLUGIN_ID).getDescriptor().getInstallURL();

		checkAction = new Action()
			{
				public void run()
				{
					cancelable.run(cancelable, new CancelableRunner.ICancelableRunnableWithProgress()
						{
							public void run(CancelableRunner.ICancelableProgressMonitor monitor)
							{
								IBookmarkTreeNode node = BookmarkManager.getBookmarks();
								List list = new ArrayList();
								getBoards(node, list);
								
								monitor.beginTask("", list.size() + 1);
								monitor.setTaskName("Checking Updates...");
								for(int i = 0; i < list.size(); i++)
								{
									monitor.worked(1);
									IBBSBoard board = (IBBSBoard)list.get(i);
									board.updateThreadList();
								}
								monitor.worked(1);
								monitor.done();
							}
						});
				}
				
				private void getBoards(IBookmarkTreeNode node, List list)
				{
					if(node instanceof IBBSReference)
					{
						IBBSBoard board = BBSServerManager.getBoardOf(((IBBSReference)node).getURL());
						if(!list.contains(board))
							list.add(board);
					}
					
					if(node.hasChildren())
					{
						IBookmarkTreeNode[] nodes = node.getChildren();
						for(int i = 0; i < nodes.length; i++)
							getBoards(nodes[i], list);
					}
				}
			};
		checkAction.setText("Check Updated");
		checkAction.setToolTipText("Check Updated");
		try
		{
			checkAction.setImageDescriptor(ImageDescriptor.createFromURL(new URL(installURL, iconPath + "refresh_nav.gif"))); //$NON-NLS-1$
		}
		catch (MalformedURLException e)
		{
		}

		abortAction = new Action()
			{
				public void run()
				{
					cancelable.cancel();
				}
			};
		abortAction.setText("Abort");
		abortAction.setToolTipText("Abort checking updated");
		abortAction.setEnabled(false);
		try
		{
			abortAction.setImageDescriptor(ImageDescriptor.createFromURL(new URL(installURL, iconPath + "stop_nav.gif"))); //$NON-NLS-1$
		}
		catch (MalformedURLException e)
		{
		}
		
		createFolderAction = new Action()
			{
				public void run()
				{
					if(viewer.getSelection() instanceof IStructuredSelection)
					{
						IStructuredSelection sel = (IStructuredSelection)viewer.getSelection();
						IBookmarkTreeNode node;
						if(0 < sel.size())
							node = (IBookmarkTreeNode)sel.getFirstElement();
						else
							node = BookmarkManager.getBookmarks();
						node.createFolder("New Folder", node.hasChildren() ? IBookmarkTreeNode.LOCATION_ON : IBookmarkTreeNode.LOCATION_AFTER);
						BookmarkManager.setBookmarks(BookmarkManager.getBookmarks());
					}
				}
			};
		createFolderAction.setText("Create New Folder");
		createFolderAction.setToolTipText("Create New Folder");
		createFolderAction.setImageDescriptor(PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_OBJ_FOLDER));
		
		removeFolderAction = new Action()
			{
				public void run()
				{
					if(viewer.getSelection() instanceof IStructuredSelection)
					{
						IStructuredSelection sel = (IStructuredSelection)viewer.getSelection();
						if(0 < sel.size())
						{
							((IBookmarkTreeNode)sel.getFirstElement()).remove();
							BookmarkManager.setBookmarks(BookmarkManager.getBookmarks());
						}
					}
				}
			};
		removeFolderAction.setText("Remove Folder");
		removeFolderAction.setToolTipText("Remove Folder");

		doubleClickAction = new Action()
			{
				public void run()
				{
					treeEditItem = null;

//System.err.println("> " + System.currentTimeMillis());
					ISelection selection = viewer.getSelection();
					if(selection instanceof IStructuredSelection)
					{
						Object first = ((IStructuredSelection)selection).getFirstElement();
						if(first instanceof IBBSReference)
						{
							logger.finest("begin");
							IBBSReference ref = (IBBSReference)first;
							IWorkbenchPage page = getSite().getPage().getWorkbenchWindow().getActivePage();
							IThreadContentProvider thread = BBSServerManager.getThreadContentProviderOf(ref.getURL());
							try
							{
								if(thread != null)
								{
									logger.finest("openEditor");
//System.err.println("a " + System.currentTimeMillis());
									ThreadEditorInput tei = new ThreadEditorInput(thread);
									tei.setPrefetch(true);
									IEditorPart part = page.openEditor(tei, ThreadViewerEditor.class.getName());
									if(part instanceof ThreadViewerEditor)
									{
//System.err.println("b " + System.currentTimeMillis());
										logger.finest("updateThread");
										((ThreadViewerEditor)part).updateThread(-1);
									}
									tei.setPrefetch(false);
								}
								else
								{
									final IBBSBoard board = BBSServerManager.getBoardOf(ref.getURL());
									if(board != null)
									{
										cancelable.run(cancelable, new CancelableRunner.ICancelableRunnableWithProgress()
											{
												public void run(CancelableRunner.ICancelableProgressMonitor monitor)
												{
													board.updateThreadList();
												}
											});
										logger.finest("openView");
										page.showView(ThreadListView.ID_THREAD_LIST);
									}
								}
							}
							catch(PartInitException ex)
							{
								ex.printStackTrace();
							}
							logger.finest("done");
						}
						else
						{
							viewer.setExpandedState(first, !viewer.getExpandedState(first));
						}
					}
				}
			};

		bookmarkAction = new BookmarkAction();
	}

	private void hookDoubleClickAction()
	{
		viewer.addDoubleClickListener(new IDoubleClickListener()
			{
				public void doubleClick(DoubleClickEvent event)
				{
					doubleClickAction.run();
				}
			});
	}
	
	private void hookDragAndDrop()
	{
		viewer.addDragSupport(DND.DROP_COPY | DND.DROP_MOVE, new Transfer[]{PluginTransfer.getInstance()}, new DragSourceListener()
			{
				private IBookmarkTreeNode[] lastData;

				public void dragStart(DragSourceEvent event)
				{
//logger.finest("enter");
					logger.finest("begin");
					lastSelected = null;
					finishTreeEditor();
					if(viewer.getSelection() instanceof IStructuredSelection)
						event.doit = 0 < ((IStructuredSelection)viewer.getSelection()).size();
					else
						event.doit = false;
					logger.finest("done " + event.doit);
				}
	
				public void dragSetData(DragSourceEvent event)
				{
//logger.finest("enter");
					logger.finest("begin");
					lastSelected = null;
					finishTreeEditor();
					if(viewer.getSelection() instanceof IStructuredSelection)
					{
						IStructuredSelection sel = (IStructuredSelection)viewer.getSelection();
						List list = new ArrayList();
						Object[] nodes = sel.toArray();
						for(int i = 0; i < nodes.length; i++)
						{
							boolean skip = false;
							IBookmarkTreeNode node = (IBookmarkTreeNode)nodes[i];
							for(int j = 0; j < nodes.length; j++)
							{
								if(i != j && ((IBookmarkTreeNode)nodes[j]).isAncestorOf(node))
								{
									skip = true;
									break;
								}
							}
							if(!skip)
								list.add(node);
						}
						lastData = new IBookmarkTreeNode[list.size()];
						list.toArray(lastData);
						event.data = BookmarkManager.getTransferData(lastData);
					}
					if(event.data == null)
					{
						event.data = new PluginTransferData(BookmarksView.class.getName() + ".invalid", new byte[0]);
						lastData = null;
					}
					logger.finest("done");
				}
	
				public void dragFinished(DragSourceEvent event)
				{
//logger.finest("enter");
					logger.finest("begin");
					lastSelected = null;
					finishTreeEditor();
					if(event.doit)
					{
						if(event.detail == DND.DROP_MOVE)
						{
							if(lastData != null)
							{
								for(int i = 0; i < lastData.length; i++)
									lastData[i].remove();
							}
						}
						
						if(event.detail == DND.DROP_MOVE || event.detail == DND.DROP_COPY)
							BookmarkManager.setBookmarks(BookmarkManager.getBookmarks());
					}
					logger.finest("done");
				}
			});

		viewer.addDropSupport(DND.DROP_COPY | DND.DROP_MOVE, new Transfer[]{PluginTransfer.getInstance()}, new ViewerDropAdapter(viewer)
			{
				protected int determineLocation(DropTargetEvent event)
				{
//logger.finest("enter");
					if(event.item instanceof TreeItem && ((Item)event.item).getData() instanceof IBookmarkTreeNode)
					{
						Item item = (Item)event.item;
						Point pt = viewer.getControl().toControl(new Point(event.x, event.y));
						if(item != null)
						{
							Rectangle bounds = getBounds(item);

							if(bounds == null)
								return LOCATION_NONE;
							
							int border;
							if(((IBookmarkTreeNode)item.getData()).hasChildren())
								border = 4;
							else
								border = bounds.height / 2;

							if((pt.y - bounds.y) <= border)
							{
								viewer.getTree().setInsertMark((TreeItem)event.item, true);
								return LOCATION_BEFORE;
							}
							else if((bounds.y + bounds.height - pt.y) <= border)
							{
								viewer.getTree().setInsertMark((TreeItem)event.item, false);
								return LOCATION_AFTER;
							}
							else
							{
								viewer.getTree().setInsertMark(null, false);
								return LOCATION_ON;
							}
						}
					}
					return LOCATION_NONE;
				}

				public boolean validateDrop(Object target, int operation, TransferData transferType)
				{
//logger.finest("enter");
					lastSelected = null;
					finishTreeEditor();
					try
					{
						return target instanceof IBookmarkTreeNode &&
								PluginTransfer.getInstance().isSupportedType(transferType);
					}
					catch(RuntimeException e)
					{
						e.printStackTrace();
						return false;
					}
				}

				public boolean performDrop(Object data)
				{
//logger.finest("enter");
					lastSelected = null;
					finishTreeEditor();
					viewer.getTree().setInsertMark(null, false);
					
					int location;
					switch(getCurrentLocation())
					{
					case LOCATION_BEFORE:
						location = IBookmarkTreeNode.LOCATION_BEFORE;
						break;

					case LOCATION_AFTER:
						location = IBookmarkTreeNode.LOCATION_AFTER;
						break;

					case LOCATION_ON:
						location = IBookmarkTreeNode.LOCATION_ON;
						break;
					
					default:
						return false;
					}

					if(data instanceof PluginTransferData && getCurrentTarget() instanceof IBookmarkTreeNode)
					{
						IBookmarkTreeNode[] nodes = BookmarkManager.getBookmarks((PluginTransferData)data);
						if(nodes != null)
						{
							IBookmarkTreeNode target = (IBookmarkTreeNode)getCurrentTarget();
							for(int i = 0; i < nodes.length; i++)
							{
								if(target == nodes[i] || nodes[i].isAncestorOf(target))
									return false;
							}
							
							for(int i = 0; i < nodes.length; i++)
								cloneNode(nodes[i], target, location);

							if(location == IBookmarkTreeNode.LOCATION_ON)
								viewer.setExpandedState(target, true);

							return true;
						}
					}

					return false;
				}
				
				private void cloneNode(IBookmarkTreeNode src, IBookmarkTreeNode refChild, int location)
				{
					if(src instanceof IBBSReference)
					{
						refChild.createLink(src.getName(), ((IBBSReference)src).getURL(), location);
					}
					else
					{
						IBookmarkTreeNode target = refChild.createFolder(src.getName(), location);
						IBookmarkTreeNode[] nodes = src.getChildren();
						for(int i = 0; i < nodes.length; i++)
							cloneNode(nodes[i], target, IBookmarkTreeNode.LOCATION_ON);
					}
				}
			});
	}

	public void setFocus()
	{
		viewer.getControl().setFocus();
	}

	class NameSorter extends ViewerSorter
	{
	}

	private class BookmarksContentProvider extends LabelProvider implements IStructuredContentProvider, ITreeContentProvider
	{
		public Object[] getElements(Object inputElement)
		{
//logger.finest("enter");
			if(inputElement instanceof IBookmarkTreeNode)
				return ((IBookmarkTreeNode)inputElement).getChildren();
			else
				return new Object[0];
		}

		public Object[] getChildren(Object parentElement)
		{
//logger.finest("enter");
			if(parentElement instanceof IBookmarkTreeNode)
				return ((IBookmarkTreeNode)parentElement).getChildren();
			else
				return new Object[0];
		}

		public Object getParent(Object element)
		{
//logger.finest("enter");
			if(element instanceof IBookmarkTreeNode)
				return ((IBookmarkTreeNode)element).getParent();
			else
				return null;
		}

		public boolean hasChildren(Object element)
		{
//logger.finest("enter");
			if(element instanceof IBookmarkTreeNode)
				return ((IBookmarkTreeNode)element).hasChildren();
			else
				return false;
		}

		public Image getImage(Object element)
		{
//logger.finest("enter");
			if(element instanceof IBBSReference)
				return MonalipsePlugin.getDefault().getLabelImageOf((IBBSReference)element);
			boolean updated = element instanceof IBookmarkTreeNode && ((IBookmarkTreeNode)element).hasNewResponses();
			return MonalipsePlugin.getDefault().getOverlayImage(MonalipsePlugin.IMG_OBJ_FOLDER, false, updated, false);
		}

		public String getText(Object element)
		{
//logger.finest("enter");
			if(element instanceof IBookmarkTreeNode)
				return ((IBookmarkTreeNode)element).getName();
			else
				return "";
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
		{
//logger.finest("enter");
		}

	}
	
	private class MappedTreeViewer extends TreeViewer
	{
		public MappedTreeViewer(Composite parent, int style)
		{
			super(parent, style);
		}
		
		public Widget getWidgetOf(Object element)
		{
			return findItem(element);
		}
	}

}