<?php
/**
 * Moony - the tiny web application framework
 *
 * @package   Moony
 * @author    YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @link      http://moony.sourceforge.jp/
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license   http://opensource.org/licenses/bsd-license.php The BSD License
 */

define('MOONY_FLASH_KEY', '_moony_flash');

/**
 * フラッシュ領域を扱うクラスです。
 * フラッシュ領域はセッションに保存され、
 * 次のリクエストまで有効となります。
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @access public
 */
class Moony_Flash
{
    /**
     * フラッシュ領域の値の連想配列
     * @var array
     */
    var $_flash_vars;

    /**
     * コンストラクタです。
     * セッションから前のリクエストで設定された値を取得します。
     *
     * @access public
     */
    function Moony_Flash()
    {
        $this->_flash_vars = array_key_exists(MOONY_FLASH_KEY, $_SESSION) ?
            $_SESSION[MOONY_FLASH_KEY] : array();
        $_SESSION[MOONY_FLASH_KEY] = array();
    }

    /**
     * フラッシュ領域の値を取得します。
     * 該当する値が存在しない場合、$defaultを返します。
     *
     * @access public
     * @param string $name 値の名称
     * @param mixed $default 値が存在しない場合のデフォルト値
     * @return mixed フラッシュ領域の値
     */
    function get($name, $default = null)
    {
        if ($this->exists($name)) {
            return $this->_flash_vars[$name];
        }
        return $default;
    }

    /**
     * フラッシュ領域に格納されている全ての値を取得します。
     *
     * @access public
     * @return array フラッシュ領域に格納されている全ての値
     */
    function getAll()
    {
        return $this->_flash_vars;
    }

    /**
     * フラッシュ領域に値を設定します。
     * $varが配列の場合、展開して全ての値を設定します。
     *
     * @access public
     * @param string|array $var 値の名称、または値の連想配列
     * @param mixed $value 設定する値
     */
    function set($var, $value = null)
    {
        if (is_array($var)) {
            foreach ($var as $name => $val) {
                $this->_flash_vars[$name] = $val;
                $_SESSION[MOONY_FLASH_KEY][$name] = $val;
            }
        } else {
            $this->_flash_vars[$var] = $value;
            $_SESSION[MOONY_FLASH_KEY][$var] = $value;
        }
    }

    /**
     * フラッシュ領域に指定された名称の値が
     * 存在するかどうか調べます。
     *
     * @access public
     * @param string $name 値の名称
     * @return boolean 存在するかどうか
     */
    function exists($name)
    {
        return array_key_exists($name, $this->_flash_vars);
    }

    /**
     * フラッシュ領域から値を除去します。
     *
     * @access public
     * @param string $name 除去する値の名称
     * @return mixed|false 除去された値
     */
    function remove($name)
    {
        if ($this->exists($name)) {
            $value = $this->_flash_vars[$name];
            unset($this->_flash_vars[$name]);
            if (array_key_exists($name, $_SESSION[MOONY_FLASH_KEY])) {
                // セッションからも削除
                unset($_SESSION[MOONY_FLASH_KEY][$name]);
            }
            return $value;
        }
    }

    /**
     * フラッシュ領域の全ての値を消去します。
     *
     * @access public
     */
    function clean()
    {
        $this->_flash_vars = array();
        $_SESSION[MOONY_FLASH_KEY] = array();
    }

    /**
     * フラッシュ領域の値を次のリクエストまで持ち越します。
     * 前のリクエストで設定された値の保持期間を延長することが可能です。
     * $nameが設定されていない場合、
     * フラッシュ領域の全ての値を次のリクエストまで持ち越します。
     *
     * @access public
     * @param string $name 持ち越す値の名称
     */
    function keep($name = null)
    {
        if (is_null($name)) {
            foreach ($this->_flash_vars as $name => $value) {
                $_SESSION[MOONY_FLASH_KEY][$name] = $value;
            }
        } else {
            if ($this->exists($name)) {
                $_SESSION[MOONY_FLASH_KEY][$name] = $this->_flash_vars[$name];
            }
        }
    }
}
?>
