<?php
/**
 * Moony - the tiny web application framework
 *
 * @package   Moony
 * @author    YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @link      http://moony.sourceforge.jp/
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license   http://opensource.org/licenses/bsd-license.php The BSD License
 */

// アクション名のパターン
define('MOONY_ACTION_NAME_PATTERN', '/^[A-Z]{1}[a-zA-Z0-9_]*$/');

// チェック
error_reporting(E_ALL);
if (php_sapi_name() != 'cli') {
    echo "   [error] Please call this script via CLI\n";
    exit;
}

// カレントディレクトリを基準として処理
$current_dir = getcwd();
$generator =& new Moony_Generator($current_dir);

// パラメータの処理
if ($_SERVER['argc'] == 2) {
    $param = $_SERVER['argv'][1];
    if ($param == '-init') {
        // -init
        $generator->createDirs();
        $generator->createEntryPoint();
        $generator->createAction('Index');
    } else if (preg_match(MOONY_ACTION_NAME_PATTERN, $param)) {
        // アクション生成
        $generator->createAction($param);
    } else {
        // ヘルプ
        $generator->showHelp();
    }
} else if ($_SERVER['argc'] == 3) {
    $param = $_SERVER['argv'][1];
    $smarty_opt = $_SERVER['argv'][2];
    if ($smarty_opt == '-smarty') {
        if ($param == '-init') {
            // -init
        $generator->createDirs(true);
        $generator->createEntryPoint(true);
        $generator->createAction('Index', true);
        } else if (preg_match(MOONY_ACTION_NAME_PATTERN, $param)) {
            // コントローラ生成
            $generator->createAction($param, true);
        } else {
            $gen->showHelp();
        }
    } else {
        $gen->showHelp();
    }
} else {
    // ヘルプ
    $generator->showHelp();
}

/**
 * コントローラを自動生成するためのクラスです。
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @access public
 */
class Moony_Generator
{
    /**
     * 自動生成の基準となるディレクトリ
     * @var string
     */
    var $_base_dir;

    /**
     * アクションクラスを格納するディレクトリ
     * @var string
     */
    var $_action_dir;

    /**
     * テンプレートファイルを格納するディレクトリ
     * @var string
     */
    var $_template_dir;

    /**
     * コンパイルされたファイルを格納するディレクトリ
     * @var string
     */
    var $_compile_dir;

    /**
     * エントリポイント
     * @var string
     */
    var $_entry_point;

    /**
     * コンストラクタです。
     *
     * @access public
     * @param string $base_dir 自動生成の基準となるディレクトリ
     */
    function Moony_Generator($base_dir)
    {
        $this->_base_dir = rtrim($base_dir, DIRECTORY_SEPARATOR);
        $this->_action_dir =
            $this->_base_dir . DIRECTORY_SEPARATOR . 'actions';
        $this->_template_dir =
            $this->_base_dir . DIRECTORY_SEPARATOR . 'templates';
        $this->_compile_dir =
            $this->_base_dir . DIRECTORY_SEPARATOR . 'templates_c';
        $this->_entry_point =
            $this->_base_dir . DIRECTORY_SEPARATOR . 'index.php';
    }

    /**
     * Moonyの動作に必要なディレクトリを生成します。
     *
     * @access public
     * @param boolean $use_smarty Smartyを使用するかどうか
     */
    function createDirs($use_smarty = false)
    {
        $this->_createDir($this->_action_dir);
        $this->_createDir($this->_template_dir);
        if ($use_smarty) {
            $this->_createDir($this->_compile_dir);
        }
    }

    /**
     * Moonyを実行するエントリポイントを生成します。
     *
     * @access public
     * @param boolean $use_smarty Smartyを使用するかどうか
     */
    function createEntryPoint($use_smarty = false)
    {
        $skeleton = $this->_getEntryPointSkeleton($use_smarty);
        $this->_createFile($this->_entry_point, $skeleton);
    }

    /**
     * アクションクラスとテンプレートファイルを生成します。
     *
     * @access public
     * @param string $action_name アクション名称
     * @param boolean $use_smarty Smartyを使用するかどうか
     */
    function createAction($action_name, $use_smarty = false)
    {
        $template_ext = $use_smarty ? '.tpl' : '.php';

        $file_name = str_replace('_', DIRECTORY_SEPARATOR, $action_name);
        $template_file = strtolower($file_name) . $template_ext;

        $action_path =
            $this->_action_dir . DIRECTORY_SEPARATOR . $file_name . '.php';
        $action_skeleton = $this->_getActionSkeleton();
        $action_skeleton = sprintf($action_skeleton, $action_name,
            str_replace(DIRECTORY_SEPARATOR, '/', $template_file));
        $this->_createFile($action_path, $action_skeleton);

        $template_path =
            $this->_template_dir . DIRECTORY_SEPARATOR . $template_file;
        $template_skeleton = $this->_getTemplateSkeleton();
        $template_skeleton = sprintf(
            $template_skeleton, $action_name, $action_name);
        $this->_createFile($template_path, $template_skeleton);
    }

    /**
     * ヘルプメッセージを表示します。
     *
     * @access public
     */
    function showHelp()
    {
        echo "\n";
        echo "Usage: moony -init\n";
        echo "       moony <action_name>\n";
        echo "       moony -init -smarty\n";
        echo "       moony <action_name> -smarty\n";
        echo "\n";
        echo "Examples:\n";
        echo "  '-init' generates:\n";
        echo "    - actions ................... the directory for action classes\n";
        echo "    - templates ................. the directory for template files\n";
        echo "    - index.php ................. the entry point script\n";
        echo "    - actions/Index.php ......... the first action class\n";
        echo "    - templates/index.php ....... the first template file\n";
        echo "  'Foo' generates:\n";
        echo "    - actions/Foo.php ........... the action class\n";
        echo "    - templates/foo.php ......... the template file\n";
        echo "  'Foo_Bar' generates:\n";
        echo "    - actions/Foo/Bar.php ....... the action class\n";
        echo "    - templates/foo/bar.php ..... the template file\n";
        echo "  '-init -smarty' generates:\n";
        echo "    - actions ................... the directory for action classes\n";
        echo "    - templates ................. the directory for template files\n";
        echo "    - templates_c ............... the directory for compiled files\n";
        echo "    - index.php ................. the entry point script\n";
        echo "    - actions/Index.php ......... the first action class\n";
        echo "    - templates/index.tpl ....... the first template file\n";
        echo "  'Foo -smarty' generates:\n";
        echo "    - actions/Foo.php ........... the action class\n";
        echo "    - templates/foo.tpl ......... the template file\n";
        echo "  'Foo_Bar -smarty' generates:\n";
        echo "    - actions/Foo/Bar.php ....... the action class\n";
        echo "    - templates/foo/bar.tpl ..... the template file\n";
        echo "\n";
        echo "The generator configures the action to dislay the specified template.\n";
        echo "\n";
    }

    /**
     * エントリポイントの雛形を取得します。
     *
     * @access protected
     * @return string エントリポイントの雛形
     * @param boolean $use_smarty Smartyを使用するかどうか
     */
    function _getEntryPointSkeleton($use_smarty = false)
    {
        $skeleton = "<?php\n"
                  . "require_once 'Moony.php';\n";
        if ($use_smarty) {
            $skeleton .= "require_once 'Pearified/Smarty/Smarty.class.php';\n";
        }
        $skeleton .= "\n"
                  .  "\$moony = new Moony();\n"
                  .  "\$moony->setActionDir('actions');\n"
                  .  "\$moony->setTemplateDir('templates');\n"
                  .  "// \$moony->setInputEncoding('SJIS-win');\n"
                  .  "// \$moony->setOutputEncoding('SJIS-win');\n"
                  .  "// \$moony->setInternalEncoding('UTF-8');\n";
        if ($use_smarty) {
            $skeleton .= "\$moony->useSmarty(array(\n"
                      .  "    'compile_dir' => 'templates_c',\n"
                      .  "    'left_delimiter' => '{{',\n"
                      .  "    'right_delimiter' => '}}'\n"
                      .  "));\n";
        }
        $skeleton .= "\$moony->start();\n"
                  .  "?>\n";
        return $skeleton;
    }

    /**
     * アクションクラスの雛形を取得します。
     *
     * @access protected
     * @return string コントローラクラスの雛形
     */
    function _getActionSkeleton()
    {
        $skeleton = "<?php\n"
                  . "class %s extends Moony_Action\n"
                  . "{\n"
                  . "    function execute()\n"
                  . "    {\n"
                  . "        \$this->display('%s');\n"
                  . "    }\n"
                  . "}\n"
                  . "?>\n";
        return $skeleton;
    }

    /**
     * テンプレートファイルの雛形を取得します。
     *
     * @access protected
     * @return string テンプレートファイルの雛形
     */
    function _getTemplateSkeleton()
    {
        $skeleton = "<html>\n"
                  . "<head>\n"
                  . "<title>action: %s</title>\n"
                  . "</head>\n"
                  . "<body>\n"
                  . "<h1>action: %s</h1>\n"
                  . "<p>Hello Moony!</p>\n"
                  . "</body>\n"
                  . "</html>\n";
        return $skeleton;
    }

    /**
     * ディレクトリを生成します。
     *
     * @access protected
     * @param string $dir ディレクトリ名称
     */
    function _createDir($dir)
    {
        if (file_exists($dir)) {
            echo "   [error] already exists: {$dir}\n";
            return;
        }
        $parent_dir = dirname($dir);
        if (!file_exists($parent_dir)) {
            $this->_createDir($parent_dir);
        }
        mkdir($dir);
        echo "  [create] {$dir}\n";
    }

    /**
     * プレーンテキストファイルを生成します。
     *
     * @access protected
     * @param string $file ファイル名称
     * @param string $content ファイルの内容
     */
    function _createFile($file, $content = null)
    {
        if (file_exists($file)) {
            echo "   [error] already exists: {$file}\n";
            return;
        }

        $dir = dirname($file);
        if (!file_exists($dir)) {
            $this->_createDir($dir);
        }

        $fp = fopen($file, 'w');
        if (!is_null($content)) {
            fwrite($fp, $content);
        }
        fclose($fp);

        echo "  [create] {$file}\n";
    }
}
?>
