<?php
/**
 * Moony - the tiny web application framework
 *
 * @package   Moony
 * @author    YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @link      http://moony.sourceforge.jp/
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license   http://opensource.org/licenses/bsd-license.php The BSD License
 */

/**
 * リクエストパラメータを取り扱うクラスです。
 * パラメータに含まれるNULLバイト文字列は除去されます。
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @access public
 */
class Moony_Request
{
    /**
     * リクエストパラメータの連想配列
     * @var array
     */
    var $_parameters;

    /**
     * アップロードファイルに関する情報の連想配列
     * @var array
     */
    var $_files;

    /**
     * クエリーストリング
     * @var string
     */
    var $_query_string;

    /**
     * コンストラクタです。
     *
     * @access public
     */
    function Moony_Request()
    {
        // POSTパラメータとGETパラメータを結合、
        // GETパラメータでPOSTパラメータを上書きしない
        $this->_parameters = $_POST + $_GET;
        $this->_parameters = $this->filter($this->_parameters);
        $this->_files = $_FILES;
        $this->_files = $this->filter($this->_files);
        if (isset($_SERVER['QUERY_STRING'])) {
            // クエリーストリング
            $this->_query_string = $this->filter($_SERVER['QUERY_STRING']);
            $this->_query_string = rawurldecode($this->_query_string);
        }
    }

    /**
     * リクエストパラメータの値を取得します。
     * 該当する値が存在しない場合、$defaultを返します。
     *
     * @access public
     * @param string $name リクエストパラメータの名称
     * @param mixed $default 値が存在しない場合のデフォルト値
     * @rerurn string|array|mixed リクエストパラメータの値
     */
    function get($name, $default = null)
    {
        if ($this->exists($name)) {
            return $this->_parameters[$name];
        }
        return $default;
    }

    /**
     * 全てのリクエストパラメータの値を連想配列として返します。
     *
     * @access public
     * @return array 全てのリクエストパラメータの連想配列
     */
    function getAll()
    {
        return $this->_parameters;
    }

    /**
     * 指定された名称のリクエストパラメータが
     * 存在するかどうか調べます。
     *
     * @access public
     * @param string $name リクエストパラメータの名称
     * @return boolean 存在するかどうか
     */
    function exists($name)
    {
        return array_key_exists($name, $this->_parameters);
    }

    /**
     * アップロードファイルに関する情報を取得します。
     * $_FILESの該当する要素が返されます。
     *
     * @access public
     * @param string $name アップロードファイルの名称
     * @return array アップロードファイルの情報
     */
    function getFile($name)
    {
        if (array_key_exists($name, $this->_files)) {
            return $this->_files[$name];
        }
    }

    /**
     * クエリーストリングを取得します。
     * 存在しない場合、nullが返されます。
     *
     * @access public
     * @return string|null クエリーストリング値
     */
    function getQueryString()
    {
        return $this->_query_string;
    }

    /**
     * リクエストパラメータのエンコーディングを変換します。
     *
     * @access public
     * @param string $from_encoding 変換元のエンコーディング
     * @param string $to_encoding 変換先のエンコーディング
     */
    function convertEncoding($from_encoding, $to_encoding)
    {
        mb_convert_variables($to_encoding, $from_encoding,
            $this->_parameters, $this->_files, $this->_query_string);
    }

    /**
     * リクエストパラメータのフィルタ処理を行います。
     * NULLバイト文字列を除去し、
     * 「magic_quotes_gpc」がONになっていた場合は
     * 「stripslashes」関数を用いて不要なスラッシュを除去します。
     *
     * @access public
     * @static
     * @param string|array 処理対象値
     * @return string|array フィルタされた値
     */
    function filter($var)
    {
        static $magic_quotes_gpc;
        if (is_null($magic_quotes_gpc)) {
            $magic_quotes_gpc = get_magic_quotes_gpc();
        }
        if (is_array($var)) {
            return array_map(array('Moony_Request', 'filter'), $var);
        }
        $var = str_replace("\0", '', $var);
        if ($magic_quotes_gpc) {
            $var = stripslashes($var);
        }
        return $var;
    }
}
?>
