<?php
/**
 * Moony - the tiny web application framework
 *
 * @package   Moony
 * @author    YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @link      http://moony.sourceforge.jp/
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license   http://opensource.org/licenses/bsd-license.php The BSD License
 */

/**
 * アクションのルーティングを行うクラスです。
 * 下記例のような順序でアクションクラスを探索、処理を起動します。
 *   - foo => Foo
 *   - foo/bar => Foo_Bar, Foo
 *   - foo/bar/baz => Foo_Bar_Baz, Foo_Bar, Foo
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @access public
 */
class Moony_Router
{
    /**
     * PATH_INFOを元に実行する
     * アクションクラスを探索、処理を起動します。
     *
     * @access public
     * @param object $moony Moony
     */
    function route(&$moony)
    {
        $action_dir = $moony->action_dir;
        if (is_null($action_dir)) {
            // アクションクラスを格納するディレクトリが設定されていない
            trigger_error('Not defined: action_dir', E_USER_ERROR);
            header('HTTP/1.0 500 Internal Server Error');
            exit;
        }

        // PATH_INFOを取得、小文字化、分割、CamelCase
        $path_info = $this->_getPathInfo();
        $path_info = strtolower($path_info);
        $paths = explode('/', trim($path_info, '/'));
        $paths = array_map('ucfirst', $paths);

        // PATH_INFO経由で渡されるパラメータを格納
        $args = array();

        while (count($paths) > 0) {

            $action_path = implode(DIRECTORY_SEPARATOR, $paths);
            $path = $action_dir . DIRECTORY_SEPARATOR . $action_path . '.php';

            if (file_exists($path)) {
                // アクションクラス
                include_once $path;
                $action_name = str_replace(
                    DIRECTORY_SEPARATOR, '_', $action_path);
                if (class_exists($action_name)) {
                    // インスタンス生成、処理を起動
                    $action =& new $action_name;
                    $action->process($moony, array_reverse($args));
                    return;
                }
            }
            $args[] = strtolower(array_pop($paths));
        }

        // 結局見つからなかった
        return false;
    }

    /**
     * PATH_INFO文字列を取得します。
     * 最後の文字がスラッシュの場合（ディレクトリを示していた場合）、
     * 末尾に「index」を付加します。拡張子は無視されます。
     *
     * @access protected
     * @return string PATH_INFO文字列
     */
    function _getPathInfo()
    {
        $path_info = '';
        if (isset($_SERVER['PATH_INFO'])) {
            $path_info = $_SERVER['PATH_INFO'];
            $path_info = str_replace("\0", '', $path_info);
            $pos = strpos($path_info, '.');
            if ($pos !== false) {
                $path_info = substr($path_info, 0, $pos);
            }
        }
        if (strlen($path_info) == 0 ||
            substr($path_info, -1, 1) == '/') {
            // PATH_INFOが存在しない or "/"で終了
            $path_info .= 'index';
        }
        return $path_info;
    }
}
?>
